# -*- coding: UTF-8 -*-
#  Copyright (c) Huawei Technologies Co., Ltd. 2023-2024. All rights reserved.

import json
import time
import re
from nvme.task_util import exec_host_cmd, get_err_msg

from com.huawei.ism.tool.storagedeploy.util.ProcessUtils import getResultFile


def get_callback_params(status, custom_result=None):
    params = {"status": status, "customResult": custom_result}
    return json.dumps(params)


def execute(context):
    return NvmeTaskService(context).run()


class NvmeTaskService:

    def __init__(self, context):
        self.context = context
        self.dev = context.get("dev")
        self.logger = context.get("logger")
        self.lang = context.get("language")
        self.cli = context.get("SSH")
        self.call_back = context.get("call_back")
        self.base_path = self.call_back.getNvmeTaskPath().getDestBasePath()
        self.methods = context.get("methods")
        self.python_cmd = "python" if str(self.call_back.getPythonType()).lower() == "python2" else "python3"

    # 入口执行方法
    def run(self):
        # 需要执行的方法
        self.logger.info("----------- NvmeTaskService start ---------")
        self.logger.info("----------- self.methods ---------:{}".format(self.methods))
        methods = self.methods.split(",")
        has_part_success = False
        all_success = True
        for method_name in methods:
            func = getattr(self, method_name, None)
            if func:
                begin_time = time.strftime("%Y-%m-%d %H:%M:%S", time.localtime())
                # 开始之前，调用一下回调的开始
                custom_result = {"beginTime": begin_time, "methodName": method_name}
                self.call_back.start(get_callback_params(CheckStatus.PASS, custom_result))
                return_value = func()
                status = return_value[0]
                err_msg = return_value[1]
                origin_info = return_value[2]
                end_time = time.strftime("%Y-%m-%d %H:%M:%S", time.localtime())
                # 组装回调的参数
                custom_result = {"beginTime": begin_time, "endTime": end_time, "detail": err_msg,
                                 "methodName": method_name, "originInfo": origin_info}
                self.call_back.update(get_callback_params(status, custom_result))
                if status == CheckStatus.NOT_PASS:
                    all_success = False
                if status == CheckStatus.WARNING:
                    has_part_success = True
            else:
                return CheckStatus.NOT_PASS, "process.task.step.detail.step_not_exist", method_name
        if not all_success:
            return CheckStatus.NOT_PASS, "", ""
        if has_part_success:
            return CheckStatus.WARNING, "", ""
        return CheckStatus.PASS, "", ""

    def check_mlnx_tools(self):
        """
        检查是否安装mlnx-tools
        """
        cmd = "mlnx_qos --version"
        ret = exec_host_cmd(self.cli, cmd, True)
        self.logger.info("ssh_ret:{}".format(ret))
        match = re.search(r'mlnx_qos \d+\.\d+', ret)
        if not match:
            return CheckStatus.NOT_PASS, "check.mlnx_tools.failed", ret
        return CheckStatus.PASS, "", ret

    def check_network_adapter(self):
        """
        主机侧网卡检查。
        """
        self.logger.info("-----------check_network_adapter start-----------")
        # 跳转到执行脚本的目录。
        cmd = "cd {}/script/".format(self.base_path)
        ret = exec_host_cmd(self.cli, cmd, True)
        self.logger.info("ssh_ret: {}".format(ret))
        # 执行配置网络脚本。
        cmd = "{} check_network_adapter.py".format(self.python_cmd)
        ret = exec_host_cmd(self.cli, cmd, True)
        self.logger.info("ssh_ret:, {}".format(ret))
        if "Execution completed" not in ret:
            return CheckStatus.NOT_PASS, "", ""
        result = json.loads(getResultFile(self.call_back, "check_network_adapter_result.json"))
        self.logger.info("origin_info: {}".format(result.get("origin_info")))
        if not result.get("status"):
            return CheckStatus.WARNING, get_err_msg(result.get("error_code"), self.lang), result.get("origin_info")
        return CheckStatus.PASS, "", result.get("origin_info", "")

    def check_nvme_cli(self):
        """
        是否安装nvme_cli
        """
        self.logger.info("-----------check_nvme_cli start-----------")
        # 跳转到执行脚本的目录。
        cmd = "cd {}/script/".format(self.base_path)
        ret = exec_host_cmd(self.cli, cmd, True)
        self.logger.info("ssh_ret: {}".format(ret))
        # 执行配置网络脚本。
        cmd = "{} check_nvme_cli.py".format(self.python_cmd)
        ret = exec_host_cmd(self.cli, cmd, True)
        self.logger.info("ssh_ret:, {}".format(ret))
        if "Execution completed" not in ret:
            return CheckStatus.NOT_PASS, "", ""
        result = json.loads(getResultFile(self.call_back, "check_nvme_cli_result.json"))
        self.logger.info("origin_info: {}".format(result.get("origin_info")))
        if not result.get("status", ""):
            return CheckStatus.NOT_PASS, get_err_msg(result.get("error_code"), self.lang), result.get("origin_info")
        return CheckStatus.PASS, result.get("error_code", ""), result.get("origin_info")

    def check_oceanstor_nof(self):
        """
        加载驱动
        """
        self.logger.info("-----------check_oceanstor_nof start-----------")
        # 跳转到执行脚本的目录。
        cmd = "cd {}/script/".format(self.base_path)
        ret = exec_host_cmd(self.cli, cmd, True)
        self.logger.info("ssh_ret: {}".format(ret))
        # 执行配置网络脚本。
        cmd = "{} check_oceanstor_nof.py".format(self.python_cmd)
        ret = exec_host_cmd(self.cli, cmd, True)
        self.logger.info("ssh_ret:, {}".format(ret))
        if "Execution completed" not in ret:
            return CheckStatus.NOT_PASS, "", ""
        result = json.loads(getResultFile(self.call_back, "check_oceanstor_nof_result.json"))
        self.logger.info("origin_info: {}".format(result.get("origin_info")))
        if not result.get("status", ""):
            return CheckStatus.WARNING, get_err_msg(result.get("error_code"), self.lang), result.get("origin_info")
        return CheckStatus.PASS, "", result.get("origin_info")


class CheckStatus:
    PASS = "PASS"
    NOT_PASS = "NOT_PASS"
    NOT_CHECK = "NOT_CHECK"
    NOT_SUPPORT = "NOT_SUPPORT"
    WARNING = "WARNING"
    UNKNOWN = "UNKNOWN"
