# coding=UTF-8
# Copyright (c) Huawei Technologies Co., Ltd. 2019-2020. All rights reserved.
# noinspection PyUnresolvedReferences
from java.lang import Exception as JException
from cbb.business.operate.expansion import common
from cbb.common.conf.productConfig import VersionManager
from cbb.frame.context import contextUtil
from cbb.frame.tlv import adTlvUtil
from cbb.frame.tlv import tlvData
from cbb.frame.tlv import tlvUtil
import cbb.frame.base.config as config


class DevInfoKey:
    """ 设备信息key """
    PDT_MODEL_ENUM = "product_model_enum"  # 产品型号枚举
    PDT_MODEL_STRING = "product_model_string"  # 产品型号string
    PDT_OEM_MODEL = "pdt_oem_model"  # 白牌型号
    INTERNAL_MODEL = "internal_model"  # 内部型号
    PDT_VERSION = "product_version"  # 产品版本
    SN = "sn"  # 原框SN


class CheckNewPdtModelForHybrid:
    """ 扩控前新扩控产品型号一致性检查（融合存储）"""

    def __init__(self, context):
        self.context = context
        self.result_dict = {"flag": True, "errMsg": "", "suggestion": ""}
        self.tlv = contextUtil.getTlv(context)
        self.lang = contextUtil.getLang(context)
        self.logger = common.getLogger(context.get("logger"), __file__)
        self.new_boards_list = contextUtil.getItem(context, "newBoardsList")
        self.logger.logInfo("new board list:" + str(self.new_boards_list))
        self.new_config_clust_type = contextUtil.getItem(context,
                                                         "newConfigClustType")
        self.origin_dev_info = dict()
        self.init_origin_dev_infos()

    def init_origin_dev_infos(self):
        """ 初始化原集群信息

        :return:
        """
        ori_sys_record = tlvUtil.getSystemRecord(self.tlv)
        origin_dev_info = tlvUtil.getInternalDeviceInfo(self.tlv)

        pdt_model_enum = tlvUtil.getRecordValue(
            ori_sys_record, tlvData.SYSTEM['productMode'])
        pdt_model_string = tlvUtil.getRecordValue(
            ori_sys_record, tlvData.SYSTEM['product_model_string'])
        pdt_oem_model = tlvUtil.getRecordValue(
            ori_sys_record, tlvData.SYSTEM['product_oem_model'])
        internal_model = tlvUtil.getInternalPdtModel(origin_dev_info)
        ori_full_product_version = tlvUtil.getFullProductVersion(self.tlv)
        ori_pdt_sn = tlvUtil.getRecordValue(
            ori_sys_record, tlvData.SYSTEM['SN'])

        self.origin_dev_info[DevInfoKey.PDT_MODEL_ENUM] = pdt_model_enum
        self.origin_dev_info[DevInfoKey.PDT_MODEL_STRING] = pdt_model_string
        self.origin_dev_info[DevInfoKey.PDT_OEM_MODEL] = pdt_oem_model
        self.origin_dev_info[DevInfoKey.INTERNAL_MODEL] = internal_model
        self.origin_dev_info[DevInfoKey.PDT_VERSION] = ori_full_product_version
        self.origin_dev_info[DevInfoKey.SN] = ori_pdt_sn
        self.logger.logInfo("origin device info: {}".format(
            self.origin_dev_info))

    def get_new_dev_info(self, board):
        """ 获取新集群节点信息

        :param board: 节点
        :return: 新集群节点信息字典
        """
        new_dev_info = dict()
        new_sys_record = adTlvUtil.getSystemRecord(self.tlv, board)
        new_internal_info = adTlvUtil.getInternalDeviceInfo(self.tlv, board)
        new_pdt_version, new_patch_version = adTlvUtil.getProductVersion(
            self.tlv, board)
        # 完整版本需将版本号和补丁号拼接
        if "Kunpeng" in new_pdt_version:
            new_pdt_version = new_pdt_version[:-8]
        full_new_pdt_version = new_pdt_version + new_patch_version

        pdt_model_enum = adTlvUtil.getRecordValue(
            new_sys_record, tlvData.SYSTEM['productMode'])
        pdt_model_string = adTlvUtil.getRecordValue(
            new_sys_record, tlvData.SYSTEM['product_model_string'])
        pdt_oem_model = adTlvUtil.getRecordValue(
            new_sys_record, tlvData.SYSTEM['product_oem_model'])
        internal_model = adTlvUtil.getInternalPdtModel(new_internal_info)

        new_dev_info[DevInfoKey.PDT_MODEL_ENUM] = pdt_model_enum
        new_dev_info[DevInfoKey.PDT_MODEL_STRING] = pdt_model_string
        new_dev_info[DevInfoKey.PDT_OEM_MODEL] = pdt_oem_model
        new_dev_info[DevInfoKey.INTERNAL_MODEL] = internal_model
        new_dev_info[DevInfoKey.PDT_VERSION] = full_new_pdt_version
        self.logger.logInfo("board: {}, device info: {}".format(board,
                                                                new_dev_info))
        return new_dev_info

    def execute(self):
        """ 执行入口

        :return:
        """
        try:
            self.check()
            if not self.result_dict.get("flag"):
                contextUtil.handleFailure(self.context, self.result_dict)
                return
            contextUtil.handleSuccess(self.context)
            self.logger.logPass()
            return
        except (Exception, JException) as ex:
            self.logger.logException(ex)
            contextUtil.handleException(self.context, ex)
            return

    def check(self):
        """ 开始检查

        :return:
        """
        err_msg_list = list()
        sugg_list = list()
        # 遍历待扩控节点检查
        for board in self.new_boards_list:
            # 待扩控节点信息
            new_dev_info = self.get_new_dev_info(board)

            # 检查产品型号枚举是否一致
            model_enum_flag = self.check_pdt_model_enum_consistent(
                new_dev_info)
            # 检查产品型号枚举是否一致
            model_string_flag = self.check_pdt_model_string_consistent(
                new_dev_info)
            # 检查产品白牌型号枚举是否一致
            model_oem_flag = self.check_pdt_oem_model_consistent(
                new_dev_info)
            # 检查内部型号是否一致,这里拦截了C+扩C
            internal_model_flag = \
                self.check_internal_model_consistent(new_dev_info)

            if not model_enum_flag or not model_string_flag \
                    or not internal_model_flag or not model_oem_flag:
                self.result_dict["flag"] = False
                err_msg, sugg = common.getMsg(self.lang,
                                              "pdt.controller.inconsistent",
                                              board["enclosureSN"])
                if err_msg not in err_msg_list:
                    err_msg_list.append(err_msg)
                sugg_list.append(sugg)

        if not self.result_dict["flag"]:
            self.result_dict["errMsg"] = "\n".join(err_msg_list)
            self.result_dict["suggestion"] = sugg_list[0]
            return

        enclosure_dict = {}
        enclosure_dict.setdefault(self.origin_dev_info[DevInfoKey.SN], {})
        enclosure_dict[self.origin_dev_info[DevInfoKey.SN]]["need_upgrade"] = \
            False
        enclosure_dict[self.origin_dev_info[DevInfoKey.SN]]["now_version"] = \
            self.origin_dev_info[DevInfoKey.PDT_VERSION]
        enclosure_dict[self.origin_dev_info[DevInfoKey.SN]]["aim_version"] = ""
        for board in self.new_boards_list:
            # 待扩控节点信息
            new_dev_info = self.get_new_dev_info(board)
            board_sn = board["enclosureSN"]
            if board_sn not in enclosure_dict:
                enclosure_dict.setdefault(board_sn, {})
                enclosure_dict[board_sn]["need_upgrade"] = False
                enclosure_dict[board_sn]["now_version"] = \
                    new_dev_info[DevInfoKey.PDT_VERSION]
                enclosure_dict[board_sn]["aim_version"] = ""

            # 检查原集群版本号
            aim_version = self.get_aim_version(new_dev_info)
            new_need_check, ori_need_check = \
                self.check_origin_version(new_dev_info, aim_version)

            # 若新集群需要升级
            if new_need_check:
                enclosure_dict[board_sn]["need_upgrade"] = True
                # 待升级级别一定要是最高的级别
                if aim_version > enclosure_dict[board_sn]["aim_version"]:
                    enclosure_dict[board_sn]["aim_version"] = aim_version
            # 若原集群需要升级
            if ori_need_check:
                enclosure_dict[self.origin_dev_info[DevInfoKey.SN]][
                    "need_upgrade"] = True
                if aim_version > \
                        enclosure_dict[self.origin_dev_info[DevInfoKey.SN]][
                            "aim_version"]:
                    enclosure_dict[self.origin_dev_info[DevInfoKey.SN]][
                        "aim_version"] = aim_version

        # 添加错误信息
        for (SN, sn_check_dict) in enclosure_dict.items():
            if sn_check_dict["need_upgrade"]:
                self.result_dict["flag"] = False
                err_msg, sugg = common.getMsg(self.lang,
                                              "pdt.version.not.meet", "",
                                              (SN,
                                               sn_check_dict["now_version"],
                                               sn_check_dict["aim_version"]))
                err_msg_list.append(err_msg)
                sugg_list.append(sugg)

        if not self.result_dict["flag"]:
            self.result_dict["errMsg"] = err_msg_list[0]
            self.result_dict["suggestion"] = "\n".join(sugg_list)
            return

    def check_pdt_model_enum_consistent(self, new_dev_info):
        """ 检查产品型号枚举一致性

        :param new_dev_info: 新节点设备信息
        :return:
        """
        key = DevInfoKey.PDT_MODEL_ENUM
        return (not new_dev_info.get(key)
                or new_dev_info.get(key) == self.origin_dev_info.get(key))

    def check_pdt_model_string_consistent(self, new_dev_info):
        """ 检查产品型号string一致性

        :param new_dev_info: 新节点设备信息
        :return:
        """
        key = DevInfoKey.PDT_MODEL_STRING
        return (not new_dev_info.get(key)
                or not self.origin_dev_info.get(key)
                or new_dev_info.get(key) == self.origin_dev_info.get(key))

    def check_pdt_oem_model_consistent(self, new_dev_info):
        """ 检查产品白牌型号string一致性

        :param new_dev_info: 新节点设备信息
        :return:
        """
        key = DevInfoKey.PDT_OEM_MODEL
        return (not new_dev_info.get(key)
                or not self.origin_dev_info.get(key)
                or new_dev_info.get(key) == self.origin_dev_info.get(key))

    def check_internal_model_consistent(self, new_dev_info):
        """ 检查内部型号一致性

        :param new_dev_info: 新节点设备信息
        :return:
        """
        origin_internal_model = \
            self.origin_dev_info.get(DevInfoKey.INTERNAL_MODEL)
        new_internal_model = new_dev_info.get(DevInfoKey.INTERNAL_MODEL)
        if self.need_check_internal_model():
            return (not origin_internal_model
                    or not new_internal_model
                    or origin_internal_model == new_internal_model
                    or self.is_ori_c_and_new_c_plus(new_dev_info))
        return True

    def is_ori_c_and_new_c_plus(self, new_dev_info):
        """ 判断是否为C扩C+

        :param new_dev_info: 新节点设备信息
        :return:
        """
        origin_internal_model = \
            self.origin_dev_info.get(DevInfoKey.INTERNAL_MODEL)
        new_internal_model = new_dev_info.get(DevInfoKey.INTERNAL_MODEL)
        return new_internal_model == origin_internal_model + "_P"

    def check_origin_version(self, new_dev_info, aim_version):
        """ 检查原集群及新集群版本号

        :param new_dev_info: 新节点设备信息
        :param aim_version: 目标版本
        :return:
        """
        return new_dev_info[DevInfoKey.PDT_VERSION] < aim_version, \
            self.origin_dev_info[DevInfoKey.PDT_VERSION] < aim_version

    def need_check_internal_model(self):
        """ 判断原集群和待扩控版本，是否需要检查内部型号

        :return:
        """
        ori_pdt_ver = contextUtil.getItem(self.context, "productVersion")
        new_pdt_ver = contextUtil.getItem(self.context, "new_pdt_ver")
        return ori_pdt_ver >= "V500R007C70" and new_pdt_ver >= "V500R007C70"

    def get_aim_version(self, new_dev_info):
        if self.new_config_clust_type == "switch":
            # V5中端
            if self.origin_dev_info.get(DevInfoKey.PDT_MODEL_STRING).upper() \
                    in config.V5_MIDDLE_NEW:
                aim_version = VersionManager.V5_MIDDLE_SWITCH
            else:
                # V5高端
                aim_version = VersionManager.V5_SWITCH
        elif self.is_ori_c_and_new_c_plus(new_dev_info):
            aim_version = VersionManager.V5_SWITCH
        else:
            aim_version = VersionManager.V5_DIRECT

        return aim_version
