# coding=UTF-8
# Copyright (c) Huawei Technologies Co., Ltd. 2019-2024. All rights reserved.
# noinspection PyUnresolvedReferences
import traceback

# noinspection PyUnresolvedReferences
from java.lang import Exception as JException
from cbb.business.operate.expansion import common
from cbb.common.conf.productConfig import VersionManager
from cbb.common.conf.productConfig import compare_dorado_version
from cbb.frame.context import contextUtil
from cbb.frame.cli import cliUtil
from cbb.frame.tlv import adTlvUtil
from cbb.frame.tlv import tlvData
from cbb.frame.tlv import tlvUtil
from cbb.frame.base import config
from cbb.frame.rest import restUtil
from cbb.frame.rest import restData
from cbb.frame.base import baseUtil
# noinspection PyUnresolvedReferences
from utils import Products

SUPPORT_DIF_CPU_EXP_VERSION = "6.1.5"

NAS_CTRL_NUM = 8  # NAS双活最多支持扩到8控
NAS_CTRL_NUM_PLUS = 16  # D5600/D6000 V7及以后版本NAS双活最多支持扩到16控
nas_sup_model_list = ["OceanStor Dorado 3000 V6", "OceanStor Dorado 5300 V6",
                      "OceanStor 5310", "OceanStor 5320", "OceanStor 5300K",
                      "OceanStor 5310 Capacity Flash", "OceanStor 5510 Capacity Flash", "OceanStor A300",
                      "OceanStor 2600", "OceanStor 2620", "OceanStor 5210", "OceanStor 5220",
                      "OceanStor Dorado 3000", "OceanStor Dorado 5300"]
nas_no_sup_c_exp_c_plus_list = ["OceanStor Dorado 5000 V6",
                                "OceanStor Dorado 5500 V6",
                                "OceanStor Dorado 5000",
                                "OceanStor Dorado 5500"]

# NAS内部型号标识
NAS_FLAG = "_F"
# NAS C+内部型号标识
NAS_AND_C_PLUS_FLAG = "_C_F"
# 新融合海外型号isOverseas的值
OVERSEAS_MODEL_VALUE = "1"

# 支持非内双活组网引擎间3P扩1P扩控的设备型号
PRODUCT_MODEL_TUPLE = ("OceanProtect X8000", "OceanProtect E8000", "OceanProtect X8000K")

INNER_MODEL_1P = "_1P"


class DevInfoKey:
    """ 设备信息key """
    PDT_MODEL = "product_model"  # 产品型号
    INTERNAL_MODEL = "internal_model"  # 内部型号
    # 是否为海外版本
    IS_OVERSEAS = "is_overseas"
    CPU_MODEL = "cpu_model"  # CPU型号
    CPU_NUMBER = "cpu_number"  # CPU数量
    PDT_VERSION = "product_version"  # 产品版本
    SN = "sn"  # 原框SN


class CheckNewPdtModelForSolid:
    """ 扩控前新扩控产品型号一致性检查（Dorado固态存储）"""

    def __init__(self, context):
        self.context = context
        self.result_dict = {"flag": True, "errMsg": "", "suggestion": ""}
        self.tlv = contextUtil.getTlv(context)
        self.cli = contextUtil.getCli(context)
        self.rest = contextUtil.getRest(context)
        self.lang = contextUtil.getLang(context)
        self.logger = common.getLogger(context.get("logger"), __file__)
        self.new_boards_list = contextUtil.getItem(context, "newBoardsList")
        self.new_config_clust_type = contextUtil.getItem(context, "newConfigClustType")
        self.has_inner_metro = contextUtil.getItem(context, "hasInnerMetroLicense")
        self.origin_dev_info = dict()
        self.init_origin_dev_infos()
        self.has_hyper_metro_domain = ""
        self.con_exp_4p_to_2p = False
        self.con_exp_3p_to_1p = False

    @staticmethod
    def is_exp_4p_to_2p(new_internal_model, origin_internal_model):
        """
        4P扩2P的场景，是否为4P扩2P
        4P（C+）扩2P（C+）
        4P（C）扩2P（C+）
        2P设备不存在C的内部型号。
        @param new_internal_model: 待扩控的内部型号
        @param origin_internal_model:原扩控的内部型号
        @return:是否为4P扩2P
        """
        return origin_internal_model + "_2P" == new_internal_model or \
               origin_internal_model + "_C_2P" == new_internal_model

    @staticmethod
    def is_exp_2p_to_4p(new_internal_model, origin_internal_model):
        """
        2P设备：内部型号中包含_2P,4P设备不包含_2P. 是否为2P扩4P
        @param new_internal_model:待扩内部型号
        @param origin_internal_model: 原扩内部型号。
        @return: 是否为2P扩4P
        """
        return "_2P" in origin_internal_model and "_2P" not in new_internal_model

    def init_origin_dev_infos(self):
        """ 初始化原集群信息

        :return:
        """
        ori_sys_record = tlvUtil.getSystemRecord(self.tlv)

        origin_pdt_model = contextUtil.getItem(self.context, "productModel")
        internal_dev_info = tlvUtil.getInternalDeviceInfo(self.tlv)
        origin_cpu_model = tlvUtil.getCpuModel(internal_dev_info)
        origin_cpu_number = tlvUtil.getCpuNumber(internal_dev_info)
        origin_internal_model = tlvUtil.getInternalPdtModel(internal_dev_info)
        origin_overseas = tlvUtil.get_overseas(internal_dev_info)
        ori_full_product_version = tlvUtil.getFullProductVersion(self.tlv)
        ori_pdt_sn = tlvUtil.getRecordValue(
            ori_sys_record, tlvData.SYSTEM['SN'])

        self.origin_dev_info[DevInfoKey.PDT_MODEL] = origin_pdt_model
        self.origin_dev_info[DevInfoKey.INTERNAL_MODEL] = origin_internal_model
        self.origin_dev_info[DevInfoKey.IS_OVERSEAS] = origin_overseas
        self.origin_dev_info[DevInfoKey.CPU_MODEL] = origin_cpu_model
        self.origin_dev_info[DevInfoKey.CPU_NUMBER] = origin_cpu_number
        self.origin_dev_info[DevInfoKey.PDT_VERSION] = ori_full_product_version
        self.origin_dev_info[DevInfoKey.SN] = ori_pdt_sn
        self.logger.logInfo("origin device info: {}".format(
            self.origin_dev_info))

    def get_new_dev_info(self, board):
        """ 获取新集群节点信息

        :param board: 节点
        :return: 新集群节点信息字典
        """
        new_dev_info = dict()
        new_internal_info = adTlvUtil.getInternalDeviceInfo(self.tlv, board)
        new_cpu_model = adTlvUtil.getCpuModel(new_internal_info)
        new_cpu_number = adTlvUtil.getCpuNumber(new_internal_info)
        new_internal_model = adTlvUtil.getInternalPdtModel(new_internal_info)
        new_overseas = adTlvUtil.get_overseas(new_internal_info)
        new_pdt_model = adTlvUtil.getProductModel(self.tlv, board)
        new_pdt_version, _ = adTlvUtil.getProductVersion(self.tlv, board)

        new_dev_info[DevInfoKey.PDT_MODEL] = new_pdt_model
        new_dev_info[DevInfoKey.INTERNAL_MODEL] = new_internal_model
        new_dev_info[DevInfoKey.IS_OVERSEAS] = new_overseas
        new_dev_info[DevInfoKey.CPU_MODEL] = new_cpu_model
        new_dev_info[DevInfoKey.CPU_NUMBER] = new_cpu_number
        new_dev_info[DevInfoKey.PDT_VERSION] = new_pdt_version
        self.logger.logInfo("board: {}, device info: {}".format(board,
                                                                new_dev_info))
        return new_dev_info

    def execute(self):
        """ 执行入口

        :return:
        """
        try:
            self.check()
            if not self.result_dict.get("flag"):
                contextUtil.handleFailure(self.context, self.result_dict)
                return
            contextUtil.handleSuccess(self.context)
            self.logger.logPass()
            return
        except (Exception, JException) as ex:
            self.logger.logException(ex)
            contextUtil.handleException(self.context, ex)
            return

    def check(self):
        # 3000/5300 V6,并开启NAS,检查待扩控版本
        if not self.is_new_controller_version_meet_for_low_end_nas():
            self.logger.logInfo("low end nas device version not meet.")
            return

        # nas双活，双活任务状态不能为Recovering
        if self.is_fs_hyper_metro_with_recovering_status():
            self.logger.logInfo("fs hyper metro status has recovering.")
            return

        # 存在NAS双活，最大不能超过扩到8控
        if not self.is_controller_num_meet_for_nas_hypermetro():
            self.logger.logInfo("controller num large than 8.")
            return

        # 检查待扩控产品型号是否一致
        if not self.check_product_model_consistent():
            self.logger.logInfo("product model not consistent.")
            return

        # X8000,X9000,A8000，微存储不需要检查待扩版本
        if self.origin_dev_info[DevInfoKey.PDT_MODEL] in config.OCEAN_PROTECT + config.OCEAN_STOR_MICRO_DEVS:
            self.logger.logInfo("is ocean protect or micro version, return")
            return

        # 检查待扩控是否满足最低版本要求
        self.check_new_controller_least_version()

    def check_new_controller_least_version(self):
        self.logger.logInfo("check all new board.")
        enclosure_dict = {}
        enclosure_dict.setdefault(self.origin_dev_info[DevInfoKey.SN], {})
        enclosure_dict[self.origin_dev_info[DevInfoKey.SN]][
            "need_upgrade"] = False
        enclosure_dict[self.origin_dev_info[DevInfoKey.SN]]["now_version"] = \
            self.origin_dev_info[DevInfoKey.PDT_VERSION]
        enclosure_dict[self.origin_dev_info[DevInfoKey.SN]]["aim_version"] = ""

        # 遍历待扩控节点检查
        for board in self.new_boards_list:
            # 待扩控节点信息
            if not self.check_one_controller_least_version(board, enclosure_dict):
                return
        self.set_check_result(enclosure_dict)

    def check_one_controller_least_version(self, board, enclosure_dict):
        board_sn = board["enclosureSN"]
        self.logger.logInfo("check board sn = {}".format(board_sn))
        new_dev_info = self.get_new_dev_info(board)
        aim_version = self.get_aim_version(new_dev_info)
        self.logger.logInfo("aim version={}".format(aim_version))
        if board_sn not in enclosure_dict:
            enclosure_dict.setdefault(board_sn, {})
            enclosure_dict[board_sn]["need_upgrade"] = False
            enclosure_dict[board_sn]["now_version"] = \
                new_dev_info[DevInfoKey.PDT_VERSION]
            enclosure_dict[board_sn]["aim_version"] = ""
        if self.is_ori_c_and_new_c_plus(new_dev_info):
            if self.is_dorado_with_hyper_metro_when_c_expansion_c_plus(new_dev_info):
                return False

            # 检查内核版本
            kernel_qualified, new_need_ver_upgrade, ori_need_ver_upgrade = \
                self.check_when_c_exp_c_plus(new_dev_info, aim_version)
            # 内核有问题，直接报出升级
            if not kernel_qualified:
                self.result_dict["flag"] = False
                self.result_dict["errMsg"], self.result_dict["suggestion"] \
                    = common.getMsg(self.lang, "kernel.version.not.meet")
                return False
        else:
            new_need_ver_upgrade, ori_need_ver_upgrade = \
                self.check_origin_version(new_dev_info, aim_version)

        # 若新集群需要升级
        if new_need_ver_upgrade:
            enclosure_dict[board_sn]["need_upgrade"] = True
            # 待升级级别一定要是最高的级别
            if aim_version > enclosure_dict[board_sn]["aim_version"]:
                enclosure_dict[board_sn]["aim_version"] = aim_version
        # 若原集群需要升级
        if ori_need_ver_upgrade:
            enclosure_dict[self.origin_dev_info[DevInfoKey.SN]][
                "need_upgrade"] = True
            if aim_version > enclosure_dict[self.origin_dev_info[DevInfoKey.SN]]["aim_version"]:
                enclosure_dict[self.origin_dev_info[DevInfoKey.SN]][
                    "aim_version"] = aim_version
        return True

    def set_check_result(self, enclosure_dict):
        err_msg_list = list()
        sugg_list = list()
        for (SN, sn_check_dict) in enclosure_dict.items():
            if sn_check_dict["need_upgrade"]:
                self.result_dict["flag"] = False
                err_msg, sugg = common.getMsg(self.lang,
                                              "pdt.version.not.meet", "",
                                              (SN,
                                               sn_check_dict["now_version"],
                                               sn_check_dict["aim_version"]))
                err_msg_list.append(err_msg)
                sugg_list.append(sugg)
        if not self.result_dict["flag"]:
            self.result_dict["errMsg"] = err_msg_list[0]
            self.result_dict["suggestion"] = "\n".join(sugg_list)

    def is_cpu_num_diff(self, new_dev_info):
        return int(self.origin_dev_info.get(DevInfoKey.CPU_NUMBER)) > int(new_dev_info.get(DevInfoKey.CPU_NUMBER))

    def check_product_model_consistent(self):
        err_msg_list = list()
        sugg_list = list()
        # 遍历待扩控节点检查
        for board in self.new_boards_list:
            # 待扩控节点信息
            new_dev_info = self.get_new_dev_info(board)

            # 检查产品型号是否一致
            pdt_model_flag = self.check_pdt_model_consistent(new_dev_info)

            # 检查内部型号是否一致
            internal_model_flag = self.check_internal_model_consistent(new_dev_info)
            if self.result_dict.get("errMsg") and not internal_model_flag:
                return False
            if not pdt_model_flag or not internal_model_flag:
                self.result_dict["flag"] = False
                err_key = "pdt.controller.bom.inconsistent" if \
                    contextUtil.getItem(self.context, "overseas_flag") else \
                    "pdt.controller.inconsistent"
                err_msg, sugg = common.getMsg(self.lang, err_key,
                                              board["enclosureSN"])
                if err_msg not in err_msg_list:
                    err_msg_list.append(err_msg)
                sugg_list.append(sugg)
        if not self.result_dict["flag"]:
            self.result_dict["errMsg"] = "\n".join(err_msg_list)
            self.result_dict["suggestion"] = sugg_list[0]
            return False
        return True

    def is_dorado_with_hyper_metro_when_c_expansion_c_plus(self, new_dev_info):
        """
        5000,5500V6 带有fs双活时，不支持c扩c+场景
        2022.3 更新：dorado 6.1.5 支持不同cpu个数调用rest接口评估是否扩容
        :return:
        """
        product_model = self.origin_dev_info.get(DevInfoKey.PDT_MODEL)
        self.logger.logInfo("product_model: {}".format(product_model))
        if product_model not in nas_no_sup_c_exp_c_plus_list or not self.has_hyper_metro_domain:
            return False
        product_version = self.origin_dev_info.get(DevInfoKey.PDT_VERSION)
        if not self.check_cpu_diff_kernel_version(product_version):
            return True

        if self.can_exp_ctrl_for_diff_cpu(is_dorado_3000=False):
            self.logger.logInfo("can_exp_ctrl_for_diff_cpu exp ctrl.{}".format(product_model))
            return True
        self.logger.logInfo("can not exp ctrl.{}".format(product_model))
        self.result_dict["flag"] = False
        self.result_dict["errMsg"], self.result_dict["suggestion"] = common.getMsg(
            self.lang, "NAS_hyper_can_not_exp")
        return True

    def get_aim_version(self, new_dev_info):
        """
        获取待扩控的最低版本要求
        :param new_dev_info: 设备信息
        :return: 最低版本
        """
        current_version = self.origin_dev_info[DevInfoKey.PDT_VERSION]
        if Products.compareVersion(
                current_version, VersionManager.V6_SUPPORT_16_CTRL) >= 0 \
                and self.new_config_clust_type == "switch":
            return VersionManager.V6_SUPPORT_16_CTRL
        if self.is_ori_c_and_new_c_plus(new_dev_info):
            return VersionManager.V6_SWITCH_AND_CP
        if self.new_config_clust_type == "direct":
            return VersionManager.V6_DIRECT
        return VersionManager.V6_SWITCH

    def check_pdt_model_consistent(self, new_dev_info):
        """ 检查产品型号一致性

        :param new_dev_info: 新节点设备信息
        :return:
        """
        key = DevInfoKey.PDT_MODEL
        if not self.check_overseas_for_hybrid_v6(new_dev_info):
            return False
        return (not new_dev_info.get(key)
                or new_dev_info.get(key) == self.origin_dev_info.get(key))

    def check_overseas_for_hybrid_v6(self, new_dev_info):
        """
        检查新融合是否为海外版本和国内版本互扩
        :param new_dev_info: 新节点设备信息
        :return: True:不是海外国内互扩， False：海外国内互扩
        """
        overseas_key = DevInfoKey.IS_OVERSEAS
        product_model = self.origin_dev_info.get(DevInfoKey.PDT_MODEL)
        product_version = self.origin_dev_info.get(DevInfoKey.PDT_VERSION)
        # 新融合6.1.3及以后不支持海外版本和国内版本互扩
        if product_model not in config.NEW_DORADO:
            return True
        if new_dev_info.get(overseas_key) == \
                self.origin_dev_info.get(overseas_key):
            return True
        # 新融合6.1.3RC2 不支持扩海外版本,海外版本的isoverseas 字段值为1
        if Products.compareVersion(product_version, "6.1.3") < 0 and \
                str(new_dev_info.get(overseas_key)) != OVERSEAS_MODEL_VALUE:
            return True
        contextUtil.setItem(self.context, "overseas_flag", True)
        return False

    def is_dorado615_or_later(self):
        product_version = self.origin_dev_info.get(DevInfoKey.PDT_VERSION)
        return compare_dorado_version(
            product_version[:len(SUPPORT_DIF_CPU_EXP_VERSION)], SUPPORT_DIF_CPU_EXP_VERSION)[1]

    def check_internal_model_consistent(self, new_dev_info):
        """
        检查内部型号一致性,内部型号一致，通过
        特殊情况：1、C扩C+，通过；2、NAS扩非NAS：通过
        其余情况下内部型号不一致，不通过
        :param new_dev_info: 新节点设备信息
        :return:
        """
        # 5000 V6/3000 V6
        product_model = self.origin_dev_info.get(DevInfoKey.PDT_MODEL)
        self.logger.logInfo("check_internal_model_consistent is cpu diff:{}".format(
            self.is_cpu_num_diff(new_dev_info)))
        if all([product_model not in nas_sup_model_list,
                self.is_dorado615_or_later(),
                self.has_hyper_metro_domain,
                self.is_cpu_num_diff(new_dev_info)
                ]):
            # 如果CPU个数不同，且存在nas双活，需要调用接口
            self.logger.logInfo(
                "check exp ctrl for nas and diff cpu{}".format(
                    product_model))
            if not self.can_exp_ctrl_for_diff_cpu(is_dorado_3000=False):
                self.logger.logInfo("can not exp ctrl for nas and diff cpu{}".format(product_model))
                self.result_dict["flag"] = False
                self.result_dict["errMsg"], self.result_dict[
                    "suggestion"] = common.getMsg(self.lang, "NAS_hyper_can_not_exp")
                return False
            self.logger.logInfo("can exp ctrl for nas and diff cpu.{}".format(product_model))

        # 6.1.3以前不支持非NAS扩NAS
        if self.is_no_nas_expand_nas_and_version_less_than_613(new_dev_info):
            return False
        if not self.check_cpu_consistent(new_dev_info):
            return False
        if not self.check_exp_3p_to_1p(new_dev_info):
            return False
        return self.is_same_internal_model_or_ori_c_expand_c_plus(new_dev_info)

    def check_cpu_consistent(self, new_dev_info):
        """
        1、当前扩控为非内双活扩控,且6.1.5及之后的版本 支持 4P->2P
        2、2P->4P 不支持
        3、4P->4P 2p->2P支持。
        @param new_dev_info:  带扩控的信息
        @return: 是否支持带扩控。
        """
        self.logger.logInfo("check_cpu_consistent start.")
        product_model = self.origin_dev_info.get(DevInfoKey.PDT_MODEL)
        if not baseUtil.isDoradoV6HighEnd(product_model):
            return True

        origin_internal_model = self.origin_dev_info.get(DevInfoKey.INTERNAL_MODEL, "")
        new_internal_model = new_dev_info.get(DevInfoKey.INTERNAL_MODEL, "")
        # 2P->4P 不支持
        if self.is_exp_2p_to_4p(new_internal_model, origin_internal_model):
            return False
        # 当前扩控为非内双活扩控,且6.1.5及之后的版本 支持 4P->2P
        if not self.is_exp_4p_to_2p(new_internal_model, origin_internal_model):
            return True
        # 内双活 不支持4P->2P 6.1.5之前不支持 4P->2P
        product_version = self.origin_dev_info.get(DevInfoKey.PDT_VERSION)
        if self.has_inner_metro:
            return False
        if not self.check_cpu_diff_kernel_version(product_version):
            return False
        self.logger.logInfo("check_cpu_consistent con expansion 4p to 2p.")
        self.con_exp_4p_to_2p = True
        return True

    def check_cpu_diff_kernel_version(self, product_version):
        cli = contextUtil.getCli(self.context)
        kernel_version = cliUtil.get_kernel_version(cli, self.lang)
        # 6.1.0 版本没有kernel version, 赋值最小值
        kernel_version = "0.0.0.0" if not kernel_version else kernel_version
        err_msg_key = ""
        if Products.compareVersion(product_version, "6.1.3") < 0:
            err_msg_key = "pdt.controller.cpu.inconsistent.roll.upgrade.to.615"
        if Products.compareVersion(product_version, "6.1.3") == 0 and kernel_version < "1.1.5.0":
            err_msg_key = "pdt.controller.cpu.inconsistent.roll.upgrade.to.615"
        if Products.compareVersion(product_version, "6.1.3") == 0 and kernel_version >= '1.1.5.0':
            err_msg_key = "pdt.controller.cpu.inconsistent.quickly.upgrade.to.615"
        if Products.compareVersion(product_version, "6.1.5RC1") >= 0 and kernel_version < '1.1.5.0':
            err_msg_key = "pdt.controller.cpu.inconsistent.need.upgrade.apollo.to.615"

        if not err_msg_key:
            return True
        self.result_dict["flag"] = False
        self.result_dict["errMsg"], self.result_dict["suggestion"] \
            = common.getMsg(self.lang, err_msg_key, errMsgArgs=(product_version, kernel_version))
        return False

    def can_exp_ctrl_for_diff_cpu(self, is_dorado_3000):
        try:
            cmd = "HyperMetroDomain/get_nas_msc_is_support_cpu_inconsistent"
            rest = contextUtil.getRest(self.context)
            uri_param_dict = restUtil.CommonRest.getUriParamDict(cmd)
            exp_param = {"expandControllerScene": "1" if is_dorado_3000 else "0"}
            record = restUtil.CommonRest.execCmd(
                rest, uri_param_dict, exp_param, restData.RestCfg.RestMethod.GET
            )
            return record.get("data", {}).get("isSupportControllerExpand") == "1"
        except Exception:
            self.logger.logInfo(traceback.format_exc())
            return False

    def is_same_internal_model_or_ori_c_expand_c_plus(self, new_dev_info):
        """
        判断是否为相同的内部型号，或者是C扩C+
        :param new_dev_info: 新节点设备信息
        :return: True: 是相同内部型号或C扩C+ False: 内部型号不相同，且不是C扩C+
        """
        origin_internal_model, new_internal_model = \
            self.get_internal_model(new_dev_info)
        return (not new_internal_model
                or self.con_exp_4p_to_2p
                or self.con_exp_3p_to_1p
                or new_internal_model == origin_internal_model
                or self.is_ori_c_and_new_c_plus(new_dev_info))

    def is_ori_c_and_new_c_plus(self, new_dev_info):
        """ 判断是否为C扩C+

        :param new_dev_info: 新节点设备信息
        :return:
        """
        origin_internal_model, new_internal_model = \
            self.get_internal_model(new_dev_info)
        return new_internal_model == origin_internal_model + "_C"

    def is_no_nas_expand_nas_and_version_less_than_613(self, new_dev_info):
        """ 判断是否为版本6.1.3以前的非NAS扩NAS

        :param new_dev_info: 新节点设备信息
        :return: True: 是版本小于6.1.3的非NAS型号扩NAS型号
                 False：不是 版本小于6.1.3的非NAS型号扩NAS型号
        """
        product_version = self.origin_dev_info.get(DevInfoKey.PDT_VERSION)
        # 6.1.3及以后不涉及
        if Products.compareVersion(product_version, "6.1.3") >= 0:
            return False
        origin_internal_model = \
            self.origin_dev_info.get(DevInfoKey.INTERNAL_MODEL)
        new_internal_model = new_dev_info.get(DevInfoKey.INTERNAL_MODEL)
        return any([new_internal_model == origin_internal_model + NAS_FLAG,
                    new_internal_model == origin_internal_model +
                    NAS_AND_C_PLUS_FLAG])

    def check_origin_version(self, new_dev_info, aim_version):
        """ 检查原集群及新集群版本号

        :param new_dev_info: 新节点设备信息
        :param aim_version: 目标版本
        :return:
        """
        res1 = compare_dorado_version(new_dev_info[DevInfoKey.PDT_VERSION],
                                      aim_version)
        res2 = compare_dorado_version(
            self.origin_dev_info[DevInfoKey.PDT_VERSION], aim_version)
        return not (res1[0] and res1[1]), not (res2[0] and res2[1])

    def check_kernel_version(self, new_dev_info):
        """ 检查原集群内核版本，C扩C+时要求

        :param new_dev_info: 新节点设备信息
        :return:
        """
        if not self.is_ori_c_and_new_c_plus(new_dev_info):
            return True
        cli = contextUtil.getCli(self.context)
        kernel_version = cliUtil.get_kernel_version(cli, self.lang)
        self.logger.logInfo("kernel_version:" + kernel_version)
        return kernel_version and \
               kernel_version not in ("1.0.0.1", "1.1.0.0")

    def check_when_c_exp_c_plus(self, new_dev_info, aim_version):
        """在c扩c+时执行检查

        :param new_dev_info 新节点设备信息
        :param aim_version 目标版本
        ：return：是否需要检查内核，新扩控需升级，原扩控需升级
        """
        # 检查原集群版本号
        new_need_ver_upgrade, ori_need_ver_upgrade = self.check_origin_version(
            new_dev_info, aim_version)

        # 若版本号不通过，无需检查内核版本
        if new_need_ver_upgrade or ori_need_ver_upgrade:
            return True, new_need_ver_upgrade, ori_need_ver_upgrade
        # 检查原集群内核版本号
        return self.check_kernel_version(new_dev_info), \
               new_need_ver_upgrade, ori_need_ver_upgrade

    def is_fs_hyper_metro_with_recovering_status(self):
        """
        检查是否存在状态为Recovering的NAS双活
        """
        product_version = self.origin_dev_info.get(DevInfoKey.PDT_VERSION)
        if Products.compareVersion(product_version, "6.1.2RC1") < 0:
            return False

        cmd = "show fs_hyper_metro_domain general"
        flag, cli_ret, err_msg = cliUtil.excuteCmdInCliMode(self.cli, cmd,
                                                            True, self.lang)
        # 不支持NAS，不支持该命令
        if flag is not True and (
                "the current device does not support NAS" in cli_ret or not cliUtil.hasCliExecPrivilege(cli_ret)):
            return False
        if flag is not True:
            self.result_dict["flag"] = False
            self.result_dict["errMsg"], self.result_dict["suggestion"] \
                = common.getMsg(self.lang, "system.abnormal")
            return False
        self.logger.logInfo("cli_ret: {}".format(cli_ret))
        if "Command executed successfully" in cli_ret:
            return False
        hyper_info_list = cliUtil.getHorizontalCliRet(cli_ret)
        for hyper_info in hyper_info_list:
            running_status = hyper_info.get("Running Status")
            if "Recovering" in running_status:
                self.result_dict["flag"] = False
                self.result_dict["errMsg"], self.result_dict["suggestion"] \
                    = common.getMsg(self.lang, "NAS_hyper_failed")
                return True
        self.has_hyper_metro_domain = True
        return False

    def is_controller_num_meet_for_nas_hypermetro(self):
        """
        判断是NAS双活下，最大支持扩到8控
        :return:
        """
        new_config_ctrl_num = contextUtil.getItem(self.context,
                                                  "newConfigCtrlNum")
        self.logger.logInfo("new_ctrl_num:{}".format(new_config_ctrl_num))
        product_model = self.origin_dev_info.get(DevInfoKey.PDT_MODEL)
        product_version = self.origin_dev_info.get(DevInfoKey.PDT_VERSION)
        if baseUtil.is_support_16_ctrl_with_nas(product_model, product_version):
            limit_nas_ctrl_num = NAS_CTRL_NUM_PLUS
            err_msg_key = "NAS.ctrl.more.16"
        else:
            limit_nas_ctrl_num = NAS_CTRL_NUM
            err_msg_key = "NAS.ctrl.more.8"
        if self.has_hyper_metro_domain and \
                int(new_config_ctrl_num) > limit_nas_ctrl_num:
            self.result_dict["flag"] = False
            self.result_dict["errMsg"], self.result_dict["suggestion"] \
                = common.getMsg(self.lang, err_msg_key)
            return False
        return True

    def get_internal_model(self, new_dev_info):
        """
        获取内部型号
        :param new_dev_info:
        :return:
        """
        origin_internal_model = \
            self.origin_dev_info.get(DevInfoKey.INTERNAL_MODEL)
        new_internal_model = new_dev_info.get(DevInfoKey.INTERNAL_MODEL)
        if origin_internal_model.endswith(NAS_FLAG):
            origin_internal_model = origin_internal_model.replace(NAS_FLAG, "")
        if new_internal_model.endswith(NAS_FLAG):
            new_internal_model = new_internal_model.replace(NAS_FLAG, "")
        return origin_internal_model, new_internal_model

    def is_new_controller_version_meet_for_low_end_nas(self):
        """
        OceanStor Dorado 3000/5300 V6且NAS开关打开，新引擎版本要求6.1.2.RC1及以上
        更新：6.1.5 nas 双活dorado 3000 需要接口返回是否可扩。
        :return:
        """
        # 返回 待扩版本是否为6.1.5及以上
        product_model = self.origin_dev_info.get(DevInfoKey.PDT_MODEL)
        if product_model not in nas_sup_model_list:
            return True

        if not self.has_hyper_metro_domain:
            # Dorado 6.1.2和Dorado 6.1.3版本Dorado 3000设备已经创建NAS双活不允许扩控操作。
            if not self.is_v6_low_end_nas():
                return True
            return self.check_new_controller_version()

        self.logger.logInfo("is_dorado615_or_later:{}".format(self.is_dorado615_or_later()))
        if self.is_dorado615_or_later():
            if self.can_exp_ctrl_for_diff_cpu(is_dorado_3000=True):
                self.logger.logInfo(
                    "can_exp_ctrl_for_diff_cpu exp ctrl.{}".format(
                        product_model))
                return True
            self.logger.logInfo("can not exp ctrl.{}".format(product_model))
            self.result_dict["flag"] = False
            self.result_dict["errMsg"], self.result_dict["suggestion"] = common.getMsg(
                self.lang, "NAS_hyper_can_not_exp")
            return False

        self.result_dict["flag"] = False
        self.result_dict["errMsg"], self.result_dict["suggestion"] \
            = common.getMsg(
            self.lang, "3000_v6_open_nas.6.1.5",
            self.origin_dev_info.get(DevInfoKey.PDT_VERSION)
        )

        return False

    def is_v6_low_end_nas(self):
        """
        判断版本是否为 3000/5300 V6 且带NAS
        :return:
        """
        product_model = self.origin_dev_info[DevInfoKey.PDT_MODEL]
        origin_internal_model = \
            self.origin_dev_info.get(DevInfoKey.INTERNAL_MODEL)
        return all([product_model in nas_sup_model_list,
                    origin_internal_model.endswith("_F")])

    def check_new_controller_version(self):
        """
        检查待扩版本是否为6.1.2RC1及以上
        :return:
        """
        for board in self.new_boards_list:
            sn = board.get("enclosureSN")
            new_dev_info = self.get_new_dev_info(board)
            new_product_version = new_dev_info[DevInfoKey.PDT_VERSION]
            self.logger.logInfo(
                "new_product_version: {}".format(new_product_version))
            if Products.compareVersion(new_product_version, "6.1.2RC1") < 0:
                self.result_dict["flag"] = False
                self.result_dict["errMsg"], self.result_dict["suggestion"] \
                    = common.getMsg(self.lang, "3000_v6_open_nas",
                                    (sn, new_product_version))
                return False
        return True

    def check_exp_3p_to_1p(self, new_dev_info):
        """
        支持x8000，x8000k非内双活组网引擎间3P扩1P扩控
        1. 1P扩1P，3P扩3扩通过
        2. 1P扩3P，不通过
        3. 3P扩1P 继续执行检查
        :return: True：支持扩控；False：不支持扩控
        """
        self.logger.logInfo("check exp controller 3P to 1P")
        if not self.is_support_exp_3p_to_1p(self.origin_dev_info.get(DevInfoKey.PDT_MODEL)):
            return True
        origin_internal_model = self.origin_dev_info.get(DevInfoKey.INTERNAL_MODEL, "")
        self.logger.logInfo("In the scenario of ctrl exp, origin_internal_model={}".format(origin_internal_model))
        new_internal_model = new_dev_info.get(DevInfoKey.INTERNAL_MODEL, "")
        self.logger.logInfo("In the scenario of ctrl exp, new_internal_model={}".format(new_internal_model))
        # 1P扩3P 不支持
        if self.is_exp_1p_to_3p(origin_internal_model, new_internal_model):
            err_msg_key = "not.support.exp.controller.1p.to.3p"
            self.result_dict["flag"] = False
            self.result_dict["errMsg"], self.result_dict["suggestion"] = common.getMsg(self.lang, err_msg_key)
            return False
        # 3P扩1P 继续检查
        if not self.is_exp_3p_to_1p(origin_internal_model, new_internal_model):
            return True
        if not self.check_origin_software_and_kernel_version():
            return False
        self.con_exp_3p_to_1p = True
        return True

    def check_origin_software_and_kernel_version(self):
        """
        1. 如果原扩软件版本1.3.RC1之前，提示滚动升级到1.3.RC1及其之后，不通过
        2. 如果原扩软件版本1.3.RC1及其之后，原扩内核版本1.1.6.0之前，提示升级内核版本，不通过
        :return:True：支持；False:不支持
        """
        product_version = self.origin_dev_info.get(DevInfoKey.PDT_VERSION)
        self.logger.logInfo("origin product version={}".format(product_version))
        cli = contextUtil.getCli(self.context)
        kernel_version = cliUtil.get_kernel_version(cli, self.lang)
        # 如果原扩没有kernel version, 赋值最小内核版本1.0.0.1
        kernel_version = "1.0.0.1" if not kernel_version else kernel_version
        self.logger.logInfo("origin kernel version={}".format(kernel_version))
        err_msg_key = ""
        # 情形一
        if Products.compareVersion(product_version, "1.3.RC1") < 0:
            err_msg_key = "exp.controller.roll.upgrade.to.130"
            self.result_dict["errMsg"], self.result_dict["suggestion"] \
                = common.getMsg(self.lang, err_msg_key, errMsgArgs=product_version)
        # 情形二
        if kernel_version < '1.1.6.0':
            err_msg_key = "exp.controller.need.apollo.upgrade"
            self.result_dict["errMsg"], self.result_dict["suggestion"] \
                = common.getMsg(self.lang, err_msg_key, errMsgArgs=kernel_version)
        if not err_msg_key:
            return True
        self.result_dict["flag"] = False
        return False

    def is_support_exp_3p_to_1p(self, product_model):
        """
        是否支持3P扩1P扩控
        :return: True：支持，False：不支持
        """
        return product_model in PRODUCT_MODEL_TUPLE

    def is_exp_1p_to_3p(self, origin_internal_model, new_internal_model):
        """
        1P设备：内部型号中包含_1P，3P设备不包含_1P(不包含则默认为3P)，是否为1P扩3P
        :param origin_internal_model: 原扩内部型号
        :param new_internal_model: 待扩内部型号
        :return: 是否为1P扩3P
        """
        return origin_internal_model.endswith(INNER_MODEL_1P) and not new_internal_model.endswith(INNER_MODEL_1P)

    def is_exp_3p_to_1p(self, origin_internal_model, new_internal_model):
        """
        1P设备：内部型号中包含_1P，3P设备不包含_1P(不包含1P则默认为3P)，是否为3P扩1P
        :param origin_internal_model: 原扩内部型号
        :param new_internal_model: 待扩内部型号
        :return: 是否为3P扩1P
        """
        return all(
            [
                not origin_internal_model.endswith(INNER_MODEL_1P) and new_internal_model.endswith(INNER_MODEL_1P),
                self.is_mix_flash_exp_ctrl(origin_internal_model, new_internal_model),
            ]
        )

    def is_mix_flash_exp_ctrl(self, origin_model, new_model):
        """
        X8000 和 x8000K 内部形态必须都为 全闪 或 混闪。不能混合扩容。
        :param origin_model: 原扩内部型号
        :param new_model: 待扩内部型号
        :return: 是否不为混阔
        """
        return any(
            [
                origin_model in config.FLASHING_INTER_MODEL and new_model in config.FLASHING_INTER_MODEL,
                origin_model not in config.FLASHING_INTER_MODEL and new_model not in config.FLASHING_INTER_MODEL,
            ]
        )
