# -*- coding: UTF-8 -*-
from cbb.business.operate.expansion import common
from cbb.frame.context import contextUtil
from cbb.frame.base import baseUtil
from cbb.frame.rest import restUtil
from cbb.frame.tlv import adTlvUtil
from cbb.frame.tlv import tlvData
from utils import Products

# 需要检查的风险型号
RISK_MODEL = ["OceanStor Dorado 5600 V6", "OceanStor Dorado 6000 V6"]

VERSION_610 = "6.1.0"
VERSION_615RC1 = "6.1.5RC1"
VERSION_616RC1 = "6.1.6RC1"
VERSION_12RC2 = "1.2.RC2"
VERSION_13RC1 = "1.3.RC1"
APOLLO_VER_5 = "1.1.5.0"
APOLLO_VER_6 = "1.1.6.0"
APOLLO_VER_0 = "1.0.0.1"

PRODUCT_VERSION_NOT_MATCH_KEY = "apollo.version.and.intf.card.not.match.low.version"
APOLLO_VERSION_NOT_MATCH_KEY = "apollo.version.and.intf.card.not.match"

# 1822 C+ 接口卡model
C_PLUS_INTF_MODEL = [
    2348, 2349, 2350, 2351, 2352, 2353, 537, 2331,
    2333, 2334, 2335, 2336, 2337, 2338, 2339, 2340,
    2354, 2355, 2356, 2357, 2358, 2359, 601, 2332,
    2341, 2342, 2343, 2344, 2345, 2346, 2347
]

# I02.0接口卡model
IO_INTERFACE_MODEL = [
    2390, 2391, 2375, 2378, 2379, 2380, 2381,
    2382, 2383, 2384, 2385, 2386, 2387, 2389, 2393,
    2394, 2395, 2396, 2397, 2398, 2399, 2400, 2401, 2402,
    2403, 2404, 2405, 2406, 2407, 2408, 2409, 2410, 2411,
    2414, 2415, 2416
]

# 新增接口卡，软件版本6.1.6，apollo版本1.1.5.0
NEW_INTERFACE_MODEL_FOR_616_WITH_APOLLO_1150 = [2417, 2376, 2377, 2388, 2412, 2413]

# 软件版本6.1.6RC1，apollo版本1.1.6.0
NEW_INTERFACE_MODEL_FOR_616_WITH_APOLLO_1160 = [2419, 2420, 2421, 2422]

# Dorado新增接口卡枚举和版本要求字典，key：接口卡枚举元组，value：（软件版本，apollo版本）
DORADO_NEW_INTERFACE_MODEL_MATCH_VERSION_DICT = {
    tuple(IO_INTERFACE_MODEL): (VERSION_615RC1, APOLLO_VER_5),
    tuple(NEW_INTERFACE_MODEL_FOR_616_WITH_APOLLO_1150): (VERSION_616RC1, APOLLO_VER_5),
    tuple(NEW_INTERFACE_MODEL_FOR_616_WITH_APOLLO_1160): (VERSION_616RC1, APOLLO_VER_6)
}

# 二级存储新增接口卡枚举和版本要求字典，key：接口卡枚举元组，value：（软件版本，apollo版本）
OCEAN_PROTECT_NEW_INTERFACE_MODEL_MATCH_VERSION_DICT = {
    tuple(IO_INTERFACE_MODEL): (VERSION_12RC2, APOLLO_VER_5),
    tuple(NEW_INTERFACE_MODEL_FOR_616_WITH_APOLLO_1150): (VERSION_13RC1, APOLLO_VER_5),
    tuple(NEW_INTERFACE_MODEL_FOR_616_WITH_APOLLO_1160): (VERSION_13RC1, APOLLO_VER_6)
}


# 查询apollo版本的参数
CMD_DEFAULT_TIMEOUT = 60 * 2
CMO_CURRENT_VERSION = {"index": 66, "type": "string"}
CMD_QUERY_APOLLO_SUPPORT = {'cmd': 59594908204, 'timeout': CMD_DEFAULT_TIMEOUT}


def execute(context):
    """
    检查待扩引擎是否支持扩控
    :param context:
    :return:
    """
    CheckNewEngines(context).check()


class CheckNewEngines:
    def __init__(self, context):
        self.logger = common.getLogger(context.get("logger"), __file__)
        self.rest = contextUtil.getRest(context)
        self.tlv = contextUtil.getTlv(context)
        self.result_dict = {"flag": True, "errMsg": "", "suggestion": ""}
        self.lang = contextUtil.getLang(context)
        self.context = context
        self.product_model = contextUtil.getItem(self.context, "productModel")
        self.product_version = contextUtil.getItem(self.context,
                                                   "productVersion")
        self.apollo_version = self.get_apollo_version()
        self.interface_model_list = list()

    def check(self):
        try:
            # 获取带扩控的接口卡枚举列表
            self.interface_model_list = self.get_interface_model()

            # 检查接口卡是否满足版本要求
            required_product_version, required_apollo_version = self.get_required_product_version_and_apollo_version()
            if required_product_version:
                self.set_err_msg(PRODUCT_VERSION_NOT_MATCH_KEY,
                                 (self.product_version, required_product_version, required_product_version))
                return
            if required_apollo_version:
                self.set_err_msg(APOLLO_VERSION_NOT_MATCH_KEY)
                return

            # 检查Dorado 5600/6000的接口卡是否兼容
            self.check_interface_model_compatible_for_dorado_5600_and_6000()

        except Exception as e:
            contextUtil.handleException(self.context, e)
            self.logger.logException(e)

    @staticmethod
    def get_max_required_version(required_version_list):
        """
        获取最大要求版本
        @param required_version_list: 版本列表
        """
        required_version = ""
        for version in required_version_list:
            if not version:
                continue
            if not required_version or Products.compareVersion(required_version, version) < 0:
                required_version = version
        return required_version

    def get_required_product_version_and_apollo_version(self):
        """
        获取要求的软件版本和apollo版本
        """
        required_product_version_list = list()
        required_apollo_version_list = list()
        for interface_model in self.interface_model_list:
            required_product_version, required_apollo_version = self.get_required_version(interface_model)
            required_product_version_list.append(required_product_version)
            required_apollo_version_list.append(required_apollo_version)
        required_product_version = self.get_max_required_version(required_product_version_list)
        required_apollo_version = self.get_max_required_version(required_apollo_version_list)
        return required_product_version, required_apollo_version

    def get_required_version(self, interface_model):
        """
        获取要求的版本
        @param interface_model:接口卡枚举
        """
        required_product_version = ""
        required_apollo_version = ""
        match_dict = OCEAN_PROTECT_NEW_INTERFACE_MODEL_MATCH_VERSION_DICT if baseUtil.is_ocean_protect(
            self.product_model) else DORADO_NEW_INTERFACE_MODEL_MATCH_VERSION_DICT
        for models, required_version in match_dict.items():
            if int(interface_model) not in models:
                continue
            # 软件版本不满足
            if Products.compareVersion(self.product_version, required_version[0]) < 0:
                required_product_version = required_version[0]
            if Products.compareVersion(self.apollo_version, required_version[1]) < 0:
                required_apollo_version = required_version[1]
            return required_product_version, required_apollo_version
        return required_product_version, required_apollo_version

    def check_interface_model_compatible_for_dorado_5600_and_6000(self):
        """
        检查Dorado 5600/6000接口卡是否兼容
        @return:
        """
        # 检查设备型号
        if any([self.product_model not in RISK_MODEL,
                not self.has_c_plus_interface_module()]):
            contextUtil.handleSuccess(self.context)
            return

        # 检查版本，小于6.1.0，不通过
        if Products.compareVersion(self.product_version, VERSION_610) < 0:
            self.set_err_msg(
                "apollo.version.and.intf.card.not.match.low.version",
                (self.product_version, VERSION_610, VERSION_610))
            return

        # 检查Apollo版本信息
        if self.apollo_version != APOLLO_VER_0:
            contextUtil.handleSuccess(self.context)
            return

        self.set_err_msg("apollo.version.and.intf.card.not.match")

    def get_apollo_version(self):
        """
        获取apollo版本
        @return: apollo版本
        """
        # Dorado 610以前的apollo版本都为1.0.0.1
        if self.product_version.startswith("6.") and \
                Products.compareVersion(self.product_version, VERSION_610) < 0:
            return APOLLO_VER_0

        recs = restUtil.Tlv2Rest.execCmd(self.rest,
                                         CMD_QUERY_APOLLO_SUPPORT, [])
        if not recs:
            raise Exception("Failed to get apollo version")
        return restUtil.Tlv2Rest.getRecordValue(recs[0], CMO_CURRENT_VERSION)

    def has_c_plus_interface_module(self):
        """
        判断待扩引擎是否有C+卡
        :return:
        """
        return self.has_risk_interface_model(C_PLUS_INTF_MODEL)

    def has_risk_interface_model(self, model_list):
        """
        判断是否有风险接口卡
        @param model_list: 风险接口卡列表
        @return:
        """
        for model in self.interface_model_list:
            if int(model) in model_list:
                return True
        return False

    def get_interface_model(self):
        """
        获取带扩控的接口卡枚举
        @return: 带扩控的接口卡枚举列表
        """
        new_boards_list = contextUtil.getItem(self.context, "newBoardsList")

        model_list = list()
        for board in new_boards_list:
            model_list.extend(self.get_interface_model_on_one_board(board))

        return model_list

    def get_interface_model_on_one_board(self, board):
        """
        获取节点上的接口卡枚举
        @param board: 节点
        @return: 接口卡枚举列表
        """
        records = adTlvUtil.getInterfaceModuleRecords(self.tlv, board)
        if not records:
            return []
        model_list = list()
        for record in records:
            model_list.append(
                adTlvUtil.getRecordValue(record, tlvData.PUB_ATTR['model']))

        return model_list

    def set_err_msg(self, key, params=()):
        """
        设置错误消息
        :param key:
        :param params:
        :return:
        """
        self.result_dict["flag"] = False
        self.result_dict["errMsg"], self.result_dict["suggestion"] = \
            common.getMsg(self.lang, key, suggestionArgs=params)

        contextUtil.handleFailure(self.context, self.result_dict)
