# -*- coding: UTF-8 -*-
from cbb.business.operate.expansion import common
from cbb.frame.context import contextUtil
from cbb.frame.rest.restUtil import CommonRestService
from cbb.frame.base import baseUtil
from cbb.frame.base.config import DORADO_DEVS_V6_MID
from cbb.frame.base.config import DORADO_DEVS_V6_HIGH
from cbb.business.operate.fru.common import FuncFactory
from cbb.frame.rest import restData
from cbb.frame.rest import restUtil
from utils import Products

# 当“containerSpec”字段取值为0时，表示容器未使能
CONTAINER_SPEC = "0"
# 当前控制器数量
CURRENT_CTRL_NUM = 6
# 扩后控制器数量
NEW_CTRL_NUM = 8

# 不支持交换机组网的 25G SO RDMA卡枚举
NOT_SUPPORT_SWITCH_NETWORK_INTF = [2309, 2338, 2361, 2399]


def execute(context):
    CheckNetWorkType(context).check()


class CheckNetWorkType:
    def __init__(self, context):
        self.logger = common.getLogger(context.get("logger"), __file__)
        self.result_dict = {"flag": True, "errMsg": "", "suggestion": ""}
        self.lang = contextUtil.getLang(context)
        self.context = context
        self.product_model = contextUtil.getItem(self.context, "productModel")
        self.product_version = contextUtil.getItem(self.context, "productVersion")
        self.logger.logInfo("product_model={}, product_version={}".format(self.product_model, self.product_version))

    def check(self):

        try:
            # 判断是否交换机组网 + 25G RDMA卡
            if self.is_not_support_switch_network():
                self.result_dict["flag"] = False
                self.result_dict["errMsg"], self.result_dict["suggestion"] = common.getMsg(
                    self.lang, "rdma.25gb.intf.card.not.support.switch",
                    errMsgArgs=contextUtil.getItem(self.context, "productVersion")
                )
                self.logger.logInfo("exist rdma 25gb intf card not support connect to switch network.")
                contextUtil.handleFailure(self.context, self.result_dict)
                return

            # 判断是否为风险版本
            if not self.is_risk_model_and_version():
                contextUtil.handleSuccess(self.context)
                return

            # 判断原阵列端容器是否使能
            if not self.is_support_container():
                contextUtil.handleSuccess(self.context)
                return

            # 检查组网类型,二级存储不支持交换机组网
            # Dorado、新融合6.1.5及以后支持交换机组网，6.1.5以前不支持交换机组网
            if self.check_network_type():
                contextUtil.handleSuccess(self.context)
                return

            self.result_dict["flag"] = False
            self.result_dict["errMsg"], self.result_dict["suggestion"] = \
                common.getMsg(self.lang, "container.not.support.switch")
            contextUtil.handleFailure(self.context, self.result_dict)

        except Exception as e:
            contextUtil.handleException(self.context, e)
            self.logger.logException(e)

    def is_not_support_switch_network(self):
        """
        判断6.1.2及之后版本 + RDMA卡枚举[2309, 2338, 2361, 2399] + 交换机组网
        不使用all(), 因为all 会把所有条件都执行完再做判断，此处需要遇错即停。
        :return: True：不支持，False：支持
        """
        return self.is_switch_network() and self.is_not_support_rdma_intf_conn_switch_version() \
               and self.is_exist_not_support_switch_intf()

    def is_exist_not_support_switch_intf(self):
        """
        是否存在 SO 25G RDMA卡枚举[2309, 2338, 2361, 2399]
        :return: True：存在，False：不存在
        """
        interface_model_list = self.get_interface_model()
        self.logger.logInfo("interface_model_list:{}".format(interface_model_list))
        return bool([model for model in interface_model_list if int(model) in NOT_SUPPORT_SWITCH_NETWORK_INTF])

    def is_switch_network(self):
        """
        判断组网类型是否为交换机组网
        :return:
        """
        return contextUtil.getItem(
            self.context, "newConfigClustType") == common.CLUST_TYPE_SWITCH

    def get_interface_model(self):
        """
        获取当前集群的接口卡枚举
        @return: 接口卡枚举列表
        """
        inter_face_list = FuncFactory.getFruListInfo(self.context, restData.Enum.ObjEnum.INTF_MODULE)
        model_list = list()
        for inter_face_info in inter_face_list:
            location = restUtil.Tlv2Rest.getRecordValue(inter_face_info, restData.Hardware.IntfModule.LOCATION)
            if not str(location).endswith("IOM2"):
                continue
            model = restUtil.Tlv2Rest.getRecordValue(inter_face_info, restData.Hardware.IntfModule.MODEL)
            model_list.append(model)
        return model_list

    def is_not_support_rdma_intf_conn_switch_version(self):
        """
        6.1.2 及之后的版本不支持使用SO 25G RDMA卡通过交换机组网扩控
        :return: True：不支持，False：支持
        """
        return self.product_model in DORADO_DEVS_V6_MID and Products.compareVersion(
            self.product_version, "6.1.2RC1") >= 0

    def is_risk_model_and_version(self):
        """
        判断是否为风险型号和版本
        :return:
        """
        if contextUtil.getItem(self.context, "has_container_feature"):
            return True
        if self.product_model not in (DORADO_DEVS_V6_HIGH + DORADO_DEVS_V6_MID):
            return False
        return Products.compareVersion(self.product_version, "6.1.2RC1") >= 0

    def is_support_container(self):
        """
        判断是否支持容器使能
        :return:
        """
        rest = contextUtil.getRest(self.context)
        uri = "get_device_global_config"
        data = CommonRestService.get(rest, uri)
        if not data:
            raise Exception("Error Response")

        return str(data.get("containerSpec", "")) != CONTAINER_SPEC

    def check_network_type(self):
        """
        检查组网类型,
        二级存储不支持交换机组网
         Dorado、新融合6.1.5及以后支持交换机组网，6.1.5以前不支持交换机组网，E8000支持交换机组网
        :return: True: 通过 False： 不通过
        """
        if contextUtil.getItem(self.context, "newConfigClustType") != common.CLUST_TYPE_SWITCH:
            return True
        # E8000支持交换机组网
        if baseUtil.is_ocean_protect_support_switch(self.product_model):
            return True
        return self.product_model in (DORADO_DEVS_V6_HIGH + DORADO_DEVS_V6_MID) and Products.compareVersion(
            self.product_version, "6.1.5") >= 0
