# -*- coding: UTF-8 -*-
import time
from cbb.business.operate.expansion import common
from cbb.frame.context import contextUtil
from cbb.frame.base import baseUtil
from cbb.frame.base.config import DORADO_DEVS_V6_MID
from cbb.frame.base.config import DORADO_DEVS_V6_HIGH
from utils import Products
from urllib import quote

CHECK_TIME = 1800
INTERVAL = 5
SLEEP_TIME = 10

# 高端扩容容器服务参数
high_end_expand_container_param_dict = {
    "4": {"nodeList": ["0C", "0D"]},
    "8": {"nodeList": ["1A", "1B", "1C", "1D"]},
}
# 中端扩容容器服务参数
mid_end_expand_container_param_dict = {
    "4": {"nodeList": ["1A", "1B"]}
}
# 二级存储X系列
OCEAN_PROTECT_X = [
    "OceanProtect X8000", "OceanProtect X9000", "OceanProtect E8000",
    "OceanProtect X6000", "OceanProtect X8000K",
    "OceanProtect X9000K",
]


def execute(context):

    ExpandContainerService(context).execute_container_service()


class ExpandContainerService:
    def __init__(self, context):
        self.logger = common.getLogger(context.get("logger"), __file__)
        self.resultDict = {"flag": True, "errMsg": "", "suggestion": ""}
        self.lang = contextUtil.getLang(context)
        self.context = context
        self.tlv = contextUtil.getTlv(context)
        self.product_model = contextUtil.getItem(context, "productModel")

    def execute_container_service(self):

        try:
            common.threadUpProcess(self.context, CHECK_TIME, INTERVAL)
            # 2引擎起扩，不需要扩容器
            if contextUtil.getItem(self.context, "ctrl_enclosure_num") >= 2 \
                    and not baseUtil.is_ocean_protect_support_switch(self.product_model):
                contextUtil.handleSuccess(self.context)
                return
            # 判断是否为风险型号和版本
            if not self.is_risk_model_and_version():
                contextUtil.handleSuccess(self.context)
                return
            # 判断是否有容器服务
            if not self.has_container_application():
                contextUtil.handleSuccess(self.context)
                return

            # 获取新扩控制器ID
            param_dict = self.get_expand_container_param_dict()
            # 给容器服务添加新扩控制器
            flag, task_id = self.expand_new_controller_to_container(param_dict)
            if not flag:
                self.process_err_info("failed.expand.container.service")
                return
            # 检查新扩控制器是否扩入容器服务
            start_time = time.time()
            while time.time() - start_time <= CHECK_TIME:
                flag, status = self.is_new_controller_in_container(task_id)
                if status == "success":
                    contextUtil.handleSuccess(self.context)
                    return
                if status == "fail":
                    self.set_fail_info(task_id)
                    return
                # 10s查询一次
                time.sleep(SLEEP_TIME)

            self.set_fail_info(task_id)
            return
        except Exception as e:
            contextUtil.handleException(self.context, e)
            self.logger.logException(e)
            return
        finally:
            common.finishProcess(self.context)

    def process_err_info(self, meg_key):
        self.resultDict["flag"] = False
        (
            self.resultDict["errMsg"],
            self.resultDict["suggestion"],
        ) = common.getMsg(self.lang, meg_key)
        contextUtil.handleFailure(self.context, self.resultDict)

    def is_risk_model_and_version(self):
        """
        判断是否为风险型号和版本
        """
        product_version = contextUtil.getItem(self.context, "productVersion")
        if contextUtil.getItem(self.context, "has_container_feature"):
            return True
        if self.product_model == "OceanProtect A8000":
            return True
        if self.product_model in OCEAN_PROTECT_X:
            # 二级存储X系列从1.2.1开始支持容器服务
            return Products.compareVersion(product_version, "1.2.1") >= 0
        if self.product_model not in (DORADO_DEVS_V6_HIGH + DORADO_DEVS_V6_MID):
            return False
        return Products.compareVersion(product_version, "6.1.2RC1") >= 0

    def has_container_application(self):
        """
        判断是否有容器服务
        :return:
        """
        cmd = "/api/v2/task/containerService/serviceStatus"
        response_info = common.get_rest_api_v2_record(self.context, cmd, "GET")
        service_status = response_info.get("data", {}).get("serviceStatus", "")
        return service_status == "Running"

    def expand_new_controller_to_container(self, param_dict):
        """
        给容器服务添加新扩控制器
        :return:
        """
        cmd = "api/v2/task/containerService/expand"
        record = common.get_rest_api_v2_record(self.context, cmd, "POST", param_dict)

        code = record.get("result", {}).get("code", "")
        task_id = record.get("data", {}).get("taskId", "")
        return bool(str(code) == "0"), task_id

    def is_new_controller_in_container(self, task_id):
        """
        检查新扩控制器是否扩入容器服务
        :param task_id: 任务ID
        :return:
        """
        cmd = "api/v2/task/{}".format(task_id)
        record = common.get_rest_api_v2_record(self.context, cmd, "GET")
        code = record.get("result", {}).get("code", "")
        status = record.get("data", {}).get("taskStatus", "")

        return bool(str(code) == "0"), status

    def get_expand_container_param_dict(self):
        """
        获取扩容容器服务的参数字典
        :return:
        """
        is_high_end = baseUtil.isDoradoV6HighEnd(self.product_model)
        required_ctrl_num = common.get_required_ctrl_num(self.context, self.product_model)

        if is_high_end:
            param_dict = high_end_expand_container_param_dict.get(
                str(required_ctrl_num))
        else:
            # 中低端2扩4
            param_dict = mid_end_expand_container_param_dict.get(
                str(required_ctrl_num))
        # E8000通过接口查询参数字典：show container_service general
        if baseUtil.is_ocean_protect_support_switch(self.product_model):
            self.update_param_dict_for_ocean_protect_support_switch(param_dict)
        self.logger.logInfo("param_dict:{}".format(param_dict))
        # A8000需要下发压力参数和重删参数
        if self.product_model == "OceanProtect A8000":
            param_dict["enableDedup"] = "false"
            param_dict["enableCompression"] = "false"
        return param_dict

    def update_param_dict_for_ocean_protect_support_switch(self, param_dict):
        """
        刷新支持交换机组网二级存储的参数字典
        @param param_dict: 参数字典
        """
        cmd = "/api/v2/task/containerService"
        response_info = common.get_rest_api_v2_record(self.context, cmd, "GET")
        undeploy_node_list = response_info.get("data", {}).get("UndeployedNodeList")
        if undeploy_node_list:
            param_dict["nodeList"] = undeploy_node_list

    def set_fail_info(self, task_id):
        """
        设置失败信息
        :return:
        """
        param_dict = {"range": quote('{"offset":0,"limit":100}'),
                      "sort": quote('{"id":"desc"}')}

        cmd = "/api/v2/task/{}/steps".format(task_id)
        response_info = common.get_rest_api_v2_record(
            self.context, cmd, "GET", param_dict)

        record = response_info.get("data", [])
        if not record:
            raise Exception("error response.")
        err_code = record[0].get("description", {}).get("code", "")
        if not err_code:
            raise Exception("error response.")
        self.resultDict["flag"] = False
        self.resultDict["errMsg"], self.resultDict["suggestion"] = \
            contextUtil.getItem(
            self.context, 'errCodeMgrObj', {}).get(int(err_code))
        if not self.resultDict["errMsg"] or not self.resultDict["suggestion"]:
            self.process_err_info("failed.expand.container.service")
        contextUtil.handleFailure(self.context, self.resultDict)
