# -*- coding: UTF-8 -*-

from cbb.business.operate.expansion import common
from cbb.frame.context import contextUtil
from cbb.frame.cli import cliUtil


class InnerHyperMetroPortsCheckItem:

    def __init__(self, context):
        self.context = context
        self.logger = common.getLogger(context.get("logger"), __file__)
        self.cli = contextUtil.getCli(context)
        self.lang = contextUtil.getLang(context)
        self.result_dict = {"flag": True, "errMsg": "", "suggestion": ""}

    def execute(self):
        """
        检查内双活端口连接正确性
        :return:
        """
        try:
            # 设备是否有内双活license
            has_inner_hyper_metro_license = contextUtil.getItem(
                self.context, "hasInnerMetroLicense")

            # 硬盘框P2端口信息
            disk_enclosure_port_infos = self.get_disk_enclosure_port_infos()

            # 检查端口状态是否正确
            return self.check_port_status(
                has_inner_hyper_metro_license, disk_enclosure_port_infos)
        except Exception as ex:
            self.result_dict["errMsg"], self.result_dict["suggestion"] \
                = common.getMsg(self.lang, "system.abnormal")
            self.logger.logException(ex)
            return False

    def get_disk_enclosure_port_infos(self):
        """获取硬盘框的P2端口信息

        :return:
        """
        cmd = "show port general logic_type=Expansion_Port"
        flag, cli_ret, err_msg = cliUtil.execCmdInCliMode(
            self.cli, cmd, True, self.lang)
        if flag is not True:
            raise Exception("can not get port info")

        port_infos = cliUtil.getHorizontalCliRet(cli_ret)
        disk_enclosure_port_infos = []
        for port_info in port_infos:
            port_id = port_info.get("ID")
            if port_id.startswith("DAE") and port_id.endswith("P2"):
                disk_enclosure_port_infos.append(port_info)

        return disk_enclosure_port_infos

    def check_port_status(
            self, has_inner_hyper_metro_license, disk_enclosure_port_infos):
        """判断端口状态是否符合要求
        如果有内双license，且所有IP框P2端口状态为LinkUp，则检查通过；
        如果没有内双活license，且所有IP框P2端口状态不为LinkUp，则检查通过；

        :param has_inner_hyper_metro_license: 是否有内双活license
        :param disk_enclosure_port_infos: 端口信息
        :return:
        """
        # 有内双活license时 硬盘框必须为智能框 智能框必有P2端口
        if has_inner_hyper_metro_license:
            return self.is_all_ports_linkup(disk_enclosure_port_infos)
        # 没有内双活license时 P2端口全不为LinkUp
        else:
            return self.is_none_port_linkup(disk_enclosure_port_infos)

    def is_all_ports_linkup(self, disk_enclosure_port_infos):
        """所有端口都是linkup状态

        :param disk_enclosure_port_infos: 端口信息
        :return:
        """
        if not disk_enclosure_port_infos:
            self.result_dict["errMsg"], self.result_dict["suggestion"] \
                = common.getMsg(
                self.lang, "precheck.inner.hyper.metro.port.error")
            return False
        error_ports = []
        for port_info in disk_enclosure_port_infos:
            port_id = port_info.get("ID")
            if port_info.get("Running Status").lower() != "link up":
                error_ports.append(port_id)

        if not error_ports:
            return True
        else:
            self.result_dict["errMsg"], self.result_dict["suggestion"] \
                = common.getMsg(
                self.lang,
                "precheck.inner.hyper.metro.port.error",
                ",".join(error_ports))
            return False

    def is_none_port_linkup(self, disk_enclosure_port_infos):
        """没有一个端口是linkup状态

        :param disk_enclosure_port_infos: 端口信息
        :return:
        """
        error_ports = []
        for port_info in disk_enclosure_port_infos:
            port_id = port_info.get("ID")
            if port_info.get("Running Status").lower() == "link up":
                error_ports.append(port_id)

        if not error_ports:
            return True
        else:
            self.result_dict["errMsg"], self.result_dict["suggestion"] \
                = common.getMsg(
                self.lang,
                "precheck.not.inner.hyper.metro.port.error",
                ",".join(error_ports))
            return False

    def get_result_dict(self):
        """获取结果字典

        :return:
        """
        return self.result_dict
