# -*- coding: UTF-8 -*-

from cbb.business.operate.expansion import common
from cbb.frame.base.config import TrVersion
from cbb.frame.context import contextUtil
from cbb.frame.tlv import adTlvUtil
from cbb.frame.tlv import tlvData

SRC_ENC_TYPE_KEY = "Src Enclosure Type"  # 始端框类型
SRC_ENG_ID_KEY = "Src Engine Id"  # 始端引擎ID
SRC_PORT_KEY = "Src Port"  # 始端端口号
DST_ENC_TYPE_KEY = "Dst Enclosure Type"  # 末端框类型
DST_ENG_ID_KEY = "Dst Engine Id"  # 末端引擎ID
DST_PORT_KEY = "Dst Port"  # 末端端口号
ENG_NUM_ONE = 1  # 引擎个数1
NET_WORK_MODE_SWITCH = 1  # 交换机组网


class MgmtCheckItem:

    def __init__(self, context):
        self.context = context
        self.logger = common.getLogger(context.get("logger"), __file__)
        self.tlv = contextUtil.getTlv(context)
        self.lang = contextUtil.getLang(context)
        self.result_dict = {"flag": True, "errMsg": "", "suggestion": ""}
        # 保存新引擎信息字典，key: 框SN, value:引擎ID
        self.bay_id_info = {}
        # 记录已获取结果的端口对（（始端端口，始端引擎），（末端端口，末端引擎））为一对
        self.pair_port_set = set()
        # 原集群节点信息列表
        self.old_boards_list = []
        self.product_version = contextUtil.getItem(context, "productVersion")
        self.is_dorado = contextUtil.getItem(context, "isDorado", False)
        self.boards_list = contextUtil.getItem(context, "boardsList")
        self.new_boards_list = contextUtil.getItem(context, "newBoardsList")
        self.new_ctrl_num = contextUtil.getItem(context, "newConfigCtrlNum")
        self.old_ctrl_num = contextUtil.getItem(context, "ctrlNum")
        self.height = contextUtil.getItem(context, "ctrlHeight")
        self.net_work_mode = common.getClustType(
            contextUtil.getItem(context, "newConfigClustType", ""))
        # 所有扩容流程是否完成
        self.all_flow_finish = contextUtil.getItem(context,
                                                   "all_flow_finish", False)
        self.init_old_boards()
        self.init_new_bay_info()

    def init_old_boards(self):
        """初始化原集群节点信息

        :return:
        """
        # 集群SN
        origin_sys_sn = contextUtil.getDevObj(self.context).get("sn")
        # 扩控后，待扩控已加入集群，需要重新查询节点，并更新ad通信路由
        if self.all_flow_finish:
            adTlvUtil.notifyAD(self.tlv)
            all_boards_list = adTlvUtil.getBoards(self.tlv)
            # 刷新每个节点的路由信息表
            adTlvUtil.refreshRouteInfo(self.tlv, self.logger)
        # 扩容前
        else:
            all_boards_list = contextUtil.getItem(self.context, "boardsList")

        self.old_boards_list = \
            [board for board in all_boards_list
             if board.get("enclosureSN") == origin_sys_sn]
        self.logger.logInfo("boards list is:{}".format(self.old_boards_list))

    def init_new_bay_info(self):
        """初始化新框信息

        :return:
        """
        board_sn_set = set()
        sorted_bord_list = sorted(self.new_boards_list,
                                  key=lambda e: e.get('enclosureSN'))
        bay_id = self.old_ctrl_num / common.getCtrlNumPerEnc(self.height) - 1
        for board in sorted_bord_list:
            if board.get("enclosureSN") in board_sn_set:
                continue
            bay_id = bay_id + 1
            self.bay_id_info[board.get("enclosureSN")] = str(bay_id)
            board_sn_set.add(board.get("enclosureSN"))
        self.logger.logInfo('bay_id_info: %s' % self.bay_id_info)

    def execute(self):
        """执行扩控线缆检测

        :return:
        """
        self.logger.logInfo("start precheck mgmt cable.")
        try:
            if self.check_version_support():
                self.check_cable_connect()
            return self.result_dict
        except Exception as e:
            self.logger.logException(e)
            self.result_dict["flag"] = False
            self.result_dict["errMsg"], self.result_dict[
                "suggestion"] = common.getMsg(self.lang,
                                              "precheck.mgmt.cable.error")
            return self.result_dict

    def check_version_support(self):
        """ 检查版本是否支持相关命令,TR6开始支持线缆检测
        :param :
        :return:
        """
        is_dorado = contextUtil.getItem(self.context, "isDorado")
        # 原集群完整版本号
        full_pdt_ver = contextUtil.getItem(self.context, "fullProductVersion")
        if is_dorado:
            if full_pdt_ver == TrVersion.V6_TR5:
                return False
        else:
            if full_pdt_ver < TrVersion.V5_TR6:
                return False
        return True

    def check_result(self, records):
        """判断检测结果，返回连接错误端口信息

        :param records: 检测结果记录
        :return:
        """
        error_cable_info_list = []
        for record in records:
            cable_test_result = adTlvUtil.getRecordValue(
                record, tlvData.SCALE_OUT["CABLE_TEST_RESULT"])
            if str(cable_test_result) == \
                    tlvData.CABLE_TEST_RESULT_FLAG["PASS"]:
                continue

            if str(cable_test_result) == \
                    tlvData.CABLE_TEST_RESULT_FLAG["NOTPASS"]:
                self.result_dict["flag"] = False
                check_result = adTlvUtil.getRecordValue(
                    record, tlvData.SCALE_OUT["CABLE_INFO"]).get(
                    "check result")
                error_cable_info_list.extend(self.deal_single_result(
                    check_result))
        return error_cable_info_list

    def deal_single_result(self, check_result):
        """处理单个检测结果，去重保存

        :param check_result:
        :return:
        """
        error_cable_info_list = []
        for result in check_result:
            src_port = result.get(SRC_PORT_KEY, "")
            src_eng = result.get(SRC_ENG_ID_KEY, "")
            dst_port = result.get(DST_PORT_KEY, "")
            dst_eng = result.get(DST_ENG_ID_KEY, "")
            if ((src_port, src_eng),
                (dst_port, dst_eng)) in self.pair_port_set \
                    or ((dst_port, dst_eng),
                        (src_port, src_eng)) in self.pair_port_set:
                continue
            # 存放端口对，用于去重
            self.pair_port_set.add(((src_port, src_eng),
                                    (dst_port, dst_eng)))
            error_cable_info_list.append(result)
        return error_cable_info_list

    def get_dev_sn(self, eng_id):
        """根据错误消息的引擎ID获取设备的SN，原集群的SN都一样

        :param eng_id: 引擎ID
        :return:
        """
        for sn, eng in self.bay_id_info.items():
            if str(eng) == str(eng_id):
                return sn

        return contextUtil.getDevObj(self.context).get("sn")

    def get_msg(self, error_cable_info_list):
        """获取错误提示及修复建议

        :param error_cable_info_list: 连接错误的线缆信息
        :return:
        """
        suggestion_list = []
        for cable_info in error_cable_info_list:
            src_enc_type = cable_info.get(SRC_ENC_TYPE_KEY)
            src_eng_id = cable_info.get(SRC_ENG_ID_KEY)
            src_port = cable_info.get(SRC_PORT_KEY)

            if int(src_enc_type) in (tlvData.ENCLOSURE_TYPE_E["DSW"],
                                     tlvData.ENCLOSURE_TYPE_E["MSW"]):
                _, src_port_info = common.getMsg(self.lang,
                                                 "precheck.mgmt.cable.switch",
                                                 suggestionArgs=(src_port,
                                                                 src_eng_id))
            else:
                dev_sn = self.get_dev_sn(src_eng_id)
                _, src_port_info = common.getMsg(
                    self.lang, "precheck.mgmt.cable.ctrl.enclosure",
                    suggestionArgs=(src_port, dev_sn))

            dst_enc_type = cable_info.get(DST_ENC_TYPE_KEY)
            dst_eng_id = cable_info.get(DST_ENG_ID_KEY)
            dst_port = cable_info.get(DST_PORT_KEY)
            if int(dst_enc_type) in (tlvData.ENCLOSURE_TYPE_E["DSW"],
                                     tlvData.ENCLOSURE_TYPE_E["MSW"]):
                _, dst_port_info = common.getMsg(self.lang,
                                                 "precheck.mgmt.cable.switch",
                                                 suggestionArgs=(dst_port,
                                                                 dst_eng_id))
            else:
                dev_sn = self.get_dev_sn(dst_eng_id)
                _, dst_port_info = common.getMsg(
                    self.lang, "precheck.mgmt.cable.ctrl.enclosure",
                    suggestionArgs=(dst_port, dev_sn))
            # 如果末端端口为空，则表示始端端口不需要连接
            if not dst_port or dst_port == "--":
                _, cable_sug = common.getMsg(
                    self.lang, "precheck.mgmt.cable.port.no.connect",
                    suggestionArgs=src_port_info)
            else:
                _, cable_sug = common.getMsg(self.lang,
                                             "precheck.mgmt.cable.port",
                                             suggestionArgs=(src_port_info,
                                                             dst_port_info))
            suggestion_list.append(cable_sug)
        suggestion = "".join(suggestion_list)
        return common.getMsg(self.lang,
                             "precheck.mgmt.cable.fail",
                             suggestionArgs=suggestion)

    def check_cable_connect(self):
        """调用接口检测线缆连接正确性

        :return:
        """
        # 连接错误的线缆信息列表
        error_cable_info_list = []
        # 扩容前检查，交换机组网场景，需要检查新扩控节点。
        if not self.all_flow_finish:
            for board in self.new_boards_list:
                enclosure_sn = board["enclosureSN"]
                bay_id = self.bay_id_info.get(enclosure_sn)
                if bay_id is None:
                    self.logger.logInfo("cannot get bay id[deviceSN:%s, "
                                        "ctrl:%s]." % (enclosure_sn,
                                                       board["controllerID"]))
                    self.result_dict["flag"] = False
                    self.result_dict["errMsg"], self.result_dict[
                        "suggestion"] = common.getMsg(
                        self.lang, "precheck.mgmt.cable.error")
                    return

                records = adTlvUtil.testMgmtCable(self.tlv, board, bay_id,
                                                  self.old_ctrl_num,
                                                  self.new_ctrl_num,
                                                  self.net_work_mode,
                                                  is_dorado=self.is_dorado)
                self.logger.logInfo("new board records:%s" % records)

                error_cable_info_list.extend(self.check_result(records))

                self.logger.logInfo(
                    "check mgmt cable over[newCtrl]:%s" % board)

        # 检测原集群
        ctrl_config_num = common.getCtrlNumPerEnc(self.height)
        for board in self.old_boards_list:
            node_id = board["controllerID"]
            bay_id = node_id / ctrl_config_num
            records = adTlvUtil.testMgmtCable(self.tlv, board, bay_id,
                                              self.old_ctrl_num,
                                              self.new_ctrl_num,
                                              self.net_work_mode,
                                              is_dorado=self.is_dorado)
            self.logger.logInfo("old board records:%s" % records)
            error_cable_info_list.extend(self.check_result(records))
            self.logger.logInfo("check mgmt cable over[oldCtrl]:%s" % board)

        if not self.result_dict.get("flag"):
            self.result_dict["errMsg"], self.result_dict[
                "suggestion"] = self.get_msg(error_cable_info_list)
        return

    def is_failed(self):
        """判断结果是否为失败，失败返回True，成功False

        :return:
        """
        if not self.result_dict.get("flag"):
            return True
        return False

    def get_result_dict(self):
        """获取结果字典

        :return:
        """
        return self.result_dict
