# -*- coding: UTF-8 -*-
from cbb.business.operate.expansion import common
from cbb.business.operate.expansion import errorCodeMgr
from cbb.common.conf.productConfig import ENC_HEIGHT_TO_CTRL_NUM
from cbb.frame.base import baseUtil
from cbb.frame.base import config
from cbb.frame.cli import cliUtil
from cbb.frame.context import contextUtil
from cbb.frame.rest import restUtil
from cbb.business.operate.expansion.scm.check_expand_scm \
    import check_need_expand_scm


def execute(context):
    """扩容控制器初始化

    :param context:
    :return:
    """

    logger = common.getLogger(context.get("logger"), __file__)
    try:
        resultDict = {"flag": True, "errMsg": "", "suggestion": ""}
        lang = contextUtil.getLang(context)
        devObj = contextUtil.getDevObj(context)
        productModel = devObj.get("type")

        # 针对扩容后再次扩容场景,清除新集群控制器数量和组网类型标记
        contextUtil.removeItem(context, "newConfigClustType")
        contextUtil.removeItem(context, "newConfigCtrlNum")

        # 检查工具与设备的连接
        isPass, errMsg = contextUtil.checkConnection(context)
        if not isPass:
            resultDict["flag"] = False
            resultDict["errMsg"], resultDict["suggestion"] = errMsg, ""
            contextUtil.handleFailure(context, resultDict)
            return

        tlv = contextUtil.getTlv(context)
        if tlv is None:
            logger.logNoPass("create tlv connection failure")
            resultDict["flag"] = False
            resultDict["errMsg"], resultDict["suggestion"] = common.getMsg(
                lang, "dev.conn.failure")
            contextUtil.handleFailure(context, resultDict)
            return

        cli = contextUtil.getCli(context)
        if cli is None:
            logger.logNoPass("create cli connection failure")
            resultDict["flag"] = False
            resultDict["errMsg"], resultDict["suggestion"] = common.getMsg(
                lang, "dev.conn.failure")
            contextUtil.handleFailure(context, resultDict)
            return

        rest = contextUtil.getRest(context)
        if rest is None:
            logger.logNoPass("create rest connection failure")
            resultDict["flag"] = False
            resultDict["errMsg"], resultDict["suggestion"] = \
                common.getMsg(lang, "dev.conn.failure")
            contextUtil.handleFailure(context, resultDict)
            return

        # 新融合不支持带业务扩容，6.1.3RC2，存在存储池则不支持扩控
        flag, err_msg = is_support_new_oceanstor_exp_ctrl(
            context, productModel, lang)
        if not flag:
            contextUtil.handleFailure(
                context,
                {"flag": False, "errMsg": err_msg[0], "suggestion": err_msg[1]})
            return
        # V6高端检查是否存在内双活
        hasInnerMetroLicense = False
        if baseUtil.isDoradoV6HighEnd(productModel):
            hasInnerMetroLicense, _ = restUtil.CommonRest.hasInnerLicense(rest)
        contextUtil.setItem(context, "hasInnerMetroLicense",
                            hasInnerMetroLicense)

        common.setSpecialMsgInfo(cli, lang, logger)

        common.getClustInfo(context, tlv)

        ctrlEnclosureHeight = contextUtil.getItem(context, "ctrlHeight")
        controllerNum = contextUtil.getItem(context, "ctrlNum")
        bayConfigCtrlNum = contextUtil.getItem(context, "configCtrlNum")
        expansionSpecDict = contextUtil.getItem(context, "expansionSpecDict")

        # 判断当前集群控制器数量是否与配置文件中控制器数量一致
        if not common.checkBayCofigConsistent(controllerNum, bayConfigCtrlNum,
                                              tlv):
            logger.logNoPass("Number of controllers is inconsistent")
            resultDict["flag"] = False
            resultDict["errMsg"], resultDict["suggestion"] = common.getMsg(
                lang, "controllers.number.inconsistent")
            contextUtil.handleFailure(context, resultDict)
            return

        # 判断当前组网方式是否支持扩容
        net_work_type = contextUtil.getItem(context, "configClustType")
        if not expansionSpecDict or is_direct_network_and_reach_max_spec(
                str(net_work_type), str(ctrlEnclosureHeight), str(controllerNum)):
            logger.logNoPass("Do not support expansion")
            resultDict["flag"] = False
            resultDict["errMsg"], resultDict["suggestion"] = common.getMsg(
                lang, "donot.support.expansion")
            contextUtil.handleFailure(context, resultDict)
            return

        originalCtrlEncInfoDict = restUtil.CommonRest.getCtrlEncInfo(tlv,
                                                                     logger)
        logger.logInfo('Original controller enclosure info dict:{}'.format(
            originalCtrlEncInfoDict))
        contextUtil.setItem(context, 'originalCtrlEncInfoDict',
                            originalCtrlEncInfoDict)

        # 弹出扩容场景选择窗口，设置新集群控制器数量和组网类型
        common.popupSelectExpansionDialog(context)

        newConfigClustType = contextUtil.getItem(context, "newConfigClustType")
        newConfigCtrlNum = contextUtil.getItem(context, "newConfigCtrlNum")

        rootPath = contextUtil.getImportRootDir(context)
        errMsgMrgObj = errorCodeMgr.ErrorCodeMgr(rootPath, lang, logger)
        contextUtil.setItem(context, 'errCodeMgrObj', errMsgMrgObj)

        if not newConfigClustType or not newConfigCtrlNum:
            resultDict["flag"] = False
            resultDict["errMsg"], resultDict["suggestion"] = common.getMsg(
                lang, "donot.select.expansion")
            contextUtil.handleFailure(context, resultDict)
            return

        # 设置扩容规格
        contextUtil.setItem(context, "expansionSpec",
                            expansionSpecDict.get(newConfigClustType, None))

        # 根据扩容场景,设置扩容流程
        if baseUtil.isV5V6HighEnd(productModel):
            setStepFlowForHigh(context, controllerNum, newConfigCtrlNum,
                               productModel, hasInnerMetroLicense)
        else:
            setStepFlow(context, productModel)

        logger.logInfo('Expansion step list:%s' % context["runStepIdList"])
        contextUtil.handleSuccess(context)
        return

    except Exception as exception:
        contextUtil.handleException(context, exception)
        logger.logException(exception)
        return


def is_support_new_oceanstor_exp_ctrl(context, product_model, lang):
    """
    判断是否支持新融合扩控
    :param context: 上下文
    :param product_model: 设备型号
    :param lang: 语言环境
    :return:
    """
    # 不是新融合，通过
    if product_model not in config.NEW_DORADO:
        return True, ""
    # 如果版本不是6.1.3RC2,通过
    if contextUtil.getItem(context, "productVersion") != "6.1.3RC2":
        return True, ""
    # 判断是否存在存储池
    if not get_pool_info(context):
        return True, ""

    return False, common.getMsg(lang,
                                "new.oceanstor.not.support.exp.ctrl.613RC2")


def get_pool_info(context):
    """
    获取存储池信息
    :param context: 上下文
    :return:
    """
    lang = contextUtil.getLang(context)
    cli = contextUtil.getCli(context)
    cmd = "show storage_pool general"
    ret = cliUtil.excuteCmdInCliMode(cli, cmd, True, lang)
    if ret[0] is not True:
        raise Exception("Failed to get storage pool info.")
    return cliUtil.getHorizontalCliRet(ret[1])


def is_direct_network_and_reach_max_spec(net_work_type, enclosure_height, controller_num):
    if net_work_type != "0":
        return False
    if enclosure_height in ["4", "6"] and controller_num == "8":
        return True
    if enclosure_height in ["2", "3"] and controller_num == "4":
        return True
    return False


def setStepFlow(context, productModel):
    """根据扩容场景,设置扩容流程（中低端）

    :param context: Jython context.
    :return:
    """
    # 根据step ID设置界面向导中的步骤
    runStepIdList = []
    # 扩容前注意事项
    newConfigClustType = contextUtil.getItem(context, "newConfigClustType")
    if newConfigClustType == "direct":
        runStepIdList.append("noticeDirect")
    else:
        runStepIdList.append("noticeSwitch")

    if common.isDorado(productModel) or \
            baseUtil.is_ocean_protect_mid(productModel):
        pre_check = "preCheckScaleOut"
    else:
        pre_check = "preCheckScaleOutV5"

    # A8000 扩容后新增两个步骤
    if productModel == "OceanProtect A8000":
        pre_check = "preCheckScaleOutForA8000"
        common_check = "commonForA8000"
        post_check = "postCheckScaleOutCommonForA8000"
    # 二级存储X系列，1.5.0及以后去掉检查项：扩充容器网段服务，1.5.0以前保持不变
    elif baseUtil.is_ocean_protect_x(productModel) and contextUtil.getItem(context, "productVersion") <= "1.3.0":
        common_check = "common"
        post_check = "postCheckScaleOutCommonForX8000"
    elif productModel in config.OCEAN_STOR_MICRO_DEVS:
        pre_check = "preCheckScaleOutMicro"
        common_check = "commonForMicro"
        post_check = "postCheckMicroCommon"
    else:
        common_check = "common"
        post_check = "postCheckScaleOutCommon"



    runStepIdList.extend([
        common_check,
        "replacingPlaneForMidAndLowEnd",
        "scanEnclosuresScaleOut",
        pre_check,
        "oneCmdExpandCtrlCommon",
        post_check,
        "finished"
    ])
    add_expand_container_if_need(productModel, runStepIdList)
    add_expand_scm_step_if_need(context, runStepIdList)

    context["runStepIdList"] = ",".join(runStepIdList)
    return


def add_expand_container_if_need(product_model, run_steps):
    """
    是否需要加入扩容容器步骤
    :param product_model: 产品型号
    :param run_steps: 所有步骤
    :return:
    """
    if product_model != "OceanProtect A8000":
        return
    run_steps.remove("finished")
    run_steps.remove("postCheckScaleOutCommonForA8000")
    run_steps.append("expandContainerForA8000")
    run_steps.append("postCheckScaleOutCommonForA8000")
    run_steps.append("finished")


def add_expand_scm_step_if_need(context, run_steps):
    """
    是否需要扩容scm步骤
    :param context: 上下文
    :param product_mode: 产品型号
    :param run_steps: 所有步骤
    :return:
    """
    if not check_need_expand_scm(context):
        return
    if "finished" in run_steps:
        run_steps.remove("finished")
        run_steps.append("selectExpandScm")
        run_steps.append("doExpandScm")
        run_steps.append("finished")

    contextUtil.setItem(context, "need_expand_scm", True)


def setStepFlowForHigh(context, controllerNum, newConfigCtrlNum, productModel,
                       hasInnerMetroLicense):
    """ 根据扩容场景,设置扩容流程。（针对高端直连组网）

    :param context: Jython context.
    :param controllerNum: 原集群控制器数量。
    :param newConfigCtrlNum: 扩容后控制器数量
    :param productModel: 产品型号
    :param hasInnerMetroLicense: 是否有内双活特性
    :return:
    """
    ctrl_enclosure_height = contextUtil.getItem(context, "ctrlHeight")
    peer_ctrl_num = ENC_HEIGHT_TO_CTRL_NUM.get(
        ctrl_enclosure_height, 4)

    run_step_id_list = []
    # 扩容前注意事项
    if contextUtil.getItem(context, "newConfigClustType") == "direct":
        notice = "noticeDirect"
    else:
        notice = "noticeSwitch"

    # Dorado V6 高端流程
    if common.isDorado(productModel) or \
            baseUtil.is_ocean_protect_high(productModel):
        if hasInnerMetroLicense:
            finalPostCheckScaleOut = "postCheckScaleOutForV6HighEnd"
        else:
            finalPostCheckScaleOut = "postCheckScaleOutCommon"
        # 不新增引擎场景
        if controllerNum % peer_ctrl_num != 0 \
                and newConfigCtrlNum - controllerNum < peer_ctrl_num:
            # 2扩4
            if controllerNum == 2:
                run_step_id_list = [
                    "noticeHybrid",
                    "common",
                    "oneCmdExpandCtrlCommon",
                    "postCheckNoneForV6HighEnd",
                    "finished"
                ]
            # 其他插入控制器场景，如6扩8,10扩12,14扩16
            else:
                run_step_id_list = [
                    "noticeHybrid",
                    "common",
                    "oneCmdExpandCtrlCommon",
                    "postCheckHybridForV6HighEnd",
                    "finished"
                ]
        # 先插入控制器，再新增引擎场景
        elif controllerNum % peer_ctrl_num != 0 \
                and newConfigCtrlNum - controllerNum >= peer_ctrl_num:
            contextUtil.setItem(context, "newCtrlNum", controllerNum + 2)
            contextUtil.setItem(context, "expand_ctrl_and_engine", True)
            if newConfigCtrlNum - controllerNum % peer_ctrl_num == 0:
                # 只有发货都是整框的场景才需要拆控
                run_step_id_list = [
                    notice,
                    "noticeSplitCtrl",
                    "common",
                    "oneCmdExpand2CTo4C",  # 一键式扩容
                    "postCheckNoneCommon",
                    "getCurCtrlsInfo",
                    "replacingPlaneForHighEnd",
                    "scanEnclosuresScaleOut",
                    "preCheckScaleOut",
                    "oneCmdExpandCtrlCommon",  # 一键式扩容（4->6)
                    "postCheckScaleOutForV6HighEnd",
                    "finished"
                ]
            else:
                run_step_id_list = [
                    notice,
                    "common",
                    "oneCmdExpand2CTo4C",  # 一键式扩容
                    "postCheckNoneCommon",
                    "getCurCtrlsInfo",
                    "replacingPlaneForHighEnd",
                    "scanEnclosuresScaleOut",
                    "preCheckScaleOut",
                    "oneCmdExpandCtrlCommon",  # 一键式扩容（4->6)
                    "postCheckScaleOutForV6HighEnd",
                    "finished"
                ]
        # 只新增引擎场景
        elif controllerNum % peer_ctrl_num == 0:
            run_step_id_list = [
                notice,
                "common",
                "replacingPlaneForHighEnd",
                "scanEnclosuresScaleOut",
                "preCheckScaleOut",
                "oneCmdExpandCtrlCommon",  # 一键式扩容（4->6 / 4->8)
                finalPostCheckScaleOut,
                "finished",
            ]

        add_expand_scm_step_if_need(context, run_step_id_list)

    # 融合存储V5 高端流程
    else:
        # 不新增引擎场景
        if controllerNum % peer_ctrl_num != 0 \
                and newConfigCtrlNum - controllerNum < peer_ctrl_num:
            # 2扩4
            if controllerNum == 2:
                run_step_id_list = [
                    "noticeHybrid",
                    "common",
                    "oneCmdExpandCtrlCommon",
                    "postCheckNoneCommon",
                    "finished"
                ]
            # 其他插入控制器场景，如6扩8,10扩12,14扩16
            else:
                run_step_id_list = [
                    "noticeHybrid",
                    "common",
                    "oneCmdExpandCtrlCommon",
                    "postCheckHybridCommon",
                    "finished"
                ]
        # 先插入控制器，再新增引擎场景
        elif controllerNum % peer_ctrl_num != 0 \
                and newConfigCtrlNum - controllerNum >= peer_ctrl_num:
            contextUtil.setItem(context, "newCtrlNum", controllerNum + 2)
            if newConfigCtrlNum - controllerNum % peer_ctrl_num == 0:
                # 只有发货都是整框的场景才需要拆控
                run_step_id_list = [
                    notice,
                    "noticeSplitCtrl",
                    "common",
                    "oneCmdExpand2CTo4C",  # 一键式扩容
                    "postCheckNoneCommon",
                    "getCurCtrlsInfo",
                    "replacingPlaneForHighEnd",
                    "scanEnclosuresScaleOut",
                    "preCheckScaleOutV5",
                    "oneCmdExpandCtrlCommon",  # 一键式扩容（4->6)
                    "postCheckScaleOutCommon",
                    "finished"
                ]
            else:
                run_step_id_list = [
                    notice,
                    "common",
                    "oneCmdExpand2CTo4C",  # 一键式扩容
                    "postCheckNoneCommon",
                    "getCurCtrlsInfo",
                    "replacingPlaneForHighEnd",
                    "scanEnclosuresScaleOut",
                    "preCheckScaleOutV5",
                    "oneCmdExpandCtrlCommon",  # 一键式扩容（4->6)
                    "postCheckScaleOutCommon",
                    "finished"
                ]
        # 只新增引擎场景
        elif controllerNum % peer_ctrl_num == 0:
            run_step_id_list = [
                notice,
                "common",
                "replacingPlaneForHighEnd",
                "scanEnclosuresScaleOut",
                "preCheckScaleOutV5",
                "oneCmdExpandCtrlCommon",
                "postCheckScaleOutCommon",
                "finished",
            ]

    context["runStepIdList"] = ",".join(run_step_id_list)
    return
