# -*- coding: UTF-8 -*-
from cbb.frame.base import product
from cbb.frame.context import contextUtil
from cbb.business.operate.expansion import common
from cbb.business.operate.expansion import FuncFactory
from cbb.frame.tlv import adTlvUtil, tlvUtil
import re

VER_6_0_0 = "6.0.0"
VER_6_0_1 = "6.0.1"

RISK_NEW_PRODUCT_VERSION_DICT = {
    "V500R007C60SPC300 Kunpeng": "V500R007C60SPH312",
    "6.0.0": "6.0.0.SPH11",
    "6.0.1": "6.0.1.SPH15"
}


@FuncFactory.handleFailureForRetry
def execute(context):
    """检查待扩容控制器版本

    :param context:
    :return:
    """
    logger = common.getLogger(context.get("logger"), __file__)
    try:
        resultDict = {"flag": True, "errMsg": "", "suggestion": ""}
        tlv = contextUtil.getTlv(context)
        lang = contextUtil.getLang(context)

        productVersion = contextUtil.getItem(context, "productVersion")
        newBoardsList = contextUtil.getItem(context, "newBoardsList")
        pdtVersion, patch_ver = "", ""
        board_info_dict = {}
        for board in newBoardsList:
            pdtVersion, patch_ver = adTlvUtil.getProductVersion(tlv, board)
            if not pdtVersion:
                continue
            sn = board["enclosureSN"]
            board_info_dict[sn] = {
                "ver": pdtVersion,
                "patch_ver": patch_ver,
            }
            board_info_dict[sn].update(board)
            # Dorado V6 数字版本和V7可以互扩
            if is_exp_digital_ver_and_v7_mix(productVersion, pdtVersion):
                continue
            # 版本一致性检查放宽到V版本，区分判断V6数字版本
            if (
                product.isDigitalVer(productVersion)
                and productVersion.split(".")[0] != pdtVersion.split(".")[0]
            ) or (
                not product.isDigitalVer(productVersion)
                and common.getVRCVersion(pdtVersion)[0:4]
                != common.getVRCVersion(productVersion)[0:4]
            ):
                logger.logNoPass(
                    "product version inconsistent" "({})".format(str(board))
                )
                resultDict["flag"] = False
                resultDict["errMsg"], resultDict["suggestion"] = common.getMsg(
                    lang, "pdt.version.inconsistent", sn
                )
                contextUtil.handleFailure(context, resultDict)
                return

        # 新增检查
        if has_fail_risk_check(board_info_dict, context):
            return

        # 保存新扩控的版本及补丁版本
        contextUtil.setItem(context, "new_pdt_ver", pdtVersion)
        contextUtil.setItem(context, "new_patch_ver", patch_ver)
        contextUtil.handleSuccess(context)
        logger.logPass()
        return

    except Exception as exception:
        contextUtil.handleException(context, exception)
        logger.logException(exception)
        return


def is_exp_digital_ver_and_v7_mix(product_version, pdt_version):
    """
    判斷是否为Dorado数字版本和V7混扩
    @param product_version: 原集群设备版本
    @param pdt_version: 待扩设备版本
    @return: True：是混扩，False：不是混扩
    """
    return (product.isDigitalVer(product_version) and not product.isDigitalVer(pdt_version) or product.isDigitalVer(
        pdt_version) and not product.isDigitalVer(product_version))


def has_fail_risk_check(board_info_dict, context):
    """
    检查新扩控和原控补丁版本是否一致
    检查标准：
        1.如果原集群或新扩集群版本不是6.0.0，则无风险；
        2.如果原集群和新扩集群版本是6.0.0，且补丁版一致，则无该风险，否则有风险。
    :param board_info_dict:
    :param context:
    :return: True:有风险
            False:无风险
    """
    tlv = contextUtil.getTlv(context)
    logger = common.getLogger(context.get("logger"), __file__)
    lang = contextUtil.getLang(context)

    origin_ver, origin_patch_ver = tlvUtil.getProductVersionAndPatchVersion(tlv)
    logger.logInfo("origin_ver:{}, origin_patch_ver:{}".format(
        origin_ver, origin_patch_ver))

    logger.logInfo("board_info_dict:{}".format(board_info_dict))
    for board_sn, info in board_info_dict.items():
        ver, patch_ver = info["ver"], info["patch_ver"]

        if (
            ver.startswith(VER_6_0_0)
            and origin_ver.startswith(VER_6_0_0)
            and (get_path_version_num(origin_patch_ver) !=
                 get_path_version_num(patch_ver) or
                 (get_path_version_num(patch_ver) < get_path_version_num(
                     RISK_NEW_PRODUCT_VERSION_DICT.get(VER_6_0_0))))
        ) or (
            ver.startswith(VER_6_0_1)
            and origin_ver.startswith(VER_6_0_1)
            and (get_path_version_num(origin_patch_ver) < 15
                 or get_path_version_num(origin_patch_ver) < get_path_version_num(patch_ver)
                 or get_path_version_num(patch_ver) < get_path_version_num(
                    RISK_NEW_PRODUCT_VERSION_DICT.get(VER_6_0_1)))
        ):
            logger.logInfo("ver:{}, patch_ver:{} not pass!".format(
                ver, patch_ver))
            err_msg, suggestion = common.getMsg(
                lang,
                "patch.version.inconsistent",
                errMsgArgs=(
                    "{}".format(origin_ver),
                    board_sn,
                    "{}".format(ver),
                ),
                msgDict=MESSAGES_DICT if ver.startswith(
                    VER_6_0_0) else MESSAGES_DICT_61,
            )
            contextUtil.handleFailure(
                context,
                {"flag": False, "errMsg": err_msg, "suggestion": suggestion},
            )
            return True
        if (ver.startswith(VER_6_0_0) or ver.startswith(VER_6_0_1)) and \
                origin_ver == ver:
            return False
        # 新增待扩控版本检查
        if not check_new_version(context, ver, patch_ver, board_sn):
            return True

    return False


def check_new_version(context, ver, patch_ver, board_sn):
    """
        检查新扩控补丁版本
        检查标准：
        check_fru_num_before_scale_out
        :param context: 上下文
        :param ver: 软件版本        如： "6.0.1.SPH2"   "V500R007C30 Kunpeng"
        :param patch_ver: 补丁版本  如： "SPH2"         "SPC300 SPH305"
                                  如： ""             ""
                                  如： "SPC100 SPH3"  "SPC300"
        :param board_sn: 节点SN号
        :return: True:无风险
                False:有风险
        """
    logger = common.getLogger(context.get("logger"), __file__)
    lang = contextUtil.getLang(context)

    if "Kunpeng" in ver:
        # Kunpeng版本需要拼接SPC版本
        spc_ver = patch_ver.split()[0] if "SPC" in patch_ver else ""
        base_version = ver.split()[0] + spc_ver + " Kunpeng"
        current_version_for_err_msg = base_version
    else:
        base_version = ver.split(".SPH")[0]
        current_version_for_err_msg = ver

    if base_version not in RISK_NEW_PRODUCT_VERSION_DICT:
        return True

    if get_path_version_num(patch_ver) < get_path_version_num(
            RISK_NEW_PRODUCT_VERSION_DICT.get(base_version)):
        logger.logInfo("ver:{}, patch_ver:{}".format(base_version, patch_ver))
        err_msg, suggestion = common.getMsg(
            lang, "new.patch.version.inconsistent",
            errMsgArgs=(board_sn, current_version_for_err_msg),
            suggestionArgs=RISK_NEW_PRODUCT_VERSION_DICT.get(base_version),
            msgDict=NEW_MESSAGES_DICT)

        contextUtil.handleFailure(
            context,
            {"flag": False, "errMsg": err_msg, "suggestion": suggestion},
        )
        return False
    return True


def get_path_version_num(path_version):
    """
    获取热补丁版本号int类型
    :param path_version: 补丁
    :return: int的版本号
    """
    pattern_hot_patch = re.compile(r"SPH(\d+)", flags=re.IGNORECASE)
    match_hot_path = pattern_hot_patch.search(path_version)
    if match_hot_path:
        path_version_num = int(match_hot_path.group(1))
    else:
        path_version_num = 0
    return path_version_num


MESSAGES_DICT = {
    "patch.version.inconsistent": {
        "errMsg_zh": u"原集群版本是%s，新扩集群(SN:%s)的版本是%s。",
        "errMsg_en": "The version of the original cluster is %s, and that of "
                     "the new (SN:%s) cluster is %s.",
        # noqa
        "suggestion_zh": u"新扩集群请安装6.0.0.SPH11及之后版本补丁,并且原集群补丁与新"
                         u"集群补丁一致。",
        "suggestion_en": "Please install patch 6.0.0.SPH11 or later for the new"
                         " cluster.And ensure that versions "
                         "of the existing controllers and the controllers "
                         "to be added are the same by installing patchs.",
        # noqa
    },
}
MESSAGES_DICT_61 = {
    "patch.version.inconsistent": {
        "errMsg_zh": u"原集群版本是%s，新扩集群(SN:%s)的版本是%s。",
        "errMsg_en": "The version of the original cluster is %s, and that of "
                     "the new (SN:%s) cluster is %s.",
        # noqa
        "suggestion_zh": u"原集群与新扩集群都安装6.0.1.SPH15及之后版本补丁，且原集群补"
                         u"丁版本大于等于新扩集群。",
        "suggestion_en": "please install the patch of 6.0.1.SPH15 and later for"
                         " the new cluster and the original cluster,"
        "And the original cluster patch version is greater than"
        "or equal to the new cluster expansion.",
        # noqa
    },
}

NEW_MESSAGES_DICT = {
    "new.patch.version.inconsistent": {
        "errMsg_zh": u"新扩集群(SN:%s)的版本是%s。",
        "errMsg_en": "The version of the new cluster(SN:%s) is %s.",

        "suggestion_zh": u"新扩集群请安装%s及之后版本补丁。",
        "suggestion_en": "please install patch %s or later for the new "
                         "cluster.",
    },
}
