# -*- coding: UTF-8 -*-

from cbb.business.operate.expansion import common
from cbb.business.operate.expansion import config
from cbb.frame.context import contextUtil
from cbb.frame.tlv import adTlvUtil
from cbb.frame.tlv import tlvData

#进度总剩余时间
LIMIT_TIME = 120
#进度刷新间隔
INTERVAL = 2

def execute(context):
    '''
    @summary: 检查端口连接正确性
    '''
    
    try:
        #进度条刷新
        common.threadUpProcess(context, LIMIT_TIME, INTERVAL)
        #进度开始
        common.inProcess(context)
        
        resultDict = {"flag":True, "errMsg":"", "suggestion":""}
        logger = common.getLogger(context.get("logger"), __file__)
        
        tlv = contextUtil.getTlv(context)
        lang = contextUtil.getLang(context)
        
        expansionSpec = contextUtil.getItem(context, "expansionSpec")
        originEnclosureSNs = contextUtil.getItem(context, "originEnclosureSNs")
        ctrlNum = contextUtil.getItem(context, "ctrlNum")
        newConfigCtrlNum = contextUtil.getItem(context, "newConfigCtrlNum")
        newConfigClustType = contextUtil.getItem(context, "newConfigClustType")
        
        boardsList = contextUtil.getItem(context, "boardsList")
        ctrlHeight = contextUtil.getItem(context, "ctrlHeight")
        
        #获取路由信息表
        logger.logInfo("start get routine info")
        routeInfoList = adTlvUtil.getRouteInfoList(tlv)
        logger.logInfo("routineInfoList:%s" % routeInfoList)
        
        #获取所有的ETH端口信息
        portInfoList = []
        for board in boardsList:
            logger.logInfo("get ports info:%s" % board)
            ports = adTlvUtil.getScaleOutPortsInfo(tlv, board, ctrlHeight, newConfigClustType)
            if len(ports) == 0:
                continue
            portInfoList.extend(ports)
        logger.logInfo("portsinfoList:%s" % portInfoList)
        
        #检查端口数量是否满足要求
        currentScaleOutPortsNum = len(portInfoList)
        requiredScaleOutPortsNum = common.getScaleOutPortsSpecNum(expansionSpec, newConfigClustType) * newConfigCtrlNum
        if currentScaleOutPortsNum != requiredScaleOutPortsNum:
            logger.logNoPass("The number of ports suitable for capacity expansion does not meet the requirement (Current number: %s; Required number: %s)." % 
                (currentScaleOutPortsNum, requiredScaleOutPortsNum))
            resultDict["flag"] = False
            resultDict["errMsg"],resultDict["suggestion"] = common.getMsg(lang, "ports.insufficeint", 
                (currentScaleOutPortsNum, requiredScaleOutPortsNum))
            contextUtil.handleFailure(context, resultDict)
            return
        
        #检查端口状态和速率，生成ETH端口信息字典，键为端口mac地址，值为端口信息
        portMacAddrDict = {}
        for portInfo in portInfoList:
            enclosureSN = portInfo["enclosureSN"]
            location = portInfo["location"]
            healthStatus = portInfo["healthStatus"]
            runningStatus = portInfo["runningStatus"]
            speed = portInfo["speed"]
            
            #检查端口健康状态
            if healthStatus != tlvData.HEALTH_STATUS_E["NORMAL"]:
                logger.logNoPass("The health status of port (enclosure SN: %s; ID: %s) is abnormal (Health Status:%s)." % 
                    (enclosureSN, location, healthStatus))
                resultDict["flag"] = False
                resultDict["errMsg"],resultDict["suggestion"] = common.getMsg(lang, "port.health.status.abnormal", 
                    (enclosureSN, location))
                contextUtil.handleFailure(context, resultDict)
                return
            
            #检查端口运行状态
            if runningStatus != tlvData.RUNNING_STATUS_E["LINK_UP"]:
                logger.logNoPass("The running status of port (enclosure SN: %s; ID: %s) is abnormal (Running Status:%s)." % 
                    (enclosureSN, location, runningStatus))
                resultDict["flag"] = False
                resultDict["errMsg"],resultDict["suggestion"] = common.getMsg(lang, "port.running.status.abnormal", 
                    (enclosureSN, location))
                contextUtil.handleFailure(context, resultDict)
                return
            
            #检查端口速率
            if speed != 10000:
                logger.logNoPass("The working rate of port (enclosure SN: %s; ID: %s) is not 10Gbps." % 
                    (enclosureSN, location))
                resultDict["flag"] = False
                resultDict["errMsg"],resultDict["suggestion"] = common.getMsg(lang, "port.working.rate.abnormal", 
                    (enclosureSN, location))
                contextUtil.handleFailure(context, resultDict)
                return
            
            portMacAddrDict[portInfo["macAddress"]] = portInfo
            
        #生成路由表
        routineTableList = []
        for routeInfo in routeInfoList:
            srcPortInfo = portMacAddrDict.get(routeInfo["srcPortMacAddr"], None)
            desPortInfo = portMacAddrDict.get(routeInfo["desPortMacAddr"], None)
            if srcPortInfo is None or desPortInfo is None:
                continue
            
            srcPortSlot = common.getPortSlot(srcPortInfo)
            desPortSlot = common.getPortSlot(desPortInfo)
            srcPortSn = srcPortInfo["enclosureSN"]
            desPortSn = desPortInfo["enclosureSN"]
            
            checkPass = True
            #根据组网规则，源端口与目的端口端口号须一致
            if srcPortSlot != desPortSlot:
                checkPass = False
            else:
                #直连组网场景特殊检查
                if newConfigClustType == common.CLUST_TYPE_DIRECT:
                    srcPortNodeName = srcPortInfo["nodeName"]
                    desPortNodeName = desPortInfo["nodeName"]
                    if (srcPortSlot == "P0" and desPortSlot == "P0") or \
                        (srcPortSlot == "P1" and desPortSlot == "P1"):
                        if srcPortSn == desPortSn or srcPortNodeName != desPortNodeName:
                            checkPass = False
                    elif (srcPortSlot == "P2" and desPortSlot == "P2") or \
                        (srcPortSlot == "P3" and desPortSlot == "P3"):
                        if srcPortSn == desPortSn or srcPortNodeName == desPortNodeName:
                            checkPass = False
            
            if not checkPass: 
                srcPortLocation = srcPortInfo["location"]
                desPortLocation = desPortInfo["location"]
                logger.logNoPass("The cable connections between port (enclosure SN: %s; ID: %s) and port (enclosure SN: %s; ID: %s) do not meet network requirements." % 
                    (srcPortSn, srcPortLocation, desPortSn, desPortLocation))
                resultDict["flag"] = False
                resultDict["errMsg"],resultDict["suggestion"] = common.getMsg(lang, "ports.routine.wrong", 
                    (srcPortSn, srcPortLocation, desPortSn, desPortLocation))
                contextUtil.handleFailure(context, resultDict)
                return
            
            routineTableList.append("%s<->%s" % (srcPortSlot, desPortSlot))
            logger.logInfo("routeInfo:%s" % routeInfo)
            
        if len(routineTableList) == 0:
            resultDict["flag"] = False
            resultDict["errMsg"],resultDict["suggestion"] = common.getMsg(lang, "ports.connect.none")
            contextUtil.handleFailure(context, resultDict)
            return
        logger.logInfo("right routineTableList:%s" % routineTableList)
        
        #检查组网连接是否正确
        netMode = config.NETWORK_SPEC_DICT.get(newConfigClustType)
        configSPec = netMode.get("%sC" % newConfigCtrlNum)
        for planeConfig in configSPec.keys():
            for planeSpec in configSPec.get(planeConfig):
                routine = planeSpec[0]
                spec = planeSpec[1]
                errPort = routine.split("<->")[0]
                currentRoutineCnt = routineTableList.count(routine)
                if currentRoutineCnt < spec:
                    logger.logInfo("routine (%s) of ports connected:%s" % (routine, currentRoutineCnt))
                    logger.logInfo("routine (%s) specification:%s" % (routine, spec))
                    logger.logNoPass("The cable connections between port (slot: %s) and port (slot: %s) do not meet network requirements." % 
                       (errPort, errPort))
                    resultDict["flag"] = False
                    resultDict["errMsg"],resultDict["suggestion"] = common.getMsg(lang, "ports.connect.wrong", 
                       (errPort, errPort))
                    contextUtil.handleFailure(context, resultDict)
                    return
        
        #生成扫描端口信息表
        portsScanList = []
        bayIdList = []
        enclosureSNbayIdDict = {}
        
        newBayIds = []
        for port in portInfoList:
            enclosureSN = port["enclosureSN"]
            portLocation = port["location"]
            
            bayId = port["bayid"]
            bayIds = []
            if not enclosureSNbayIdDict.has_key(enclosureSN):
                if enclosureSN not in originEnclosureSNs:
                    #6U只扩最后2控场景时（如6控扩8控），不需要重新生成柜ID
                    if ctrlHeight == 6 and (ctrlNum + 2) == newConfigCtrlNum and newConfigCtrlNum % 4 == 0:
                        bayId = bayIdList[-1]
                    else:
                        bayId = common.generateNewBayId(bayIdList)
                bayIds = [bayId]
            else:
                bayIds = enclosureSNbayIdDict.get(enclosureSN)
                bayId += bayIds[0]
                bayIds.append(bayId)
                
            bayIdList.append(bayId)
            enclosureSNbayIdDict[enclosureSN] = bayIds
            
            portDict = {}
            portDict["enclosureSN"] = enclosureSN
            portDict["bayid"] = bayId
            portDict["controllerName"] = ("%s%s") % (bayId, port["nodeName"])
            portDict["plane"] = common.getDswID(portLocation, newConfigClustType)
            portDict["portID"] = port["portID"]
            portDict["location"] = common.generateNewPortLocation(portLocation, bayId)
            portDict["macAddress"] = port["macAddress"]
            portsScanList.append(portDict.copy())
            
            if enclosureSN not in originEnclosureSNs:
                newBayIds.append(str(bayId))
        
        portsScanList.sort(lambda x,y:common.cmpPortDict(x,y))
        logger.logInfo("bayIdList:%s" % bayIdList)
        logger.logInfo("enclosureSNbayIdDict:%s" % enclosureSNbayIdDict)
        logger.logInfo("portsScanList:%s" % portsScanList)
        logger.logInfo("newBayIds:%s" % newBayIds)
        
        contextUtil.setItem(context, "routeInfoList", routeInfoList)
        contextUtil.setItem(context, "enclosureSNbayIdDict", enclosureSNbayIdDict)
        contextUtil.setItem(context, "portsScanList", portsScanList)
        contextUtil.setItem(context, "newBayIds", newBayIds)
        
        switchOffBoardsList = contextUtil.getItem(context, "switchOffBoardsList")
        #关闭框的定位灯
        if switchOffBoardsList is not None:
            for board in switchOffBoardsList:
                setEnclosureSwitchOffRet = adTlvUtil.setEnclosureSwitchOff(tlv, board)
                logger.logInfo("board %s setEnclosureSwitchOff result:%s" % (board, setEnclosureSwitchOffRet))
        contextUtil.setItem(context, "switchOffBoardsList", [])
        
        contextUtil.handleSuccess(context)
        logger.logPass()
        return

    except Exception as exception:
        contextUtil.handleException(context, exception)
        logger.logException(exception)
        return
    finally:
        #进度条刷为完成状态
        common.finishProcess(context)
