# -*- coding: UTF-8 -*-

from cbb.business.operate.expansion import common
from cbb.frame.context import contextUtil
from cbb.frame.base import baseUtil
from cbb.common.query.hardware.switch import Switch
from cbb.business.product.product_selector import get_product_adapter
from cbb.business.collect.products_adapter import compare_version

# 交换机的型号、版本、补丁要求
DORADO_SWITCH_REQUIREMENT = {
    "CE6865": ("V200R005C10SPC800", "V200R005SPH017")
}
CONVERGED_HIGH_END_SWITCH_REQUIREMENT = {
    "CE8850": ("V200R005C10SPC800", "V200R005SPH017"),
    "CE8851": ("V300R020C10SPC200", ""),
    "CE8850-SAN": ("V300R020C10SPC200", "")
}
CONVERGED_LOW_END_SWITCH_REQUIREMENT = {
    "CE6865": ("V200R005C10SPC800", "V200R005SPH017"),
    "CE6866": ("V200R005C10SPC800", "V200R005SPH017"),
}


def execute(context):
    '''
    @summary: 配置交换机
    '''

    try:
        resultDict = {"flag":True, "errMsg":"", "suggestion":""}
        logger = common.getLogger(context.get("logger"), __file__)
        lang = contextUtil.getLang(context)

        connectorFactory = context.get("connectorFactory")
        try:
            curDataDict = common.jsonStr2Dict(context.get("curData"))
        # 界面输入非法字符时，处理json数据会失败
        except:
            resultDict["flag"] = False
            resultDict["errMsg"], resultDict["suggestion"] = common.getMsg(lang, "switch.illegal.input")
            contextUtil.handleFailure(context, resultDict)
            return

        switchInfoList = []

        # 校验界面输入项
        for i in range(0, 2):
            switch_info_dict = get_ui_input(context, curDataDict, i)
            if not switch_info_dict:
                return
            switchInfoList.append(switch_info_dict.copy())

        # 校验2个交换机的ip是否相同
        if switchInfoList[0].get("ipaddress") == \
                switchInfoList[1].get("ipaddress"):
            resultDict["flag"] = False
            resultDict["errMsg"], resultDict["suggestion"] = common.getMsg(
                lang, "switch.ip.identical.error")
            contextUtil.handleFailure(context, resultDict)
            return

        check_flag, check_msg_list = check_config_switch(connectorFactory, context, lang, logger, switchInfoList)

        if not check_flag:
            logger.logInfo("config switch failure")
            resultDict["flag"] = False
            resultDict["errMsg"] = "\n\n".join(check_msg_list)
            contextUtil.handleFailure(context, resultDict)
            return

        logger.logInfo("popupConfigSwtichDialog end")
        contextUtil.handleSuccess(context)
        return

    except Exception as exception:
        contextUtil.handleException(context, exception)
        logger.logException(exception)
        return


def check_config_switch(connectorFactory, context, lang, logger, switchInfoList):
    check_flag = True
    check_msg_list = []
    switch_config_map = {0: False, 1: False}
    # 连接交换机，对交换机进行配置
    switch_model_list = []
    for i in range(0, 2):
        # 创建与交换机的连接
        connectSwitchRet = common.connectSwitch(connectorFactory, switchInfoList[i])
        logger.logInfo("connectSwitchRet:%s" % connectSwitchRet)
        if not connectSwitchRet["flag"]:
            check_flag = False
            errMsg, suggestion = common.getMsg(lang, "switch.connect.error", i)
            check_msg_list.append(errMsg)
            continue
        cli = connectSwitchRet["cli"]
        current_switch = Switch(cli, logger)

        try:
            # 检查交换机型号
            dev_obj = contextUtil.getDevObj(context)
            product_model = dev_obj.get("type")
            product_version = dev_obj.get("version")
            require_model = []
            requirement = get_requirement(product_model, require_model, product_version)
            switch_model = current_switch.get_model()
            switch_model_list.append(switch_model)
            if check_switch_model(i, switch_model_list):
                check_flag = False
                set_msg_if_model_not_same(check_msg_list, lang)

            if not requirement or not current_switch.check_model(requirement):
                check_flag = False
                errMsg, suggestion = common.getMsg(lang, "switch.model.check.error", (i, "/".join(require_model)))
                check_msg_list.append(errMsg)
                continue
            # 检查交换机版本和补丁
            if not current_switch.check_version_patch(requirement):
                check_flag = False
                set_msg_if_version_not_same(check_msg_list, i, lang, requirement, switch_model)
                continue

            # 对交换机进行配置
            configSwitchRet = common.configSwitch(cli, switch_model, i)
            if not configSwitchRet["flag"]:
                check_flag = False
                errMsg, suggestion = common.getMsg(lang, "switch.config.error", i)
                check_msg_list.append(errMsg)
                continue
            # 记录交换机配置成功
            switch_config_map[i] = True

        except Exception as exception:
            logger.logException(exception)
            check_flag = False
            errMsg, suggestion = common.getMsg(lang, "switch.connect.error", i)
            check_msg_list.append(errMsg)
    return check_flag, check_msg_list


def set_msg_if_version_not_same(check_msg_list, switch_id, lang, requirement, switch_model):
    version_requirement = requirement.get(switch_model)[0] \
        if switch_model in ("CE8851", "CE8850-SAN", "CE8855", "CE8850E") \
        else "+".join(requirement.get(switch_model))
    err_msg, suggestion = common.getMsg(lang, "switch.version.check.error", (switch_id, version_requirement))
    check_msg_list.append(err_msg)


def set_msg_if_model_not_same(check_msg_list, lang):
    err_msg, suggestion = common.getMsg(lang, "switch.model.not.same")
    check_msg_list.append(err_msg)


def check_switch_model(index, switch_model_list):
    return index == 1 and switch_model_list[0] != switch_model_list[1] and \
        ("CE8851" in switch_model_list or
         "CE8850-SAN" in switch_model_list or
         "CE8855" in switch_model_list or
         "CE8850E" in switch_model_list)


def get_ui_input(context, ui_input, switch_id):
    """从界面输入获取交换机配置信息

    :param context: 上下文
    :param ui_input: 界面输入数据
    :param switch_id: 交换机id
    :return:
    """
    try:
        ipaddress = ui_input['ipaddress%s' % switch_id]
        check_empty(context, ipaddress, "switch.ipaddress.empty", switch_id)

        username = ui_input['username%s' % switch_id]
        check_empty(context, username, "switch.username.error", switch_id)

        password = ui_input['password%s' % switch_id]
        check_empty(context, password, "switch.password.error", switch_id)

        port = ui_input['port%s' % switch_id]
        check_empty(context, port, "switch.port.empty", switch_id)

        check_illegal_ip(context, ipaddress, switch_id)

        check_illegal_port(context, port, switch_id)

        switch_info_dict = {
            "ipaddress": ipaddress,
            "username": username,
            "password": password,
            "port": port
        }

        return switch_info_dict
    except Exception as ex:
        logger = common.getLogger(context.get("logger"), __file__)
        logger.logException(ex)
        return {}


def check_empty(context, input_str, error_key, switch_id):
    """检查输入是否为空

    :param context: 上下文
    :param input_str: 输入字符串
    :param error_key: 错误消息key
    :param switch_id: 交换机id
    :return:
    """
    if len(input_str.strip()) == 0:
        handle_failure(context, error_key, switch_id)
        raise Exception(error_key)


def check_illegal_ip(context, input_str, switch_id):
    """检查ip是否合法

    :param context: 上下文
    :param input_str: 输入字符串
    :param switch_id: 交换机id
    :return:
    """
    if not common.isIllegalManIpAddr(input_str, None):
        handle_failure(context, "switch.ipaddress.error", switch_id)
        raise Exception("switch.ipaddress.error")


def check_illegal_port(context, input_str, switch_id):
    """检查ip是否合法

    :param context: 上下文
    :param input_str: 输入字符串
    :param switch_id: 交换机id
    :return:
    """
    if not common.isPureDigit(input_str):
        handle_failure(context, "switch.port.error", switch_id)
        raise Exception("switch.port.error")


def handle_failure(context, error_key, switch_id):
    """错误处理

    :param context: 上下文
    :param error_key: 错误消息key
    :param switch_id: 交换机id
    :return:
    """
    lang = contextUtil.getLang(context)
    result_dict = {}
    result_dict["flag"] = False
    result_dict["errMsg"], result_dict["suggestion"] = common.getMsg(
        lang, error_key, switch_id)
    contextUtil.handleFailure(context, result_dict)


def get_requirement(product_model, require_model, product_version):
    product_adapter = get_product_adapter()
    if product_adapter:
        require_model.append("CE8850-SAN")
        require_model.append("CE8850E")
        requirement = product_adapter.get_support_switch()
        # 6.1.5及以后支持CE8855交换机, E8000支持交换机组网
        if compare_version(product_version, "6.1.5") >= 0 or baseUtil.is_ocean_protect_support_switch(product_model):
            require_model.append("CE8855")
            requirement["CE8855"] = ("", "")
        return requirement
    if baseUtil.isDoradoDev(product_model):
        require_model.append("CE6865")
        return DORADO_SWITCH_REQUIREMENT
    elif baseUtil.is_v5_middle_new(product_model):
        require_model.append("CE6865")
        require_model.append("CE6866")
        return CONVERGED_LOW_END_SWITCH_REQUIREMENT
    elif baseUtil.is_v5_high_new(product_model):
        require_model.append("CE8850-SAN")
        require_model.append("CE8850")
        return CONVERGED_HIGH_END_SWITCH_REQUIREMENT
    else:
        return []
