#  coding=UTF-8
#  Copyright (c) Huawei Technologies Co., Ltd. 2019-2019. All rights reserved.

# -*- coding: UTF-8 -*-

from __future__ import with_statement

import os
import ast

from cbb.business.operate.expansion import common
from cbb.frame.context import contextUtil
from cbb.frame.rest import commonRestUtil
from cbb.frame.base import config

# 配置数据持久化目录
CFG_DATA_PERSIST_DIR = "\\temp\\expanEval"

# 目录分割
FILE_DEPSIGN = "\\"

CFG_DATA_PERSIST_FILENAME = "expansionEvalConfig.dat"
# 向上两级目录
DIR_RELATIVE_CMD = "..\.."

# 进度总剩余时间
LIMIT_TIME = 120
# 进度刷新间隔
INTERVAL = 2

# 扩空池时，需要将diskNum传入-2，产品特殊处理。
EMPTY_POOL_FLAG = -2


def execute(context):
    """检查空闲盘是否满足扩容配置

    :param context: 变量上下文
    :return:
    """

    logger = contextUtil.getLogger(context)
    lang = contextUtil.getLang(context)
    try:
        # 进度条刷新
        common.threadUpProcess(context, LIMIT_TIME, INTERVAL)

        result_dict = {"flag": True, "errMsg": "", "suggestion": ""}

        # 1.读取接口文件获取扩容配置
        cfg_data = get_config_data(logger)
        logger.info("expansion config data is %s" % cfg_data)
        if not cfg_data:
            result_dict["flag"] = False
            result_dict["errMsg"], result_dict["suggestion"] = \
                common.getMsg(lang, "expdd.check.config.data.nil.cfgData")
            contextUtil.handleFailure(context, result_dict)
            return

        # 2.通过rest接口获取空闲盘
        rest = contextUtil.getRest(context)
        flag, free_disk_info, member_disk_info = \
            commonRestUtil.get_formatted_engine_disk_info(
                rest, None, None, logger, lang)

        if flag is False:
            logger.info("expdd.check.config.data.nil.disk")
            result_dict["flag"] = False
            result_dict["errMsg"], result_dict["suggestion"] = \
                common.getMsg(lang, "expdd.check.config.data.nil.disk")
            contextUtil.handleFailure(context, result_dict)
            return
        # 性能层空闲盘需要单独处理
        if contextUtil.getItem(context, "productModel") in config.NEW_DORADO + config.OCEAN_STOR_MICRO_DEVS:
            performance_layer_free_disk_info = commonRestUtil. \
                get_performance_layer_free_disk_info(rest, logger)
            free_disk_info.extend(performance_layer_free_disk_info)

        logger.info("free disk is: %s" % free_disk_info)

        # 3.比较扩容配置中的硬盘数量和实际空闲盘的数量
        # 3.1.扩容配置按引擎ID, 硬盘类型, 硬盘容量汇总硬盘数量
        summary_cfg_disk = get_summary_cfg_disk(cfg_data, logger)
        logger.info("summary config disk is: %s" % summary_cfg_disk)
        # 3.2.空闲盘按引擎ID, 硬盘类型, 硬盘容量汇总硬盘数量
        summary_free_disk = get_summary_free_disk(free_disk_info)
        logger.info("summary free disk is: %s" % summary_free_disk)

        # 4.比较扩容配置是否在空闲盘范围内
        flag, err_list = check_cfg_disk(summary_cfg_disk, summary_free_disk,
                                        logger)
        logger.info("result of checking is: %s" % flag)
        if not flag:
            set_err_msg(context, result_dict, err_list, lang)
            return

        contextUtil.setItem(context, "EXPAND_DD_EVAL_CONFIG_DATA", cfg_data)
        contextUtil.handleSuccess(context)
        return
    except Exception as exception:
        contextUtil.handleException(context, exception)
        logger.error("check config data error: %s" % str(exception))
        return
    finally:
        # 进度条刷为完成状态
        common.finishProcess(context)


def set_err_msg(context, result_dict, err_list, lang):
    """
    设置错误信息
    :param context: 上下文
    :param result_dict: 结果字典
    :param err_list: 错误信息列表
    :param lang: lang
    :return:
    """
    has_enclosure_redundant = contextUtil.getItem(
        context, "hasEnclosureRedundant", False)
    result_dict["flag"] = False
    for err_key in err_list:
        if len(err_key) >= 4:
            err_tuple = (err_key[0], err_key[1], err_key[2], err_key[3])
            err_msg, sugg = common.getMsg(lang,
                                          "expdd.check.match.error"
                                          ".with.enc.id", err_tuple)
            result_dict["errMsg"] += err_msg + "\n"
            result_dict["suggestion"] = sugg
        elif not has_enclosure_redundant:
            err_tuple = (err_key[0], err_key[1], err_key[2])
            err_msg, sugg = common.getMsg(lang,
                                          "expdd.check.match.error",
                                          err_tuple)
            result_dict["errMsg"] += err_msg + "\n"
            result_dict["suggestion"] = sugg
    contextUtil.handleFailure(context, result_dict)


def get_config_data(logger):
    """从文件中读取扩容配置数据
    数据在扩容评估检查项check_item_hardware_disk_capacity_cut_dorado_intf中写入

    :param logger: 日志句柄
    :return config_data_list: List类型的扩容配置数据
    """
    try:
        # 工具箱基本目录
        base_dir = os.path.abspath(DIR_RELATIVE_CMD)
        persist_dir = base_dir + CFG_DATA_PERSIST_DIR
        is_exists_data_dir = os.path.exists(persist_dir)
        if not is_exists_data_dir:
            logger.info("The directory does not exist: %s" % persist_dir)
            os.makedirs(persist_dir)

        persist_file = persist_dir + FILE_DEPSIGN + CFG_DATA_PERSIST_FILENAME
        with open(persist_file, 'r') as fileObj:
            config_data = fileObj.read()
            config_data_list = ast.literal_eval(config_data)
            logger.info("get_config_data is: %s" % config_data_list)
    except Exception as ex:
        logger.error("get_config_data error: %s" % str(ex))
        return []
    return config_data_list


def get_summary_cfg_disk(cfg_data, logger):
    """扩容配置按引擎ID, 硬盘类型, 硬盘容量汇总硬盘数量

    :param cfg_data: 扩容配置数据
    :return summaryCfgDisk: 汇总后的扩容配置数据
    """

    summaryCfgDisk = {}
    for a_data in cfg_data:
        engine_id = a_data.get("engine")
        disk_model = a_data.get("diskModel")
        disk_capacity = a_data.get("diskCapacity")
        disk_num = a_data.get("diskNum")
        # 空池不需要check硬盘是否够。
        if disk_num == EMPTY_POOL_FLAG:
            logger.info("exp empty pool, no need check disk number.")
            continue
        is_enclosure_redundant = str(a_data.get("isEnclosureRedundant"))
        key_tuple = (engine_id, disk_model, disk_capacity)
        tmp_disk_num = summaryCfgDisk.get(key_tuple, 0) + disk_num
        summaryCfgDisk[key_tuple] = tmp_disk_num
        if is_enclosure_redundant == "1":
            disk_enclosure = str(a_data.get("diskEnclosureName"))
            key_tuple = (engine_id, disk_model, disk_capacity, disk_enclosure)
            tmp_disk_num = summaryCfgDisk.get(key_tuple, 0) + disk_num
            summaryCfgDisk[key_tuple] = tmp_disk_num

    return summaryCfgDisk


def get_summary_free_disk(free_disk_info):
    """空闲盘按引擎ID, 硬盘类型, 硬盘容量汇总硬盘数量

    :param free_disk_info: 空闲盘数据
    :return summary_free_disk: 汇总后的空闲盘数据
    """

    summary_free_disk = {}
    for line in free_disk_info:
        engine_ids = line.get("engineIds")
        disks = line.get("disks")
        for engine_id in engine_ids:
            for disk in disks:
                disk_model = disk.get("type")
                disk_capacity = disk.get("capacity")
                disk_num = int(disk.get("count"))
                key_tuple = (engine_id, disk_model, disk_capacity)
                tmp_disk_num = summary_free_disk.get(key_tuple, 0) + disk_num
                summary_free_disk[key_tuple] = tmp_disk_num
                enclosure_infos = disk.get("enclosureInfo")
                for enclosure_info in enclosure_infos:
                    disk_enc_id = enclosure_info.get("name")
                    disk_enc_count = enclosure_info.get("count")
                    key_tuple_with_enclosure = (engine_id, disk_model,
                                                disk_capacity, disk_enc_id)
                    tmp_disk_num_with_enclosure = summary_free_disk.get(
                        key_tuple_with_enclosure, 0) + int(disk_enc_count)
                    summary_free_disk[key_tuple_with_enclosure] = \
                        tmp_disk_num_with_enclosure

    return summary_free_disk


def check_cfg_disk(summary_cfg_disk, summary_free_disk, logger):
    """比较扩容配置是否在空闲盘的范围内

    :param summary_cfg_disk: 按引擎ID, 硬盘类型, 硬盘容量汇总硬盘数量后的扩容配置
    :param summary_free_disk: 按引擎ID, 硬盘类型, 硬盘容量汇总硬盘数量后的空闲盘
    :param logger: 日志
    :return: True(扩容配置在空闲盘范围内)/False(扩容配置超过空闲盘范围)
    """

    err_list = []
    for cfg_key in summary_cfg_disk:
        cfg_disk_num = summary_cfg_disk.get(cfg_key)
        match_flag = False
        for free_key in summary_free_disk:
            free_disk_num = summary_free_disk.get(free_key)
            if is_same_disk_info(cfg_key, free_key):
                if cfg_disk_num > free_disk_num:
                    break
                else:
                    match_flag = True
                    break
        if match_flag is False:
            err_list.append(cfg_key)
            logger.info("err_key:" + str(cfg_key))

    if not err_list:
        return True, err_list
    else:
        return False, err_list


def is_same_disk_info(cfg_key, free_key):
    """判断两个硬盘信息是否相同

    :param cfg_key: 扩容配置硬盘信息
    :param free_key: 空闲硬盘信息
    :return: True(相同)/False(不相同)
    """
    cfg_engine_id = cfg_key[0]
    cfg_disk_modle = cfg_key[1]
    cfg_disk_capacity = float(cfg_key[2])
    free_engine_id = free_key[0]
    free_disk_modle = free_key[1]
    free_disk_capacity = float(free_key[2])
    cfg_disk_enclosure_id = ""
    free_disk_enclosure_id = ""
    if len(cfg_key) > 3:
        cfg_disk_enclosure_id = cfg_key[3]
    if len(free_key) > 3:
        free_disk_enclosure_id = free_key[3]

    if cfg_engine_id != free_engine_id:
        return False

    if cfg_disk_modle.upper() != free_disk_modle.upper():
        return False

    # 容量相差10G以内可以认为相等
    if abs(cfg_disk_capacity - free_disk_capacity) > 10.0:
        return False

    if cfg_disk_enclosure_id != free_disk_enclosure_id:
        return False

    return True
