# -*- coding: UTF-8 -*-

#  coding=UTF-8
#  Copyright (c) Huawei Technologies Co., Ltd. 2019-2024. All rights reserved.

import ast

from cbb.business.operate.checkitems import open_failover
from cbb.business.operate.expansion import common
from cbb.business.operate.expansion import config
from cbb.frame.cli import cliUtil
from cbb.frame.context import contextUtil
from cbb.frame.rest import commonRestUtil
from cbb.frame.rest import restData
from cbb.frame.rest import restUtil

# 进度总剩余时间
LIMIT_TIME = 1020
# 进度刷新间隔
INTERVAL = 2

# 扩空池时，需要将diskNum传入-2，产品特殊处理。
EMPTY_POOL_FLAG = "-2"

# 扩性能层的结果
EXP_PERFORMANCE_LAYER_RESULT = {}

# 硬盘框列表
FRAME_ID_LIST = "frameIdList"
# 硬盘数量列表
DISK_NUM_LIST = "diskNumList"
# 硬盘容量列表
DISK_CAP_LIST = "diskCapacityList"


def execute(context):
    '''
    @summary: 执行扩容硬盘域
    '''

    try:
        # 进度条刷新
        common.threadUpProcess(context, LIMIT_TIME, INTERVAL)

        logger = common.getLogger(context.get("logger"), __file__)
        lang = contextUtil.getLang(context)
        cli = contextUtil.getCli(context)

        cfg_data = contextUtil.getItem(context,
                                       "EXPAND_DD_EVAL_CONFIG_DATA")
        logger.logInfo("cfgData is: %s" % cfg_data)
        # 打开SMB开关
        flag, err_msg, suggestion = open_smb(context, lang, logger, cli,
                                             cfg_data)
        if not flag:
            result_dict = {"flag": False, "errMsg": err_msg,
                           "suggestion": suggestion}
            contextUtil.handleFailure(context, result_dict)
            return

        # 区分存储池和性能层
        storage_cfg_data = []
        performance_cfg_data = []
        for data in cfg_data:
            if str(data.get("performanceLayerModel")) == "1":
                performance_cfg_data.append(data)
            else:
                storage_cfg_data.append(data)

        err_msg_list, suggestion_list = expand_domain_and_performance_layer(
            context, storage_cfg_data, performance_cfg_data)
        if err_msg_list or suggestion_list:

            result_dict = {"flag": False,
                           "errMsg": "\n".join(err_msg_list),
                           "suggestion": "\n".join(suggestion_list)}
            contextUtil.handleFailure(context, result_dict)
            return

        logger.logInfo("excute expansion finished")
        contextUtil.setItem(context, "cifs_status", None)
        contextUtil.handleSuccess(context)
        logger.logPass()
        return

    except Exception as exception:
        contextUtil.handleException(context, exception)
        logger = common.getLogger(context.get("logger"), __file__)
        logger.logException(exception)
        return
    finally:
        # 进度条刷为完成状态
        common.finishProcess(context)


def is_need_exp_disk_domain_first(context, storage_cfg_data, performance_cfg_data):
    """
    是否需要先扩存储池（新融合框级冗余，同时扩容存储池和性能层时，谁的框多则先扩谁，框数一样则先扩性能层）
    @param context: 上下文
    @param storage_cfg_data: 硬盘域数据
    @param performance_cfg_data: 性能层数据
    @return: True：需要 False：不需要
    """
    # 不是同时扩存储池和性能层
    if not storage_cfg_data or not performance_cfg_data:
        return False
    # 不是框级冗余
    if storage_cfg_data and storage_cfg_data[0].get("isEnclosureRedundant") != "1":
        return False
    logger = common.getLogger(context.get("logger"), __file__)
    disk_domain_id = storage_cfg_data[0].get("domainId")
    performance_layer_id = performance_cfg_data[0].get("poolId")
    disk_domain_enc, performance_layer_enc, engine_list = get_member_disk_enc_and_engine_list(
        context, disk_domain_id, performance_layer_id)
    logger.logInfo("disk_domain_enc: {}, performance_layer_enc: {}, engine_list:{}".format(
        disk_domain_enc, performance_layer_enc, engine_list))
    # 新扩引擎时，先扩性能层
    new_engine_list = list()
    new_engine_list.extend([a_data.get("engine") for a_data in storage_cfg_data])
    new_engine_list.extend([a_data.get("engine") for a_data in performance_cfg_data])
    logger.logInfo("new_engine_list:{}".format(new_engine_list))
    for engine in new_engine_list:
        if engine not in engine_list:
            return False

    # 框级冗余，同时扩容存储池和性能层时，存储池的框比性能层的框多
    disk_domain_enc.extend([a_data.get("diskEnclosureName") for a_data in storage_cfg_data])
    performance_layer_enc.extend([a_data.get("diskEnclosureName") for a_data in performance_cfg_data])
    logger.logInfo("disk_domain_enc:{}, disk_domain_enc:{}".format(disk_domain_enc, performance_layer_enc))
    return len(set(disk_domain_enc)) > len(set(performance_layer_enc))


def get_member_disk_enc_and_engine_list(context, disk_domain_id, performance_layer_id):
    """
    获取成员硬盘框,引擎列表
    @param context: 上下文
    @param disk_domain_id: 硬盘域ID
    @param performance_layer_id: 性能层ID
    @return: 成员硬盘框列表
    """
    engine_list = list()
    rest = contextUtil.getRest(context)
    all_domain_disk_info = commonRestUtil.get_all_engine_disk_info_by_rest(rest, disk_domain_id)
    disk_domain_enc, disk_domain_engine_list = get_enc_and_engine_list(all_domain_disk_info)

    all_performance_disk_info = commonRestUtil.get_performance_layer_disk_info_by_rest(rest, performance_layer_id)
    performance_layer_enc, performance_layer_engine_list = get_enc_and_engine_list(all_performance_disk_info)

    engine_list.extend(disk_domain_engine_list)
    engine_list.extend(performance_layer_engine_list)

    return disk_domain_enc, performance_layer_enc, list(set(engine_list))


def get_enc_and_engine_list(all_disk_info):
    """
    获取硬盘框框列表,引擎列表
    @param all_disk_info: 所有的盘信息
    @return:
    """
    all_engine_disk_info_str = all_disk_info.get("data"). \
        get("allEngineDiskInfo").replace("true", "True"). \
        replace("false", "False")
    member_disk_list = ast.literal_eval(all_engine_disk_info_str).get("memberDisk", [])
    disk_domain_enc = list()
    engine_list = list()
    for member_disk in member_disk_list:
        for engine_info, engine_enc_info in member_disk.items():
            # 获取引擎
            engine_list.append(engine_info.rsplit(":")[-1])
            disk_enc = get_one_engine_disk_enc(engine_enc_info)
            disk_domain_enc.extend(disk_enc)
    return disk_domain_enc, engine_list


def get_one_engine_disk_enc(engine_enc_info):
    """
    获取一个引擎上的硬盘框
    @param engine_enc_info: 隐单个引擎上的硬盘框信息
    @return: 硬盘框列表
    """
    disk_enc_list = list()
    for enc_disk_info in engine_enc_info:
        for enc_info in enc_disk_info.get("enclosureInfo", []):
            disk_enc_list.append(enc_info.get("name"))
    return disk_enc_list


def expand_domain_and_performance_layer(context, storage_cfg_data, performance_cfg_data):
    """
    扩容硬盘域和性能层
    :param context: 上下文
    :param storage_cfg_data: 硬盘域配置数据
    :param performance_cfg_data: 性能层配置数据
    :return:
    """
    all_err_msg_list = list()
    all_suggestion_list = list()
    logger = common.getLogger(context.get("logger"), __file__)
    # 新融合框级冗余，同时扩容存储池和性能层时，谁的框多则先扩谁，框数一样则先扩性能层
    if is_need_exp_disk_domain_first(context, storage_cfg_data, performance_cfg_data):
        disk_domain_err_msg_list, disk_domain_suggestion_list = exp_disk_domain(context, storage_cfg_data, logger)
        performance_layer_err_msg_list, performance_layer_suggestion_list = exp_performance_layer(
            context, logger, performance_cfg_data, storage_cfg_data)
    else:
        performance_layer_err_msg_list, performance_layer_suggestion_list = exp_performance_layer(
            context, logger, performance_cfg_data, storage_cfg_data)
        disk_domain_err_msg_list, disk_domain_suggestion_list = exp_disk_domain(context, storage_cfg_data, logger)

    all_err_msg_list.extend(disk_domain_err_msg_list)
    all_err_msg_list.extend(performance_layer_err_msg_list)
    all_suggestion_list.extend(disk_domain_suggestion_list)
    all_suggestion_list.extend(performance_layer_suggestion_list)

    return all_err_msg_list, all_suggestion_list


def exp_disk_domain(context, storage_cfg_data, logger):
    """
    扩容存储池
    @param context: 上下文
    @param storage_cfg_data: 存储池扩容数据
    @param logger: 日志
    @return: err_msg_list， suggestion_list
    """
    err_msg_list = list()
    suggestion_list = list()
    # 当存储池已经扩容成功或不需要扩存储池时，重新执行不需要再扩
    if contextUtil.getItem(context, "EXP_STORAGE_POOL_SUCCESS", False) or not storage_cfg_data:
        return err_msg_list, err_msg_list

    # 扩容硬盘域
    exp_disk_domain_param = get_exp_disk_domain_param(logger, storage_cfg_data)

    # 按硬盘域循环下发命令
    for domain_id in exp_disk_domain_param:
        param = exp_disk_domain_param.get(domain_id)
        try:
            expan_disk_domain(context, domain_id, param)
            contextUtil.setItem(context, "EXP_STORAGE_POOL_SUCCESS", True)
        except Exception as exception:
            err_key = "expdd.check.error.common.info"
            set_err_msg_and_suggestion(context, err_key, domain_id, exception,
                                       err_msg_list, suggestion_list)
    return err_msg_list, err_msg_list


def exp_performance_layer(context, logger, performance_cfg_data, storage_cfg_data):
    """
    扩容性能层
    @param context: 上下文
    @param logger: 日志
    @param performance_cfg_data: 性能层扩容数据
    @param storage_cfg_data: 存储池扩容数据
    @return: err_msg_list， suggestion_list
    """
    err_msg_list = list()
    suggestion_list = list()
    # 不需要扩性能层
    if not performance_cfg_data or contextUtil.getItem(context, "EXP_PERFORMANCE_LAYER_SUCCESS", False):
        return err_msg_list, suggestion_list

    lang = contextUtil.getLang(context)
    cli = contextUtil.getCli(context)
    is_cross_engine = is_exp_pool_cross_engine(cli, lang, logger, storage_cfg_data, performance_cfg_data)
    exp_performance_layer_evaluate_param = get_exp_performance_layer_param(performance_cfg_data)
    msg = common.getRes(lang, "default_assignment")
    dialog_util = context.get("dialogUtil")
    quota_result = dialog_util.showWarningDialog(msg)
    for performance_layer_id, performance_param in exp_performance_layer_evaluate_param.items():
        try:
            exp_param = get_performance_quota_param(is_cross_engine, logger, performance_layer_id, quota_result)
            exp_param.update(performance_param)
            expand_performance_layer(context, performance_layer_id, exp_param)
            contextUtil.setItem(context, "EXP_PERFORMANCE_LAYER_SUCCESS", True)
        except Exception as exception:
            err_key = "exp.performance.layer.check.error.common.info"
            set_err_msg_and_suggestion(context, err_key, performance_layer_id,
                                       exception, err_msg_list, suggestion_list)
    return err_msg_list, suggestion_list


def get_performance_quota_param(is_cross_engine, logger, performance_layer_id, quota_result):
    """
    获取性能层配额占比参数
    @param is_cross_engine: 是否为扩引擎扩池
    @param logger: 日志
    @param performance_layer_id: 性能层ID
    @param quota_result: 配额占比结果
    @return: 配额占比参数
    """
    if is_cross_engine:
        logger.logInfo("exist exp pool cross engine.")
        exp_param = {"id": str(performance_layer_id)}
    else:
        logger.logInfo("no exp pool cross engine.")
        exp_param = {"id": str(performance_layer_id),
                     "autoAllocQuotaType": 1 if quota_result else 0}
    return exp_param


def set_err_msg_and_suggestion(context, err_key, pool_id, exception,
                               err_msg_list, suggestion_list):
    """
    扩容失败时，设置错误信息和修复建议
    :param context: 上下文
    :param err_key: 错误信息码
    :param pool_id: 硬盘域或性能层ID
    :param exception: 异常信息
    :param err_msg_list: 错误信息
    :param suggestion_list: 修复建议
    :return:
    """
    logger = common.getLogger(context.get("logger"), __file__)
    lang = contextUtil.getLang(context)
    error_code = exception[0]
    error_param = ()
    if len(exception.args) >= 3:
        error_param = tuple(exception[2].split(","))

    sug_key = "expdd.errorcode.{}".format(str(exception[0]))
    logger.logInfo("exception error code is {}, sug_key:{}, err_param:{}".format(
        error_code, sug_key, error_param)
    )
    msg = common.getMsg(lang, err_key, pool_id)[0]
    sug = common.getMsg(lang, sug_key, suggestionArgs=error_param)[1]
    logger.logException(exception)
    err_msg_list.append(msg)
    suggestion_list.append(sug)


def get_domain_eng_info(cli, lang):
    """
    获取硬盘域的引擎
    :param cli: ssh连接
    :param lang: 语言
    :return: 硬盘域的引擎
    """
    cmd = "show disk_domain general"
    flag, ret, msg = cliUtil.execCmdInCliMode(cli, cmd, True, lang)
    if flag is not True:
        return {}

    domain_detail_cmd_list = []
    domain_detail_cmd = 'show disk_domain general disk_domain_id={}'
    ret_list = cliUtil.getHorizontalCliRet(ret)
    [domain_detail_cmd_list.append(domain_detail_cmd.format(line.get("ID"))) for line in ret_list]

    domain_eng_dict = {}
    for cmd in domain_detail_cmd_list:
        flag, ret, msg = cliUtil.execCmdInCliMode(cli, cmd, True, lang)
        if flag is not True:
            continue
        ret_list = cliUtil.getVerticalCliRet(ret)
        for line in ret_list:
            domain_eng_dict[line.get("ID")] = line.get("Controller enclosure", "").split(",")

    return domain_eng_dict


def get_performance_layer_domain_info(cli, lang):
    """
    获取性能层关联的硬盘域ID
    :param cli: ssh连接
    :param lang: 语言
    :return: 性能层关联的硬盘域
    """
    cmd = "show performance_layer general"
    flag, ret, msg = cliUtil.execCmdInCliMode(cli, cmd, True, lang)
    if flag is not True:
        return {}

    performance_layer_cmd_list = []
    performance_layer_detail_cmd = 'show performance_layer general performance_layer_id={}'
    ret_list = cliUtil.getHorizontalCliRet(ret)
    [performance_layer_cmd_list.append(performance_layer_detail_cmd.format(line.get("ID")))
     for line in ret_list]

    performance_layer_domain_dict = {}
    for cmd in performance_layer_cmd_list:
        flag, ret, msg = cliUtil.execCmdInCliMode(cli, cmd, True, lang)
        if flag is not True:
            continue
        ret_list = cliUtil.getVerticalCliRet(ret)
        for line in ret_list:
            performance_layer_domain_dict[line.get("ID")] = line.get(
                "Associated Disk Domain Id List", "").split(",")
    return performance_layer_domain_dict


def is_exp_pool_cross_engine(cli, lang, logger, storage_cfg_data, performance_cfg_data):
    """
    是否跨引擎扩池场景
    :param cli: ssh连接
    :param lang: 语言
    :param logger: 日志
    :param storage_cfg_data: 存储池扩容参数
    :param performance_cfg_data: 性能层扩容参数
    :return: True 存在新扩引擎扩存储池， False 不存在
    """
    domain_eng_dict = get_domain_eng_info(cli, lang)
    performance_layer_domain_dict = get_performance_layer_domain_info(cli, lang)
    exp_disk_domain_param = get_exp_disk_domain_param(logger, storage_cfg_data)
    if not storage_cfg_data or not performance_cfg_data:
        return False
    for layer_cfg_data in performance_cfg_data:
        layer_id = layer_cfg_data.get("poolId")
        associated_disk_domain_list = performance_layer_domain_dict.get(layer_id, [])
        for domain_id, param_dict in exp_disk_domain_param.items():
            if domain_id not in associated_disk_domain_list:
                continue
            exp_engine_id_list = param_dict.get("engineIdList", [])
            exist_domain_list = domain_eng_dict.get(domain_id, [])
            if [eng for eng in exp_engine_id_list if eng not in exist_domain_list]:
                logger.logInfo(
                    "domain:{} is cross engin, exp_engine_id_list:{}, exist_domain_list:{}".format(
                        domain_id, exp_engine_id_list, exist_domain_list))
                return True
    return False


def expand_performance_layer(context, performance_layer_id, exp_param):
    """
    扩容性能层
    :param context: 上下文
    :param performance_layer_id: 性能层ID
    :param exp_param: 扩容性能层参数
    :return:
    """
    # 弹出提示框，由客户选择是否自动分配份额
    logger = common.getLogger(context.get("logger"), __file__)
    lang = contextUtil.getLang(context)
    rest = contextUtil.getRest(context)
    if EXP_PERFORMANCE_LAYER_RESULT.get(performance_layer_id, False):
        return
    uri_param_dict = restUtil.CommonRest.getUriParamDict(
        restData.RestCfg.OBJ.EXPANSION_PERFORMANCE_LAYER)
    logger.logInfo("exp_param:{}".format(exp_param))
    restUtil.CommonRest.execCmd(rest, uri_param_dict, exp_param,
                                restData.RestCfg.RestMethod.PUT)
    EXP_PERFORMANCE_LAYER_RESULT[performance_layer_id] = True


def get_exp_performance_layer_param(cfg_data_list):
    exp_performance_layer_evaluate_param = {}
    capacity_number_dict = {}
    for cfg_data in cfg_data_list:
        a_performance_evaluate_param = exp_performance_layer_evaluate_param.get(cfg_data.get("poolId"), {})
        disk_domain_id = cfg_data.get("poolId")
        disk_num = cfg_data.get("diskNum")
        disk_capacity_2_gb = cfg_data.get("diskCapacity")
        engine_id = cfg_data.get("engine")
        disk_enclosure_id = str(cfg_data.get("diskEnclosureId"))
        is_enc_redundant = str(cfg_data.get("isEnclosureRedundant")) == "1"
        tmp_dict = capacity_number_dict.get(disk_domain_id, {})
        if is_enc_redundant:
            if not tmp_dict.get(disk_enclosure_id):
                tmp_dict[disk_enclosure_id] = [disk_capacity_2_gb, int(disk_num)]
            else:
                tmp_dict.get(disk_enclosure_id)[1] += int(disk_num)
        else:
            tmp_dict[disk_capacity_2_gb] = tmp_dict.get(
                disk_capacity_2_gb, 0) + int(disk_num)
        capacity_number_dict[disk_domain_id] = tmp_dict
        a_performance_evaluate_param["engineIdList"] = list(set(
            a_performance_evaluate_param.get("engineIdList", []) + [
                engine_id]
        ))
        exp_performance_layer_evaluate_param[cfg_data.get("poolId")] = a_performance_evaluate_param
        set_frame_id_param(
            exp_performance_layer_evaluate_param, disk_enclosure_id, is_enc_redundant, disk_domain_id)
    set_exp_performance_layer_eval_param(
        capacity_number_dict, exp_performance_layer_evaluate_param
    )
    return exp_performance_layer_evaluate_param


def set_frame_id_param(exp_performance_layer_evaluate_param, disk_enclosure_id, is_enc_redundant, disk_domain_id):
    """
    设置一个性能层参数
    @param exp_performance_layer_evaluate_param: 性能层参数
    @param disk_enclosure_id: 硬盘框ID
    @param is_enc_redundant: 框级冗余
    @param disk_domain_id: 性能层ID
    """
    if not is_enc_redundant:
        return
    exp_performance_layer_evaluate_param.get(disk_domain_id)["frameLevelCheck"] = "1"
    if exp_performance_layer_evaluate_param.get(disk_domain_id).get(FRAME_ID_LIST):
        exp_performance_layer_evaluate_param.get(disk_domain_id)[FRAME_ID_LIST].append(disk_enclosure_id)
    else:
        exp_performance_layer_evaluate_param.get(disk_domain_id)[FRAME_ID_LIST] = [disk_enclosure_id]


def set_exp_performance_layer_eval_param(disk_cap_num_dict, eval_param):
    """
    设置性能层容量点和数量
    :param disk_cap_num_dict:
    :param eval_param:
    :return:
    """
    for domain_id, tmp_dict in disk_cap_num_dict.items():
        eval_param.get(domain_id)["id"] = domain_id
        if eval_param.get(domain_id).get(FRAME_ID_LIST):
            for enc_id in eval_param.get(domain_id).get(FRAME_ID_LIST):
                disk_num = eval_param.get(domain_id).get(DISK_NUM_LIST, [])
                disk_num.append(str(tmp_dict.get(enc_id)[1]))
                eval_param.get(domain_id)[DISK_NUM_LIST] = disk_num
                disk_cap = eval_param.get(domain_id).get(DISK_CAP_LIST, [])
                disk_cap.append(tmp_dict.get(enc_id)[0])
                eval_param.get(domain_id)[DISK_CAP_LIST] = disk_cap
            continue
        for cap, disk_num in tmp_dict.items():
            eval_param.get(domain_id)[DISK_NUM_LIST] = \
                eval_param.get(domain_id).get(DISK_NUM_LIST, []) + [str(disk_num)]
            eval_param.get(domain_id)[DISK_CAP_LIST] = \
                eval_param.get(domain_id).get(DISK_CAP_LIST, []) + [str(cap)]


def get_exp_disk_domain_param(logger, cfg_data):
    """
    获取扩容硬盘域参数
    :param logger: 日志
    :param cfg_data: 扩容配置数据
    :return:
    """
    # 整理配置数据 按硬盘域归集
    exp_disk_domain_param = {}
    for a_row_data in cfg_data:
        disk_domain_id = a_row_data.get("domainId")
        disk_num = a_row_data.get("diskNum")
        disk_capacity = str(int(a_row_data.get("diskCapacity")))
        engine_id = a_row_data.get("engine")
        disk_work_mode = a_row_data.get("diskWorkMode")
        disk_model = a_row_data.get("diskModel")
        disk_bom = a_row_data.get("diskBom")
        logger.logInfo("disk_bom: {}".format(disk_bom))
        disk_enclosure_id = str(a_row_data.get("diskEnclosureId"))
        disk_enclosure_redundant = str(a_row_data.get("isEnclosureRedundant"))
        a_exp_param = exp_disk_domain_param.get(disk_domain_id, None)
        if a_exp_param is None:
            a_exp_param = {"diskDomainId": disk_domain_id,
                           "diskNumList": [], "diskCapacityList": [],
                           "engineIdList": [], "frameIdList": [],
                           "nlsasDiskNumList": [],
                           "nlsasdiskCapacityList": [],
                           "tier0DiskType": [],
                           "tier2DiskType": [],
                           "frameLevelCheck": disk_enclosure_redundant,
                           "diskWorkMode": disk_work_mode}

        a_exp_param = get_a_exp_param(a_exp_param, disk_model, disk_bom,
                                      disk_enclosure_redundant, disk_capacity,
                                      disk_enclosure_id, disk_num, engine_id)
        exp_disk_domain_param[disk_domain_id] = a_exp_param

    return switch_disk_num_to_str(logger, exp_disk_domain_param)


def get_a_exp_param(a_exp_param, disk_model, disk_bom, disk_enclosure_redundant,
                    disk_capacity, disk_enclosure_id, disk_num, engine_id):
    """
    获取一个扩容参数
    :param a_exp_param: 扩容参数
    :param disk_model: 盘类型
    :param disk_bom: 盘BOM
    :param disk_enclosure_redundant: 是否框冗余
    :param disk_capacity: 盘容量
    :param disk_enclosure_id: 框ID
    :param disk_num: 盘数量
    :param engine_id: 引擎ID
    :return:
    """
    if disk_model in config.HDD_DISK_TYPE:
        disk_num_list_key = "nlsasDiskNumList"
        disk_capacity_list_key = "nlsasdiskCapacityList"
        disk_type_key = "tier2DiskType"
    else:
        disk_num_list_key = "diskNumList"
        disk_capacity_list_key = "diskCapacityList"
        disk_type_key = "tier0DiskType"

    if disk_enclosure_redundant == "1":
        info_need_add = True
        for i, _ in enumerate(a_exp_param.get(disk_capacity_list_key)):
            if a_exp_param[disk_capacity_list_key][i] == disk_capacity and \
                    a_exp_param["frameIdList"][i] == disk_enclosure_id:
                a_exp_param[disk_num_list_key][i] += disk_num
                info_need_add = False
                break
        if info_need_add:
            a_exp_param.get(disk_num_list_key).append(disk_num)
            a_exp_param.get(disk_capacity_list_key).append(disk_capacity)
            a_exp_param.get(disk_type_key).append(
                commonRestUtil.get_disk_type_enum(disk_model, disk_bom))
            a_exp_param.get("frameIdList").append(disk_enclosure_id)

    else:
        if disk_capacity not in a_exp_param.get(disk_capacity_list_key):
            a_exp_param.get(disk_num_list_key).append(disk_num)
            a_exp_param.get(disk_capacity_list_key).append(disk_capacity)
            a_exp_param.get(disk_type_key).append(
                commonRestUtil.get_disk_type_enum(disk_model, disk_bom))
        else:
            i = a_exp_param.get(disk_capacity_list_key).index(disk_capacity)
            a_exp_param.get(disk_num_list_key)[i] += disk_num
    if engine_id not in a_exp_param.get("engineIdList"):
        a_exp_param.get("engineIdList").append(str(engine_id))

    return a_exp_param


def switch_disk_num_to_str(logger, exp_disk_domain_param):
    """
    将硬盘数量从int转换为str
    :param logger: 日志
    :param exp_disk_domain_param: 扩容硬盘域参数
    :return:
    """
    for key in exp_disk_domain_param:
        item = exp_disk_domain_param.get(key)
        ssd_int_list = item.get("diskNumList")
        item["diskNumList"] = list(map(str, ssd_int_list))
        hdd_int_list = item.get("nlsasDiskNumList")
        item["nlsasDiskNumList"] = list(map(str, hdd_int_list))
    logger.logInfo("before expand storage pool, expEvaluateParam is: %s"
                   % str(exp_disk_domain_param))
    return exp_disk_domain_param


def expan_disk_domain(context, domain_id, param):
    """执行扩容硬盘域

    :param context: 变量上下文
    :param domain_id: 硬盘域ID
    :param param: 参数
    :return:
    """
    logger = common.getLogger(context.get("logger"), __file__)
    disk_num_list = param.get("diskNumList")
    disk_capacity_list = param.get("diskCapacityList")
    engine_id_list = param.get("engineIdList")
    nlsas_disk_num_list = param.get("nlsasDiskNumList")
    nlsasdisk_capacity_list = param.get("nlsasdiskCapacityList")
    tier0_disk_type = param.get("tier0DiskType")
    tier2_disk_type = param.get("tier2DiskType")
    frame_id_list = None
    if param.get("frameLevelCheck") == "1":
        frame_id_list = param.get("frameIdList")

    # 下发扩容评估命令
    uri_param_dict = \
        restUtil.CommonRest.getUriParamDict(
            restData.RestCfg.OBJ.EXPANSION_DISKDOMAIN, [domain_id])
    rest = contextUtil.getRest(context)
    exp_param = {}
    exp_param["TYPE"] = restData.Enum.ObjEnum.DiskPool
    exp_param["ID"] = domain_id
    if len(disk_num_list) > 0:
        exp_param["TIER0DISKTYPE"] = tier0_disk_type[0]
        exp_param["diskNumList"] = disk_num_list
        exp_param["diskCapacityList"] = disk_capacity_list
    if len(nlsas_disk_num_list) > 0:
        exp_param["nlsasDiskNumList"] = nlsas_disk_num_list
        exp_param["nlsasDiskCapacityList"] = nlsasdisk_capacity_list
        exp_param["TIER2DISKTYPE"] = tier2_disk_type[0]
    exp_param["ENGINEIDLIST"] = engine_id_list
    if frame_id_list:
        exp_param["frameIdList"] = frame_id_list
    # 扩空池，重置必要参数
    if EMPTY_POOL_FLAG in exp_param.get("diskNumList", []) or \
            EMPTY_POOL_FLAG in exp_param.get("nlsasDiskNumList", []):
        exp_param = {}
        exp_param["SSDDISKNUM"] = EMPTY_POOL_FLAG
        exp_param["ID"] = param.get("diskDomainId")
        exp_param["ENGINEIDLIST"] = param.get("engineIdList", ["CTE0"])
    logger.logInfo("in expan_disk_domain, param is: %s" % str(exp_param))
    record = \
        restUtil.CommonRest.execCmd(
            rest, uri_param_dict, exp_param,
            restData.RestCfg.RestMethod.PUT, 120, True)
    return record


def open_smb(context, lang, logger, cli, cfg_data):
    """
    打开SMB业务开关
    :param context:
    :param lang:
    :param logger:
    :param cli:
    :return:
    """
    # 获取存储池对应的引擎号集合
    cmd = "show storage_pool general"
    ret = cliUtil.excuteCmdInCliMode(cli, cmd, True, lang)
    if ret[0] is not True:
        raise Exception("Failed to get storage pool info.")
    pool_info_list = cliUtil.getHorizontalCliRet(ret[1])
    engine_id_list = []
    for pool_info in pool_info_list:
        dd_id = pool_info.get("Disk Domain ID")
        temp_id_list = get_dd_engine(cli, lang, dd_id)
        engine_id_list.extend(temp_id_list)
    logger.logInfo("engine_id_list: {}".format(str(engine_id_list)))

    # 获取新扩盘的引擎集合
    new_engine_id_list = []
    for data in cfg_data:
        engine_id = data.get("engine")
        new_engine_id_list.append(engine_id[3])
    logger.logInfo("new_engine_id_list: {}".format(str(new_engine_id_list)))
    # 判断是否原存储池新扩新引擎上的盘
    for new_engine_id in new_engine_id_list:
        if new_engine_id not in engine_id_list:
            return open_failover.execute(context)
    return True, "", ""


def get_dd_engine(cli, lang, dd_id):
    """
    获取存储池的引擎ID
    :param cli:
    :param lang:
    :param dd_id:
    :return:
    """
    cmd = "show disk_domain general disk_domain_id={}".format(dd_id)
    ret = cliUtil.excuteCmdInCliMode(cli, cmd, True, lang)
    if ret[0] is not True:
        raise Exception("Failed to query domain[{}]".format(dd_id))
    info = cliUtil.getVerticalCliRet(ret[1])
    engines = info[0].get("Controller enclosure", "").split(",")
    return [x[3] for x in engines]
