# -*- coding: UTF-8 -*-

import re
import os
import operator
import ast

from cbb.business.operate.expansion import common
from cbb.business.operate.expansion import config
from cbb.frame.base import baseUtil
from cbb.frame.context import contextUtil
from cbb.frame.tlv import tlvUtil
from expandDisk.script.moveSasCard.util import checkSymFreeCardSlot


def execute(context):
    '''
    @summary: 扩容硬盘框初始化
    '''

    logger = common.getLogger(context.get("logger"), __file__)
    try:
        resultDict = {"flag": True, "errMsg": "", "suggestion": ""}
        lang = contextUtil.getLang(context)

        # 设置内部IP
        devObj = contextUtil.getDevObj(context)
        innerIps = devObj.get("devInnerIpList")
        contextUtil.setItem(context, "innerIps", innerIps)
        logger.logInfo("innerIps:%s" % innerIps)

        # 针对扩容后再次扩容场景,清除新硬盘柜数量和硬盘框标记
        contextUtil.removeItem(context, "newConfigClustType")
        contextUtil.removeItem(context, "newConfigCtrlNum")

        # 检查工具与设备的连接
        isPass, errMsg = contextUtil.checkConnection(context)
        if not isPass:
            resultDict["flag"] = False
            resultDict["errMsg"], resultDict["suggestion"] = errMsg, ""
            contextUtil.handleFailure(context, resultDict)
            return

        tlv = contextUtil.getTlv(context)
        if tlv is None:
            logger.logNoPass("create tlv connection failure")
            resultDict["flag"] = False
            resultDict["errMsg"], resultDict["suggestion"] = common.getMsg(lang, "dev.conn.failure")
            contextUtil.handleFailure(context, resultDict)
            return

        cli = contextUtil.getCli(context)
        # 针对V3R3C20以后版本的资料归一问题
        common.setSpecialMsgInfo(cli, lang, logger)

        # 获取设备信息
        common.getClsInfoForExpandDisk(context, tlv)
        common.initClsInfoForEnclosures(context, tlv)

        # 上次扩容系统柜是否完成
        (isFinished, abnormalCtrls) = common.isExpandCtrlFinished(context)
        if not isFinished:
            logger.logNoPass("Expand controller is not finished")
            abnormalCtrlInfo = ",".join(abnormalCtrls)
            resultDict["flag"] = False
            resultDict["errMsg"], resultDict["suggestion"] = common.getMsg(lang, "cluster.expand.ctrl.not.finished",
                                                                           abnormalCtrlInfo)
            contextUtil.handleFailure(context, resultDict)
            return

        # 判断系统柜是否达到系统最大柜配置
        if common.checkMaxEnclosureConfig(context, tlv):
            logger.logNoPass("Max config of disk enclosure")
            resultDict["flag"] = False
            resultDict["errMsg"], resultDict["suggestion"] = common.getMsg(lang, "cluster.max.diskEnclosure.conf")
            contextUtil.handleFailure(context, resultDict)
            return

        # 4U框是否超过域的限制高度
        errEnclosureList = common.isOverAreaHeight(context)
        if len(errEnclosureList) > 0:
            logger.logNoPass("error connect enclosures: %s ." % str(errEnclosureList))
            resultDict["flag"] = False
            resultDict["errMsg"], resultDict["suggestion"] = common.getMsg(lang, "error.connet.enclosures",
                                                                           ', '.join(errEnclosureList))
            contextUtil.handleFailure(context, resultDict)
            return

        # 根据扩容场景,设置扩容流程
        couldOperate = setStepFlow(context)
        if not couldOperate:
            return

        contextUtil.handleSuccess(context)
        return

    except Exception as exception:
        contextUtil.handleException(context, exception)
        logger.logException(exception)
        return


def setStepFlow(context):
    '''
    @summary: 根据扩容场景,设置扩容流程
    '''
    # 根据step ID设置界面向导中的步骤
    baseDir = os.path.abspath("../..")
    needMoveCardFile = os.path.join(baseDir,
                                    config.CFG_DATA_PERSIST_DIR,
                                    contextUtil.getDevObj(context).get('sn') + '_needMoveSasCard')
    needMoveCard = os.path.exists(needMoveCardFile)
    runStepIdList = []

    logger = common.getLogger(context.get("logger"), __file__)
    logger.logInfo("need move sas card:%s" % needMoveCard)
    logger.logInfo("need move sas card file:%s" % needMoveCardFile)
    if needMoveCard:
        expData = readData(needMoveCardFile, contextUtil.getDevObj(context).get('sn'))
        logger.logInfo("expData is =" + str(expData))
        if not expData.get('couldMove'):
            lang = contextUtil.getLang(context)
            resultDict = dict()
            logger.logNoPass("could not move card, should re-evaluate")
            resultDict["flag"] = False
            resultDict["errMsg"], resultDict["suggestion"] = common.getMsg(lang, "could.not.move.card")
            contextUtil.handleFailure(context, resultDict)
            return False

        moveCards = expData.get("moveCardInfo", [])

        def cmpKey(moveCardInfo):
            mvCardCtrlSeqDict = {'B': 0, 'A': 1, 'D': 2, 'C': 3,
                                 'L1.IOM0': 0,
                                 'R1.IOM0': 1,
                                 'L1.IOM1': 2,
                                 'R1.IOM1': 3,
                                 }
            cardLocFields = moveCardInfo.get("srcLoc").split('.')
            if len(cardLocFields) == 2:
                engNum = cardLocFields[0].strip()
                ctrlId = cardLocFields[1][0]
                pri = mvCardCtrlSeqDict.get(ctrlId)
                return engNum, pri
            elif len(cardLocFields) == 3:
                engNum = cardLocFields[0].strip()
                slotNum = cardLocFields[1] + cardLocFields[2]
                pri = mvCardCtrlSeqDict.get(slotNum)
                return engNum, pri
            return ''

        moveCards.sort(key=lambda moveCardInfo: cmpKey(moveCardInfo))

        contextUtil.setItem(context, "moveCardInfo", expData)
        needMove, couldMove = checkSymFreeCardSlot.execute(context)
        # 需要移卡
        if needMove and couldMove:
            insertLoc = map(lambda x: x["srcLoc"], moveCards)
            contextUtil.setItem(context, "moveCardContext",
                                {"insertLoc": insertLoc,
                                 "allNum": len(moveCards),
                                 "index": 1 if len(moveCards) else 0})
            logger.logInfo('Move card confirmed.')
            runStepIdList.append("moveCardNotice")
            runStepIdList.append("moveCardPreCheck")
            [runStepIdList.append("poweroffAndMoveCard") for v in moveCards]
            runStepIdList.append("insertPCIeScaleup")

    if contextUtil.getItem(context, 'isDorado18000', False):
        runStepIdList.append("notice")
        runStepIdList.append("preCheck")
        runStepIdList.append("selectConfig")
        runStepIdList.append("notice.connectLine")
        runStepIdList.append("joinSystem")
        runStepIdList.append("confirmDiskEnc")
        runStepIdList.append("checkCables")
        runStepIdList.append("postCheck")
        runStepIdList.append("finished")
    else:
        runStepIdList.append("notice")
        runStepIdList.append("preCheck")
        runStepIdList.append("selectConfig")
        runStepIdList.append("notice.connectLine")
        runStepIdList.append("joinSystem")
        runStepIdList.append("checkCables")
        runStepIdList.append("postCheck")
        runStepIdList.append("finished")

    context["runStepIdList"] = ",".join(runStepIdList)
    return True


def handleDiskDomainInfoForDorado(context):
    """查询Dorado18000 V3 设备的硬盘域信息，并由此生成扩容配置规则
    :param context:
    :return:
    """
    logger = common.getLogger(context.get("logger"), __file__)
    resultDict = {}
    lang = contextUtil.getLang(context)
    tlv = contextUtil.getTlv(context)
    if tlv is None:
        logger.logNoPass("create rest connection failure")
        resultDict["flag"] = False
        resultDict["errMsg"], resultDict["suggestion"] = common.getMsg(lang, "dev.conn.failure")
        contextUtil.handleFailure(context, resultDict)
        return False

    controllerNum = tlvUtil.getControllersNum(tlv)
    logger.logInfo("controllerNum:%s" % controllerNum)

    isDorado18000 = contextUtil.getItem(context, 'isDorado18000', False)
    if isDorado18000:
        common.initAllCabinetUnusedDiskEncInfo(context)

    cli = contextUtil.getCli(context)
    qrySucc, diskDomainInfoDict = common.queryDiskDomain(cli, lang)
    if not qrySucc:
        logger.logNoPass('Query disk domain failed.')
        resultDict["flag"] = False
        resultDict["errMsg"], resultDict["suggestion"] = common.getMsg(lang, "query.disk.domain.info.failure")
        contextUtil.handleFailure(context, resultDict)
        return False

    allPosibleConfDictList = handleCtrlDiskDomain(context, diskDomainInfoDict, controllerNum)
    contextUtil.setItem(context, 'allPosibleConfDictList', allPosibleConfDictList)
    availableOwningCtrlList = [confDict.get('owningCtrl') for confDict in allPosibleConfDictList]

    contextUtil.setItem(context, "availableOwningCtrls", availableOwningCtrlList)
    contextUtil.setItem(context, "sysAvailableOwingCtrls", availableOwningCtrlList[:])
    contextUtil.setItem(context, "selectedOwingCtrls", [])

    logger.logInfo('allPosibleConfDictList: %s' % (str(allPosibleConfDictList)))
    logger.logInfo('availableOwningCtrlList: %s' % (str(availableOwningCtrlList)))
    return True


def isSymmetricalDiskDomain(domainOwningCtrl):
    """判断硬盘域是否跨引擎(逻辑引擎），如果跨引擎，则需要对称扩框。
    :param domainOwningCtrl:
    :return:
    """
    return len(domainOwningCtrl.split(',')) > 2


def standardizeSymmetricDiskDomainOwningCtrl(owningCtrl):
    """硬盘域归属引擎标准化，和DeviceManager上展示一致：CTEx(xA/xB) CTEy(yC/yD)
    或者 "CTEx(xA/xB/xC/xD)"
    :param owningCtrl:
    :return:
    """
    ctrls = str(owningCtrl).split(',')
    ctrls = map(str.upper, ctrls)
    smb0Ctrls = filter(lambda ctrl: ctrl.startswith('0'), ctrls)
    smb1Ctrls = filter(lambda ctrl: ctrl.startswith('1'), ctrls)
    smbNum = 2 if all([smb0Ctrls, smb1Ctrls]) else 1
    smb0Below = True if '0A' in smb0Ctrls else False
    smb1Below = True if '1A' in smb1Ctrls else False
    if smbNum == 2:
        if smb0Below and smb1Below:
            return 'CTE0(0A/0B) CTE1(1A/1B)'
        elif smb0Below and not smb1Below:
            return 'CTE0(0A/0B) CTE1(1C/1D)'
        elif not smb0Below and smb1Below:
            return 'CTE0(0C/0D) CTE1(1A/1B)'
        else:
            return 'CTE0(0C/0D) CTE1(1C/1D)'
    else:
        if smb0Ctrls:
            return 'CTE0(0A/0B/0C/0D)'
        else:
            return 'CTE1(1A/1B/1C/1D)'


def standardizeSymmetricDiskDomainOwningCtrlForNonDorado18000(owningCtrl):
    """硬盘域归属引擎标准化，和DeviceManager上展示一致：CTEx(xA/xB) CTE3(3A/3B)

    :param owningCtrl:
    :return:
    """
    ctrls = str(owningCtrl).split(',')
    ctrls = map(str.upper, ctrls)
    engineIdList = sorted([owningCtrl[0] for owningCtrl in ctrls])
    engineId0 = engineIdList[0]
    engineId1 = engineIdList[2]
    return 'CTE%(engineId0)s(%(engineId0)sA/%(engineId0)sB) CTE%(engineId1)s(%(engineId1)sA/%(engineId1)sB)' \
           % {'engineId0': engineId0, 'engineId1': engineId1}


def handleCtrlDiskDomain(context, diskDomainInfoDict, controllerNum):
    """根据当前系统控制器个数和硬盘域的归属控制器，计算硬盘框扩框规则。
    跨引擎（逻辑引擎：一个双控.）的硬盘域必须对称扩框，其余的引擎独立扩框。
    :param context:
    :param diskDomainInfoDict:
    :param controllerNum:
    :return:
    """
    logger = common.getLogger(context.get("logger"), __file__)
    isDorado18000 = contextUtil.getItem(context, 'isDorado18000', False)
    symmetricDDIdList = filter(lambda ddId: isSymmetricalDiskDomain(diskDomainInfoDict.get(ddId).get('Controller')),
                               diskDomainInfoDict)

    symDDOwingEncIdPairList = []
    for ddId in symmetricDDIdList:
        owningCtrlIdList = diskDomainInfoDict.get(ddId).get('Controller').split(',')
        owningCteIdList = list(set(map(lambda ctrlId: ctrlId[0], owningCtrlIdList)))
        if len(owningCteIdList) >= 2:
            symDDOwingEncIdPairList.append((owningCteIdList[0], owningCteIdList[1]))
        else:
            symDDOwingEncIdPairList.append((owningCteIdList[0], owningCteIdList[0]))

    contextUtil.setItem(context, 'symDDOwingEncIdPairList', symDDOwingEncIdPairList)

    logger.logInfo('DiskDomainInfoDict:%s' % (str(diskDomainInfoDict)))
    logger.logInfo('symmetricDDIdList:%s' % (str(symmetricDDIdList)))
    logger.logInfo('symDDOwingEncIdPairList:%s' % (str(symDDOwingEncIdPairList)))

    allPosibleConfDictList = []
    for symmetricDDId in symmetricDDIdList:
        ddCtrl = diskDomainInfoDict.get(symmetricDDId).get('Controller')
        if not isDorado18000:
            symmetricDDCtrl = standardizeSymmetricDiskDomainOwningCtrlForNonDorado18000(ddCtrl)
            allPosibleConfDictList.append(
                {'owningCtrl': symmetricDDCtrl,
                 'independentExp': False,  # 独立扩容
                 'ruleDict': getSymmetricExpRuleForDoradoNon18000(context, symmetricDDCtrl),
                 })
        else:
            symmetricDDCtrl = standardizeSymmetricDiskDomainOwningCtrl(ddCtrl)
            allPosibleConfDictList.append(
                {'owningCtrl': symmetricDDCtrl,
                 'independentExp': False,  # 独立扩容
                 'ruleDict': getSymmetricExpRule(context, symmetricDDCtrl),
                 })

    symmetricDDOwningCtrlList = [confDict.get('owningCtrl') for confDict in allPosibleConfDictList]
    allOwningCtrlList = getAllPosibleOwningCtrlList(isDorado18000, controllerNum)

    def isNotInAnySymmetriDDOwningCtrl(owningCtrl):
        """判断一个单引擎ID是否不在任意1个跨引擎的硬盘域的部署引擎ID中.
        :param owningCtrl:
        :return:
        """
        for symmetricDDOwningCtrl in symmetricDDOwningCtrlList:
            if owningCtrl[4:].strip('(').strip(')') in symmetricDDOwningCtrl:
                return False
        return True

    independentExpCtrlList = list(filter(isNotInAnySymmetriDDOwningCtrl, allOwningCtrlList))
    for owningCtrl in independentExpCtrlList:
        if not isDorado18000:
            allPosibleConfDictList.append(
                {'owningCtrl': owningCtrl,
                 'independentExp': True,  # 独立扩容
                 'ruleDict': getAsymmetricExpRuleForDoradoNon180000(context, owningCtrl),
                 })
        else:
            allPosibleConfDictList.append(
                {'owningCtrl': owningCtrl,
                 'independentExp': True,  # 独立扩容
                 'ruleDict': getAsymmetricExpRule(context, owningCtrl),
                 })
    return allPosibleConfDictList


def getAllPosibleOwningCtrlList(isDorado18000, controllerNum):
    """根据控制器个数获取可能的归属引擎的列表

    :param isDorado18000: 是Dorado18000设备型号。
    :param controllerNum:
    :return:
    """
    if not isDorado18000:
        allOwningCtrlList = ['CTE%s(%sA/%sB)' % (engId, engId, engId) for engId in range(controllerNum/2)]
    else:
        if controllerNum == 2:
            allOwningCtrlList = ['CTE0(0A/0B)']
        elif controllerNum == 4:
            allOwningCtrlList = ['CTE0(0A/0B)', 'CTE0(0C/0D)']
        elif controllerNum == 6:
            allOwningCtrlList = ['CTE0(0A/0B)', 'CTE0(0C/0D)', 'CTE1(1A/1B)']
        else:
            allOwningCtrlList = ['CTE0(0A/0B)', 'CTE0(0C/0D)', 'CTE1(1A/1B)', 'CTE1(1C/1D)']
    return allOwningCtrlList


def getSymmetricExpRule(context, symmetricDDOwningCtrl):
    """
    获取跨引擎的硬盘域扩容的规则。
    :param context:
    :param symmetricDDOwningCtrl: 跨引擎的硬盘域的部署引擎，如："CTE0(0A/0B) CTE1(1C/1D)"
    :return:
    """
    allFreeDiskEncInfoDict = contextUtil.getItem(context, 'allCabinetDiskEncInfoDict', {})
    ownCtrlList = symmetricDDOwningCtrl.split()
    if len(ownCtrlList) == 2:
        ownCtrl1 = ownCtrlList[0]
        ownCtrl2 = ownCtrlList[1]
        if 'A' in ownCtrl1:
            position1 = 'below'
        else:
            position1 = 'above'
        if 'A' in ownCtrl2:
            position2 = 'below'
        else:
            position2 = 'above'
        cabinetName1 = 'SMB' + ownCtrl1.split('(')[0].split('CTE')[1]
        cabinetName2 = 'SMB' + ownCtrl2.split('(')[0].split('CTE')[1]
    else:
        cabinetName1 = 'SMB' + ownCtrlList[0].split('(')[0].split('CTE')[1]
        cabinetName2 = cabinetName1
        position1, position2 = 'above', 'below'
    freeKeyWord1 = position1 + 'Free'
    freeDiskEncList1 = allFreeDiskEncInfoDict.get(cabinetName1, {}).get(freeKeyWord1, [])
    freeKeyWord2 = position2 + 'Free'
    freeDiskEncList2 = allFreeDiskEncInfoDict.get(cabinetName2, {}).get(freeKeyWord2, [])
    totFreeNum = len(freeDiskEncList1) + len(freeDiskEncList2)
    # 两套设备已经占用的硬盘框的个数的差值.
    usedOffset = abs((8 - len(freeDiskEncList1)) - (8 - len(freeDiskEncList2)))
    totRules = range(usedOffset + 1, totFreeNum + 1)
    if totFreeNum % 2 == 0:
        ruleList = range(usedOffset + 1) + filter(lambda num: num % 2 == 0, totRules)
    else:
        ruleList = range(usedOffset + 1) + filter(lambda num: num % 2 == 1, totRules)

    ruleDict = {}
    ruleList = map(str, ruleList)
    ruleDict['rule'] = ','.join(ruleList)
    ruleDict['freeDiskEncList1'] = freeDiskEncList1
    ruleDict['freeDiskEncList2'] = freeDiskEncList2
    return ruleDict


def getSymmetricExpRuleForDoradoNon18000(context, symmetricDDOwningCtrl):
    """获取跨引擎的硬盘域扩容的规则(5000/6000)。

    :param context:
    :param symmetricDDOwningCtrl: 跨引擎的硬盘域的部署引擎，如："CTE0(0A/0B) CTE1(1A/1B)"
    :return:
    """
    productModel = contextUtil.getItem(context, 'productModel')
    isDorado = contextUtil.getItem(context, "isDorado", False)
    if isDorado:
        singleMaxDaeNum = \
            config.DORADO_V6_SINGLE_ENG_MAX_DAE_NUM.get(productModel, 0)
        interModel = contextUtil.getItem(context, "interProductModel")
        if interModel:
            tepMaxSupportDaeNum = \
                config.DORADO_V6_SINGLE_ENG_MAX_DAE_NUM.get(interModel, 0)
            if tepMaxSupportDaeNum:
                singleMaxDaeNum = tepMaxSupportDaeNum
    else:
        singleMaxDaeNum = config.V5R8C00_MAX_DAE_NUM.get(productModel, 0)

    symDDEngIdList = re.findall(r'(\d)\(', symmetricDDOwningCtrl)
    originDaeIdSet = contextUtil.getItem(context, "originDiskEncIdSet", set())
    symDDEngEncIdSet = filter(lambda daeId: daeId[3] in symDDEngIdList, originDaeIdSet)

    originSingleEngDaeNum = len(symDDEngEncIdSet) / 2
    remain = len(symDDEngEncIdSet) % 2 # 若扩容前不对称，扩容时只能选择奇数框

    singleEngMaxFreeNum = singleMaxDaeNum - originSingleEngDaeNum
    ruleList = [(2 * daeNum - remain) for daeNum in range(1, singleEngMaxFreeNum + 1)]

    ruleDict = {}
    ruleList = map(str, ruleList)
    ruleDict['rule'] = ','.join(ruleList)
    return ruleDict


def getAsymmetricExpRuleForDoradoNon180000(context, asymmetricDDOwningCtrl):
    """获取单引擎的部署的硬盘域扩容的规则(Dorado5000/6000 设备型号）。
    :param context:
    :param asymmetricDDOwningCtrl: 不跨引擎的硬盘域的部署引擎，如："CTE1(1C/1D)"
    :return:
    """
    originDaeIdSet = contextUtil.getItem(context, "originDiskEncIdSet", set())
    asymDDOwningEng = asymmetricDDOwningCtrl[3]
    asymDDEngEncIdList = filter(lambda daeId: daeId[3] == asymDDOwningEng, originDaeIdSet)

    productModel = contextUtil.getItem(context, 'productModel')
    isDorado = contextUtil.getItem(context, "isDorado", False)
    if isDorado:
        singleMaxDaeNum = config.DORADO_V6_SINGLE_ENG_MAX_DAE_NUM.get(
            productModel, 0)
        interModel = contextUtil.getItem(context, "interProductModel")
        if interModel:
            tepMaxSupportDaeNum = config.DORADO_V6_SINGLE_ENG_MAX_DAE_NUM.get(
                interModel, 0)
            if tepMaxSupportDaeNum:
                singleMaxDaeNum = tepMaxSupportDaeNum

        freeDaeNum = singleMaxDaeNum - len(asymDDEngEncIdList)
    else:
        maxSupportDaeNum = config.V5R8C00_MAX_DAE_NUM.get(productModel, 0)
        freeDaeNum = maxSupportDaeNum - len(asymDDEngEncIdList)

    ruleDict = {}
    ruleList = map(str, range(1, freeDaeNum + 1))
    ruleDict['rule'] = ','.join(ruleList)
    return ruleDict


def getAsymmetricExpRule(context, asymmetricDDOwningCtrl):
    """获取单引擎的部署的硬盘域扩容的规则。
    :param context:
    :param asymmetricDDOwningCtrl: 跨引擎的硬盘域的部署引擎，如："CTE1(1C/1D)"
    :return:
    """
    allFreeDiskEncInfoDict = contextUtil.getItem(context, 'allCabinetDiskEncInfoDict', {})
    ownCtrl = asymmetricDDOwningCtrl.split()[0]
    if 'A' in ownCtrl:
        position = 'below'
    else:
        position = 'above'
    cabinetName = 'SMB' + ownCtrl.split('(')[0].split('CTE')[1]
    freeKeyWord = position + 'Free'
    freeDiskEncList = allFreeDiskEncInfoDict.get(cabinetName, {}).get(freeKeyWord, [])
    freeNum = len(freeDiskEncList)
    totRules = range(freeNum + 1)
    ruleDict = {}
    ruleList = map(str, totRules)
    ruleDict['rule'] = ','.join(ruleList)
    ruleDict['freeDiskEncList'] = freeDiskEncList
    return ruleDict


def readData(needMoveCardFile, sn):
    # 创建临时文件collectInfos.bat
    try:
        # 工具箱基本目录
        # 工具箱基本目录
        fileObj = open(needMoveCardFile, 'r')
        configData = fileObj.read()
        configDataDict = ast.literal_eval(unicode(configData))
    except Exception as ex:
        return {}
    finally:
        fileObj.close()
    return configDataDict
