# -*- coding: UTF-8 -*-
#  Copyright (c) Huawei Technologies Co., Ltd. 2021-2024. All rights reserved.

import json
import time
import traceback

import os
import re
from com.huawei.ism.exception import IsmException
from com.huawei.ism.tool.devicemanager.utils import LicenseFeatureQueryUtil
from com.huawei.ism.tool.obase.entity import EntityUtils
from com.huawei.ism.tool.protocol.tlv.exception import TLVException
from java.lang import Exception as JException
from psdk.checkitem.common.base_dsl_check import BaseCheckItem
from psdk.dsl import fault_mode as ft
from psdk.dsl.dsl_common import get_version_info
from psdk.platform.entity.check_status import CheckStatus
from psdk.platform.protocol.cli_con_mgr import get_available_conn
from psdk.platform.protocol.cli_factory import create_cli_conn
from psdk.platform.util.echo_parser import get_vertical_cli_ret, get_horizontal_cli_ret


def get_result(status, err_msg=None, data=None):
    return {"status": status, "detail": err_msg, "data": data}


def get_callback_params(status, custom_result=None):
    params = {"status": status, "customResult": custom_result}
    return json.dumps(params)


DEPLOYED_STATUS = "deployed"
DISK_DOMAIN_NAME = "disk_domain_0"
STORAGE_POOL_NAME = "storage_pool_0"
PERFORMANCE_LAYER_NAME="performance_layer_0"
DEPLOYED_TYPE = {"OceanProtect X8000": "d0", "OceanProtect X8000K": "d0", "OceanProtect A8000": "d0",
                 "OceanProtect X6000": "d1" , "OceanProtect X9000": "d6", "OceanProtect X9000K": "d6"}
YES = "y"
# 查询间隔时间15秒
QUERY_INTERVAL = 15
# 等待设备重启查询次数
RESTART_TIMES = 120
# 等待容器部署查询次数
ACTIVATION_TIMES = 360
RESTART_TIMEOUT = 30 * 60
ACTIVATION_TIMEOUT = 2 * 60 * 60
OM_OPERATE_FAIL_CODE = 1077949004
SPECIAL_IMAGE_PATH_1 = "/OSM/coffer_data/dataprotect/OceanProtect_DataProtect_1.2.1RC2_image_ARM_64\s(1).tgz"
SPECIAL_IMAGE_PATH_2 = "/OSM/coffer_data/dataprotect/OceanProtect_DataProtect_1.2.1RC2_image_ARM_64 (1).tgz"


class PyTaskService(BaseCheckItem):

    def __init__(self, base_context, call_back):
        super(PyTaskService, self).__init__(base_context)
        self.dev = base_context.get_dev_node().origin_dev
        self.work_path = os.path.dirname(__file__)
        self.config_path = os.path.join(self.work_path, "config")
        self.script_path = os.path.join(self.work_path, "scripts")
        self.scene_path = os.path.join(self.work_path, "scene")
        self.call_back = call_back
        # 初始化几个私有变量
        self._chart_path = None
        self._image_path = None
        self._raid_level = None
        self._ctrl_num = None
        self._ctrl_list = None
        self._pool_id = None
        self._disk_domain_id = None
        self._performance_layer_id = None
        self._package_version = None
        self._all_disks_ids = None
        self.dev_version = self.get_dev_version()
        self._gauss_db_pwd = None

    @staticmethod
    def compare_version(version, reference_version):
        """
        比较版本，返回原版本与参考版本的大小关系{前者大于后者 ：1；相等：0；小于：-1；不比较：-2}
        :param version:
        :param reference_version: 参照版本
        :return:
        """
        from cbb.business.collect.products_adapter import compare_version
        return compare_version(version, reference_version)

    def get_dev_version(self):
        version_info = get_version_info(self.dsl)
        return version_info.get("base_version").get("Current Version")

    # 入口执行方法
    def run(self, *args, **kwargs):
        self.parse_task_info(kwargs)
        # 需要执行的方法
        methods = kwargs.get("methods").split(",")
        for method_name in methods:
            func = getattr(self, method_name, None)
            if func:
                begin_time = time.strftime("%Y-%m-%d %H:%M:%S", time.localtime())
                # 开始之前，调用一下回调的开始
                custom_result = {"beginTime": begin_time, "methodName": method_name}
                self.call_back.start(get_callback_params(CheckStatus.PASS, custom_result))
                return_value = func()
                status = return_value[0]
                err_msg = return_value[1]
                data = return_value[2] if len(return_value) >= 3 else None
                end_time = time.strftime("%Y-%m-%d %H:%M:%S", time.localtime())
                # 组装回调的参数
                custom_result = {"beginTime": begin_time, "endTime": end_time, "detail": err_msg,
                                 "methodName": method_name, "data": data}
                if status == CheckStatus.NOT_PASS:
                    # 执行不通过也回调一下
                    self.call_back.update(get_callback_params(status, custom_result))
                    return get_result(status, err_msg, data)
                else:
                    # 执行通过，回调一下
                    self.call_back.update(get_callback_params(status, custom_result))
            else:
                return get_result(CheckStatus.NOT_PASS, "process.task.step.detail.step_not_exist", method_name)
        return get_result(CheckStatus.PASS, "")

    def parse_task_info(self, kwargs):
        # 解析raid级别
        self._raid_level = kwargs.get("raidLevel")
        image_path = kwargs.get("imagePath")
        # 解析镜像包路径
        if image_path:
            if image_path.startswith("error"):
                self._image_path = image_path
            paths = image_path.split(",")
            for path in paths:
                if 'image' in path and path.strip().endswith('.tgz'):
                    self._image_path = SPECIAL_IMAGE_PATH_1 if path == SPECIAL_IMAGE_PATH_2 else path
                elif 'chart' in path and path.strip().endswith('.tgz'):
                    self._chart_path = path
        # 解析设备硬盘设置相关信息
        self._all_disks_ids = kwargs.get("allDisksIds")
        # 解析高斯db数据库初始密码
        if kwargs.get("gaussDbPwd"):
            self.logger.info("The GaussDB database initialization information is read.")
            self._gauss_db_pwd = kwargs.get("gaussDbPwd")
        self.logger.info("start parse dev({}) info. raidLevel is {}. imagePath is {}. allDisksIds is {}. methods is {}."
            .format(self.context.dev_node.ip, self._raid_level, image_path, self._all_disks_ids, kwargs.get("methods")))

    def check_package(self):
        """
        检查软件包是否存在
        """
        if self._image_path and self._image_path.startswith("error"):
            return CheckStatus.NOT_PASS, "process.task.step.detail.import_image_0A_controller_ip_not_exist", ""
        if not self._image_path:
            return CheckStatus.NOT_PASS, "process.task.step.detail.no_image", ""
        if not self._chart_path:
            return CheckStatus.NOT_PASS, "process.task.step.detail.no_chart", ""
        return CheckStatus.PASS, "", ""

    def check_admin(self):
        """
        判断是否是管理员登录
        """
        if self.context.dev_node.user_name == 'admin':
            return CheckStatus.PASS, "", ""
        return CheckStatus.NOT_PASS, "process.task.step.detail.is_not_admin", ""

    def check_controller_status(self):
        """
        查看控制器状态是否正常
        """
        result = self.dsl("exec_mini_system 'showsysstatus' | vertical_parser | get_index(0)")
        node_cfg = int(result.get("node cfg"))
        result = self.dsl("exec_mini_system 'showsysstatus' | horizontal_parser")
        if node_cfg > len(result):
            return CheckStatus.NOT_PASS, "process.task.step.detail.controller_error", ""
        if not all(c.get("status") == 'normal' for c in result):
            return CheckStatus.NOT_PASS, "process.task.step.detail.controller_error", ""
        # 获取控制器列表
        self._ctrl_num = len(result)
        result = self.dsl("exec_cli 'show controller general' | vertical_parser")
        self._ctrl_list = ",".join(c.get("Controller") for c in result)
        return CheckStatus.PASS, "", ""

    def check_container_service_status(self):
        """
        查询容器应用是否正常
        """
        result = self._query_application_status()
        if not result or not result.get("Status"):
            return CheckStatus.PASS, "", ""
        if result.get("Status") == DEPLOYED_STATUS:
            return CheckStatus.NOT_PASS, "process.task.step.detail.have_do", ""
        if result.get("Status") == 'pending-install':
            return CheckStatus.NOT_PASS, "process.task.step.detail.app_doing", ""
        if result.get("Status") != DEPLOYED_STATUS:
            return CheckStatus.NOT_PASS, "process.task.step.detail.app_not_normal", ""

    def _query_application_status(self):
        """
        查询容器应用状态
        """
        if 'dataprotect' not in self.dsl("exec_cli 'show container_application general'"):
            return {}

        result = self.dsl("exec_cli 'show container_application general name=dataprotect'")
        cmd = 'show container_application general name=dataprotect namespace=dpa' \
            if 'namespace=?' in result \
            else 'show container_application general name=dataprotect'

        return self.dsl("exec_cli '{}' | vertical_parser | get_index(0)".format(cmd), need_log=False, return_if={ft.FindException: {}})

    def check_disk_domain_name(self):
        """
        检查硬盘域的名称(存在名称不为disk_domain_0的硬盘域报错)
        """
        result = self.dsl("exec_cli 'show disk_domain general' | horizontal_parser")
        if not result:
            return CheckStatus.PASS, "", ""
        if result and len(result) == 1 and result[0].get("Name") == DISK_DOMAIN_NAME:
            self._disk_domain_id = result[0].get("ID")
            return CheckStatus.PASS, "", ""
        return CheckStatus.NOT_PASS, "process.task.step.detail.disk_domain_name_error", ""

    def check_storage_pool_name(self):
        """
        检查存储池的名称(存在名称不为storage_pool_0的存储池报错)
        """
        result = self.dsl("exec_cli 'show storage_pool general' | horizontal_parser",
                          return_if={ft.FindException: [{"name": "error"}]})
        if not result:
            return CheckStatus.PASS, "", ""
        if result and len(result) == 1 and result[0].get("Name") == STORAGE_POOL_NAME:
            self._pool_id = result[0].get("ID")
            # 如果已经存在，判断一下是否和用户选择的raid级别一样
            command = "exec_cli 'show storage_pool general pool_id={pool_id}' | vertical_parser"
            result = self.dsl(command.format(pool_id=self._pool_id))
            if result[0].get("Raid Level") == self._raid_level:
                return CheckStatus.PASS, "", ""
            else:
                return CheckStatus.NOT_PASS, "process.task.step.detail.storage_pool_level_error", \
                       result[0].get("Raid Level")
        return CheckStatus.NOT_PASS, "process.task.step.detail.storage_pool_name_error", ""

    def create_disk_domain(self):
        """
        创建硬盘域
        """
        if self._disk_domain_id:
            return CheckStatus.PASS, "", ""
        # 检查硬盘域是否存在
        result = self.dsl("exec_cli 'show disk_domain general' | horizontal_parser",
                          return_if={ft.FindException: [{"name": "error"}]})
        if result and len(result) == 1 and result[0].get("Name") == DISK_DOMAIN_NAME:  # 已经存在硬盘域，不再创建
            self._disk_domain_id = result[0].get("ID")
            return CheckStatus.PASS, "", ""
        if not result:  # 还没有硬盘域，创建硬盘域
            if LicenseFeatureQueryUtil.hasHyperEncryptionFeature(EntityUtils.toOldDev(self.context.java_dev)):
                return self.create_disk_domain_has_encryption()
            return self.create_disk_domain_no_encryption()
        return CheckStatus.NOT_PASS, "process.task.step.detail.disk_domain_name_error", ""

    def create_disk_domain_no_encryption(self):
        command = "exec_cli 'create disk_domain name={disk_domain_name} disk_list=all raid_level={raid_level}'"
        result = self.dsl(command.format(disk_domain_name=DISK_DOMAIN_NAME, raid_level=self._raid_level))
        if 'successfully' in result:
            return CheckStatus.PASS, "", ""
        else:
            return CheckStatus.NOT_PASS, "process.task.step.detail.create_disk_domain_fail", result

    def create_disk_domain_has_encryption(self):
        flag, error_key, msg = self.check_device_password_management_service()
        if flag is CheckStatus.NOT_PASS:
            return flag, error_key, msg
        command = "exec_cli 'create disk_domain name={} disk_list=all raid_level={} disk_encryption_switch=on disk_encryption_algorithm=AES'"
        result = self.dsl(command.format(DISK_DOMAIN_NAME, self._raid_level))
        if 'successfully' in result:
            return CheckStatus.PASS, "", ""
        else:
            return CheckStatus.NOT_PASS, "process.task.step.detail.create_disk_domain_fail", result

    def create_storage_pool(self):
        """
        创建存储池
        """
        if self._pool_id:
            return CheckStatus.PASS, "", ""
        # 检查存储池是否存在
        result = self.dsl("exec_cli 'show storage_pool general' | horizontal_parser",
                          return_if={ft.FindException: [{"name": "error"}]})
        if result and len(result) == 1 and result[0].get("Name") == 'storag_pool_0':  # 已经存在存储池，不再创建
            self._pool_id = result[0].get("ID")
            return CheckStatus.PASS, "", ""
        if not result:  # 创建存储池
            if LicenseFeatureQueryUtil.hasHyperEncryptionFeature(EntityUtils.toOldDev(self.context.java_dev)):
                return self.create_encrypt_storage_pool()
            return self.create_storage_pool_no_license()
        return CheckStatus.NOT_PASS, "process.task.step.detail.storage_pool_name_error", ""

    def create_storage_pool_no_license(self):
        command = "exec_cli 'create storage_pool name={} disk_domain_id=0 raid_level={}'"
        result = self.dsl(command.format(STORAGE_POOL_NAME, self._raid_level))
        if 'successfully' in result:
            return CheckStatus.PASS, "", ""
        else:
            return CheckStatus.NOT_PASS, "process.task.step.detail.create_storage_pool_fail", result

    def create_encrypt_storage_pool(self):
        flag, error_key, msg = self.check_device_password_management_service()
        if flag is CheckStatus.NOT_PASS:
            return flag, error_key, msg
        command = "exec_cli 'create storage_pool name={} disk_domain_id=0'"
        result = self.dsl(command.format(STORAGE_POOL_NAME))
        if 'successfully' in result:
            return CheckStatus.PASS, "", ""
        else:
            return CheckStatus.NOT_PASS, "process.task.step.detail.create_storage_pool_fail", result

    def create_container_service(self):
        """
        创建容器服务
        """
        # 查询下容器服务是否已经拉起
        result = self.dsl("exec_cli 'show container_service general' | vertical_parser")
        if result[0].get("Deployed Node List") == self._ctrl_list and result[0].get("Undeployed Node List") == '--':
            if result[0].get("Enabled") == "On":
                return CheckStatus.PASS, "", ""
            else:
                return CheckStatus.NOT_PASS, "process.task.step.detail.container_service_not_normal", ""
        # 查询一下存储池id
        if not self._pool_id:
            result = self.dsl("exec_cli 'show storage_pool general' | horizontal_parser",
                              return_if={ft.FindException: [{"name": "error"}]})
            self._pool_id = result[0].get("ID")
        # 创建容器服务
        command = "exec_cli 'create container_service general node_list={ctrl_list} " \
                  "image_repository_pool_id={pool_id} application_pool_id_list={pool_id} " \
                  "compression_enabled=no dedup_enabled=no' | regex '\"(.*?)\"'"
        result = self.dsl(command.format(ctrl_list=self._ctrl_list, pool_id=self._pool_id))
        if result and "show task general" in result[0]:  # 后台在执行创建容器的服务
            command = "exec_cli '{}' | vertical_parser".format(result[0])
            while True:
                time.sleep(10)
                result = self.dsl(command, return_if={ft.FindException: [{"Status": "failed"}]})
                if not result:
                    return CheckStatus.NOT_PASS, "process.task.step.detail.create_container_service_fail", ""
                if result[0].get("Status") == 'executing' or result[0].get("Status") == 'wait':
                    continue
                if result[0].get("Status") == 'success':
                    return CheckStatus.PASS, "", ""
                return CheckStatus.NOT_PASS, "process.task.step.detail.create_container_service_fail", ""
        return CheckStatus.NOT_PASS, "process.task.step.detail.create_container_service_fail", ""

    def import_helm_char(self):
        """
        导入应用Helm Chart包
        """
        # 查询一下是否已经导入过
        command = "exec_cli 'show helm_chart general' | horizontal_parser"
        result = self.dsl(command)
        if result and self.check_import_software_package(result[0].get("Application")):
            return CheckStatus.PASS, "", ""
        # 导入包
        command = "exec_cli 'import helm_chart ip={ip} user=admin password={password} " \
                  "path={chart_path}'"
        self.logger.info("import helm_chart ip={} user=admin password=**** path={}".format(
            self.context.dev_node.ip, self._chart_path))
        master_ip = self.query_controller_0a_ip()
        if not master_ip:
            self.logger.error("import container_image fail, 0A controller ip address not exist!")
            return CheckStatus.NOT_PASS, "process.task.step.detail.import_image_0A_controller_ip_not_exist", ""
        result = self.dsl(command.format(ip=master_ip,
                                         password=self.context.dev_node.password,
                                         chart_path=self._chart_path), need_log=False)
        if 'successfully' in result:
            return CheckStatus.PASS, "", ""
        return CheckStatus.NOT_PASS, "process.task.step.detail.import_char_fail", result

    def import_container_app_image(self):
        """
        导入应用镜像软件包
        """
        # 查询一下是否已经导入过
        command = "exec_cli 'show container_image general' | horizontal_parser"
        result = self.dsl(command)
        if result and self.check_import_software_package(result[0].get("Application")):
            self._package_version = result[0].get("Version")
            return CheckStatus.PASS, "", ""
        # 导入包
        command = "exec_cli 'import container_image ip={ip} user=admin" \
                  " password={password} path={image_path}'"
        self.logger.info("import container_image ip={} user=admin password=**** path={}".format(
            self.context.dev_node.ip, self._image_path))
        master_ip = self.query_controller_0a_ip()
        if not master_ip:
            self.logger.error("import container_image fail, 0A controller ip address not exist!")
            return CheckStatus.NOT_PASS, "process.task.step.detail.import_image_timeout", ""
        result = self.dsl(command.format(ip=master_ip,
                                         password=self.context.dev_node.password,
                                         image_path=self._image_path), need_log=False, time_out=60 * 60)
        if 'successfully' in result:
            return CheckStatus.PASS, "", ""
        if "TOOLKIT_SEND_CMD_TIME_OUT" == result:
            return CheckStatus.NOT_PASS, "process.task.step.detail.import_image_timeout", ""
        return CheckStatus.NOT_PASS, "process.task.step.detail.import_image_fail", result

    def query_controller_0a_ip(self):
        command = "exec_cli 'show upgrade package' | horizontal_parser"
        result = self.dsl(command)
        self.logger.info("query controller origin information {}".format(result))
        if not result:
            return ""
        for item in result:
            if item.get("Name") == "0A":
                return item.get("IP")
        return ""

    def check_import_software_package(self, application):
        """
        判断软件包是否已导入，涉及Helm Chart包、镜像软件包，具体规则如下
        1、判断版本是否在1.3.RC1及之后（该版本前后导入标识发生变化）
        2、根据导入标识判断是否已导入：1.3.RC1版本之前为OceanProtect-DataProtect，1.3.RC1及之后为DataBackup
        :param application: 导入标识
        :return:
        """
        if PyTaskService.compare_version(self.dev_version, '1.3.RC1') >= 0:
            return application == 'DataBackup'
        return application == 'OceanProtect-DataProtect'

    def create_container_application(self):
        """
        创建容器应用
        """
        # 查询一下镜像包的版本号
        if not self._package_version:
            command = "exec_cli 'show container_image general' | horizontal_parser"
            result = self.dsl(command)
            self._package_version = result[0].get("Version")
        # 1.3.RC1版本之前为OceanProtect-DataProtect，1.3.RC1及之后为DataBackup
        application = 'DataBackup' \
            if PyTaskService.compare_version(self.dev_version, '1.3.RC1') >= 0 \
            else 'OceanProtect-DataProtect'
        # 创建应用，超时时间30min（与cli超时时间保持一致）
        command = "exec_cli 'create container_application general app={application} " \
                  "version={package_version} namespace=dpa name=dataprotect " \
                  "dynamic_config=\"timeout={time_out},global.deploy_type={deploy_type},global.replicas={ctrl_num}," \
                  "global.gaussdbpwd={gaussdbpwd}\"'"
        self.logger.info("create container_application general app={} version={} namespace=dpa name=dataprotect "
            "dynamic_config=\"timeout=1800,global.deploy_type={},global.replicas={},global.gaussdbpwd=****\"".format(
            DEPLOYED_TYPE.get(self.context.dev_node.model), self._package_version, self._ctrl_num,application ))
        result = self.dsl(command.format(deploy_type=DEPLOYED_TYPE.get(self.context.dev_node.model),
                                         package_version=self._package_version, ctrl_num=self._ctrl_num,
                                         time_out=str(30 * 60), application=application,
                                         gaussdbpwd=self._gauss_db_pwd), need_log=False, time_out=30 * 60)
        if 'successfully' in result:
            return CheckStatus.PASS, "", ""
        if "TOOLKIT_SEND_CMD_TIME_OUT" == result:  # 返回超时，通过查询状态来判断
            while True:
                time.sleep(15)
                result = self._query_application_status()
                if not result:
                    return CheckStatus.NOT_PASS, "process.task.step.detail.create_container_application_fail", "Unknown cause."
                if result and result.get("Status") and result.get("Status") == 'deployed':
                    return CheckStatus.PASS, "", ""
                if result and result.get("Status") and result.get("Status") == 'pending-install':
                    continue
                return CheckStatus.NOT_PASS, "process.task.step.detail.create_container_application_fail", \
                       result.get("Error", "The system is abnormal.")
        if "the application is being deployed" in result:
            return CheckStatus.NOT_PASS, "process.task.step.detail.app_doing", result
        return CheckStatus.NOT_PASS, "process.task.step.detail.create_container_application_fail", \
               self.parse_error_msg(result)

    def check_env(self):
        """
        检查初始化状态(检查应用状态、检查硬盘域、检查存储池三个动作合并)
        """
        # 检查应用状态
        return_value = self.check_container_service_status()
        if return_value[0] == CheckStatus.NOT_PASS:
            return return_value
        # 检查硬盘域
        return_value = self.check_disk_domain_name()
        if return_value[0] == CheckStatus.NOT_PASS:
            return return_value
        # 检查存储池
        return_value = self.check_storage_pool_name()
        return return_value

    def import_package(self):
        """
        导入软件包(导入chart和image合并)
        """
        # 导入chart包
        return_value = self.import_helm_char()
        if return_value[0] == CheckStatus.NOT_PASS:
            return return_value
        # 导入image包
        return_value = self.import_container_app_image()
        return return_value

    def parse_error_msg(self, msg):
        """
        解析回显中的错误信息（解析不到的时候，返回原始回显信息）

        :param msg: 回显信息
        :return: 提取的错误信息
        """
        try:
            temp_result = get_vertical_cli_ret(msg)
            if temp_result and temp_result[0].get("Error"):
                return temp_result[0].get("Error")
        except Exception:
            self.logger.info("parse error: {}".format(traceback.format_exc()))
        return msg

    def activate_container_service(self):
        """
        激活容器服务；如果容器已激活继续进行部署流程，如果未激活进行激活。
        :return:  容器服务是否激活成功  报错信息 “”
        """
        # 查询容器服务是否已经激活
        result = self.dsl("exec_cli 'show container_service general' | vertical_parser")

        # 如果容器已激活，直接返回通过，继续进行部署流程
        if result and result[0].get("Enabled") == "On":
            return CheckStatus.PASS, "", ""

        # 激活容器服务
        self.logger.info("start to activate the container service")
        conn = get_available_conn(self.context)
        result = conn.execCmdNoLog(
            "change container_service active enabled=on password={}".format(self.context.dev_node.password))
        if result and "Have you read warning message carefully?(y/n)" in result:
            result = conn.execCmd(YES)
        if result and "Are you sure you really want to perform the operation?(y/n)" in result:
            result = conn.execCmd(YES)
        conn.close()

        # 轮询容器服务是否激活
        if result and "show task general" in result:
            self.logger.info("Waiting for activation of the container service.")
            task_id = re.search(r"task_id=(\d+)", result)
            command = "show task general task_id={}".format(task_id.group(1) if task_id else 0)
            flag, msg = self.check_activate_container_service_step_status(command)
            return flag, msg, ""

        return CheckStatus.NOT_PASS, "process.task.step.detail.activate_container_service_fail", ""

    def switch_container_interface_model(self):
        """
        切换A控和B控0号槽位为容器卡模式
        @return: 切卡模式是否成功 错误信息 “”
        """
        # 如果软件版本低于1.5.0，则直接通过
        self.logger.info("dev_version: {}".format(self.dev_version))
        if PyTaskService.compare_version(self.dev_version, '1.5.0') < 0:
            return CheckStatus.PASS, "", ""

        # 获取需要切换的接口卡ID
        conn = get_available_conn(self.context)
        cli_ret = conn.execCmd("show interface_module")
        interface_infos = get_horizontal_cli_ret(cli_ret)
        enc_cli_ret = conn.execCmd("show enclosure")
        enc_infos = get_horizontal_cli_ret(enc_cli_ret)
        # 获取IOM0槽位正常的接口卡数量
        interface_iom0_ids = [info.get("ID") for info in interface_infos if
                              "IOM0" in info.get("ID") and info.get("Health Status") == "Normal" and info.get(
                                  "Running Status") == "Running"]
        ctrl_enc_info = [enc_info for enc_info in enc_infos if enc_info.get("ID").startswith("CTE")]
        self.logger.info("interface_iom0_ids: {}, ctrl_enc_info: {}".format(interface_iom0_ids, ctrl_enc_info))
        # 如果IOM0槽位正常接口卡数量 < 引擎数 * 2，则IOM0号槽位接口卡没插或不正常，不通过
        if len(interface_iom0_ids) < len(ctrl_enc_info) * 2:
            return CheckStatus.NOT_PASS, "process.task.step.detail.interface_num_not_enough", "/".join(
                interface_iom0_ids)
        interface_ids = [info.get("ID") for info in interface_infos if
                         "IOM0" in info.get("ID") and info.get("Usage Type").lower() != "container front end"]

        # 切换接口卡模式为容器卡模式
        switch_failed_ids = list()
        for interface_id in interface_ids:
            cmd = "change interface_module id={} usage_type=container_front_end".format(interface_id)
            cli_ret = conn.execCmd(cmd)
            if cli_ret and "Have you read danger alert message carefully?(y/n)" in cli_ret:
                cli_ret = conn.execCmd(YES)
            if cli_ret and "Are you sure you really want to perform the operation?(y/n)" in cli_ret:
                cli_ret = conn.execCmd(YES)
            if "Command executed successfully" not in cli_ret:
                switch_failed_ids.append(interface_id)
        self.logger.info("switch_failed_ids: {}".format(switch_failed_ids))
        conn.close()
        if not switch_failed_ids:
            return CheckStatus.PASS, "", ""
        return CheckStatus.NOT_PASS, "process.task.step.detail.switch_container_interface_model_fail", "/".join(
            switch_failed_ids)


    def retry(sleep=QUERY_INTERVAL, times=3):
        def retry_func(fun):
            def wrapper(*args):
                for i in range(times):
                    time.sleep(sleep)
                    flag = fun(*args)
                    if flag == PyTaskService.Param.CONTINUE:
                        continue
                    if flag == CheckStatus.NOT_PASS:
                        return flag, "process.task.step.detail.activate_container_service_fail"
                    if flag == CheckStatus.PASS:
                        return flag, ""
                    if flag == PyTaskService.Param.REBOOT_SUCCESS:
                        return PyTaskService.Param.CONTINUE, ""
                return CheckStatus.NOT_PASS, "process.task.step.detail.activate_container_service_fail"

            return wrapper

        return retry_func

    @retry(times=ACTIVATION_TIMES)
    def check_activate_container_service_step_status(self, command):
        """
        查询容器服务是否激活
        :param command: 查询命令
        :return: boolean, msg
        """
        try:
            time.sleep(QUERY_INTERVAL)
            conn = get_available_conn(self.context)
            cli_ret = conn.execCmd(command)
            if "upgrade:/>" in cli_ret:
                conn.reConnect()
                return PyTaskService.Param.CONTINUE
            result = get_vertical_cli_ret(cli_ret)
            if not result:
                return PyTaskService.Param.CONTINUE
            status = result[0].get("Status")
            if status == PyTaskService.Param.EXECUTING or status == PyTaskService.Param.WAIT:
                return PyTaskService.Param.CONTINUE
            if status == PyTaskService.Param.FAILED or status == PyTaskService.Param.FAIL:
                return CheckStatus.NOT_PASS
            if status == PyTaskService.Param.SUCCESS:
                return CheckStatus.PASS
        except JException, e:
            self.logger.error("device activate container service progress error. {}".format(traceback.format_exc()))
            if isinstance(e, TLVException) or isinstance(e, IsmException):
                self.logger.error("Execute upgrade command exception!!The errorId: {}".format(e.getErrorId()))
                if OM_OPERATE_FAIL_CODE == e.getErrorId():
                    time.sleep(3)
                    return PyTaskService.Param.CONTINUE
            flag, msg = self.check_dev_reboot()
            if flag == CheckStatus.NOT_PASS:
                return flag
            else:
                return PyTaskService.Param.CONTINUE

    @retry(times=RESTART_TIMES)
    def check_dev_reboot(self):
        """
        查询设备是否重启
        :return:  boolean 是否重启
        """
        try:
            time.sleep(QUERY_INTERVAL)
            conn = create_cli_conn(self.context, self.context.dev_node.ip)
            if not conn:
                return PyTaskService.Param.CONTINUE
            conn.close()
            self.logger.info("dev reboot success!")
            return PyTaskService.Param.REBOOT_SUCCESS
        except Exception, e:
            self.logger.error("dev reboot after connect error. {}".format(traceback.format_exc()))

    def check_device_license(self):
        """
        查询设备license
        :return: 设备是否有备份特性 报错信息 “”
        """
        flag = LicenseFeatureQueryUtil.hasDataBackupFeature(EntityUtils.toOldDev(self.context.java_dev))
        if not flag:
            return CheckStatus.NOT_PASS, "process.task.step.detail.check_device_license_fail", ""
        return CheckStatus.PASS, "", ""

    def check_device_password_management_service(self):
        """
        检测存储设备上是否开启密管服务，如果没有开启，提示用户开启密管服务，流程失败；如果开启，则继续流程
        :return:
        """
        result = self.dsl("exec_cli 'show key_service general' | vertical_parser")
        if not result or result[0].get("Service Type") == "Disabled":
            return CheckStatus.NOT_PASS, "process.task.step.check_device_password_management_service_fail", ""
        return CheckStatus.PASS, "", ""

    def create_disk_domain_by_disks(self):
        """
        通过指定的硬盘，创建硬盘域；备份一体机1.6及之后版本使用
        """
        if self._disk_domain_id:
            return CheckStatus.PASS, "", ""
        # 检查硬盘域是否存在
        result = self.dsl("exec_cli 'show disk_domain general' | horizontal_parser",
                          return_if={ft.FindException: [{"name": "error"}]})
        # 已经存在硬盘域，不再创建
        if result and len(result) == 1 and result[0].get("Name") == DISK_DOMAIN_NAME:
            self._disk_domain_id = result[0].get("ID")
            return CheckStatus.PASS, "", ""
        if not result:  # 还没有硬盘域，创建硬盘域
            if LicenseFeatureQueryUtil.hasHyperEncryptionFeature(EntityUtils.toOldDev(self.context.java_dev)):
                return self.create_disk_domain_by_disks_has_encryption()
            return self.create_disk_domain_by_disks_no_encryption()
        return CheckStatus.NOT_PASS, "process.task.step.detail.disk_domain_name_error", ""

    def create_disk_domain_by_disks_no_encryption(self):
        """
        没有加密特性，创建硬盘域
        """
        cmd = "exec_cli 'create disk_domain name={disk_domain_name} disk_list={disk_list} raid_level={raid_level}'"
        result = self.dsl(
            cmd.format(disk_domain_name=DISK_DOMAIN_NAME, disk_list=self._all_disks_ids, raid_level=self._raid_level))
        if 'successfully' in result:
            return CheckStatus.PASS, "", ""
        else:
            return CheckStatus.NOT_PASS, "process.task.step.detail.create_disk_domain_fail", result

    def create_disk_domain_by_disks_has_encryption(self):
        """
        带加密特性创建硬盘域
        """
        flag, error_key, msg = self.check_device_password_management_service()
        if flag is CheckStatus.NOT_PASS:
            return flag, error_key, msg
        cmd = "exec_cli 'create disk_domain name={disk_domain_name} disk_list={disk_list} raid_level={raid_level} " \
              "disk_encryption_switch=on disk_encryption_algorithm=AES'"
        result = self.dsl(
            cmd.format(disk_domain_name=DISK_DOMAIN_NAME, disk_list=self._all_disks_ids, raid_level=self._raid_level))
        if 'successfully' in result:
            return CheckStatus.PASS, "", ""
        else:
            return CheckStatus.NOT_PASS, "process.task.step.detail.create_disk_domain_fail", result

    class Param():
        EXECUTING = "executing"
        FAILED = "failed"
        SUCCESS = "success"
        CONTINUE = "continue"
        REBOOT_SUCCESS = "reboot_success"
        WAIT = "wait"
        FAIL = "fail"
