# -*- coding: UTF-8 -*-
from cbb.frame.checkitem.base_dsl_check import BaseCheckItem, CheckStatus
from cbb.frame.adapter import replace_adapter
from cbb.frame.base import data_parser
from cbb.frame.dsl import fault_mode as ft
from cbb.frame.dsl.adapter import NEED_NEW_CTRL_CONN

INVOLVE_VERSION_DICT = {
    "V300R003C10SPC100": "V300R003C10SPH118",
    "V300R003C20SPC200": "V300R003C20SPH217",
    "V300R005C00SPC300": "V300R005C00SPH308",
    "V300R006C00SPC100": "V300R006C00SPH111",
    "V300R006C10SPC100": "V300R006C10SPH105",
    "V500R007C00SPC100": "V500R007C00SPH105",
    "V300R006C20": "V300R006C20SPH006",
    "V500R007C10": "V500R007C10SPH006",
    "V500R007C10SPC100": "V500R007C10SPH115",
    "V300R006C20SPC100": "V300R006C20SPH115"
}


class CheckItem(BaseCheckItem):

    def __init__(self, context):
        super(CheckItem, self).__init__(context)
        self.dev_version = ""
        self.patch_version = ""
        self.cache_obj_num = 0
        self.tool_type = context.get("tool_type")
        self.context[NEED_NEW_CTRL_CONN] = True

    def execute(self):
        self.logger.info("cache_obj_num_check.py start.")
        flag, self.dev_version, self.patch_version, cli_ret = \
            replace_adapter.get_system_version_with_ret(self.context)
        self.add_origin_info(cli_ret)
        if not flag:
            self.logger.info("query version error. nocheck")
            return CheckStatus.NOCHECK, self.get_msg("query.result.abnormal")

        if self.dev_version not in INVOLVE_VERSION_DICT:
            self.logger.info("version is not involve, pass")
            return CheckStatus.PASS, ""

        # V300R003C20SPH220补丁问题规避
        if self.patch_version != "V300R003C20SPH220":
            all_ret = self.dsl("exec_on_all {}", self.cache_show_ckg,
                               need_node_id=True)
            self.logger.info("all_ret={}".format(all_ret))
            result, err_msg = CheckStatus.merge_all_ctrl_result(self.lang,
                                                                all_ret)
            if result != CheckStatus.UNKNOWN:
                self.logger.info("can run cache show ckg, check finished.")
                return result, err_msg

        need_patch_ver = INVOLVE_VERSION_DICT.get(self.dev_version)
        if not self.patch_version \
                or self.patch_version < need_patch_ver:
            self.logger.info("not install patch, warning")
            if self.tool_type == "fru":
                return CheckStatus.PASS, ""
            else:
                return CheckStatus.WARNING, self.get_msg(
                    "cache.obj.check.no.patch", need_patch_ver)
        if self.is_exist_row_lun():
            self.logger.info("exist row lun, pass.")
            return CheckStatus.PASS, ""

        self.cache_obj_num = self.query_cache_obj_sum()

        all_ctrl_result = self.dsl("exec_on_all {}",
                                   self.check_total_obj_num,
                                   is_continue=False, need_node_id=True)
        status, err_msg = CheckStatus.merge_all_ctrl_result(self.lang,
                                                            all_ctrl_result)
        return status, err_msg

    def cache_show_ckg(self):
        ret = self.dsl(
            "exec_diagnose 'cache show ckg' | regex 'remain:(?P<remain>\d+), "
            "newPool:(?P<newPool>\d+), samePool:(?P<samePool>\d+), normal:"
            "(?P<normal>\d+)'",
            return_if={ft.FindException: ""})
        if not ret:
            return CheckStatus.UNKNOWN, ""
        if (int(ret.get("remain")) + int(ret.get("newPool")) + int(
                ret.get("samePool"))) > 0:
            return CheckStatus.NOTPASS, self.get_msg("cache.obj.check.not.pass")
        return CheckStatus.PASS, ""

    def is_exist_row_lun(self):
        row_lun_ids = self.dsl(
            "exec_developer 'show lun dedup_compress |filterColumn include "
            "columnList=ID' | horizontal_parser",
            return_if={
                ft.CmdNoSupport: {}})
        if row_lun_ids:
            return True
        row_lun_ids = self.dsl(
            "exec_developer 'show dedup_compress_lun general |filterColumn "
            "include columnList=ID' | horizontal_parser",
            return_if={
                ft.CmdNoSupport: {}})
        if row_lun_ids:
            return True
        return False

    def query_cache_obj_sum(self):
        fs_ids = self.dsl(
            "exec_developer 'show file_system general |filterColumn "
            "include columnList=ID' | horizontal_parser",
            return_if={ft.CmdNoSupport: {}})
        lun_ids = self.dsl(
            "exec_developer 'show lun general |filterColumn include "
            "columnList=ID' | horizontal_parser")
        snap_ids = self.dsl(
            "exec_developer 'show snapshot general |filterColumn include "
            "columnList=ID' | horizontal_parser")

        cache_num = len(lun_ids) + len(snap_ids) + len(fs_ids)
        self.logger.info("cache num is {}".format(cache_num))
        return cache_num

    def check_total_obj_num(self):
        cli_ret = self.dsl("exec_diagnose 'cache show obj all'")
        all_cache_num = self.dsl(
            "echo {} | regex 'Total obj num\((?P<obj_num>\d+)\)' "
            "| get_key('obj_num')'", cli_ret, return_if={ft.FindException: "0"})

        cache_dict = data_parser.horizontal_parser_for_diagnose(
            cli_ret, ["volumeId"])

        no_cache_obj = [cache for cache in cache_dict
                        if int(cache.get("volumeId"), 16) >=
                        int('0x2000000000', 16)]
        self.logger.info("no cache obj={}".format(no_cache_obj))

        if int(all_cache_num) - len(no_cache_obj) > 2 * self.cache_obj_num:
            self.logger.info("cache obj abnormal, not pass.")
            return CheckStatus.NOTPASS, self.get_msg(
                "cache.obj.check.not.pass")
        return CheckStatus.PASS, ""
