# -*- coding: UTF-8 -*-
import time
import traceback
# noinspection PyUnresolvedReferences
import java.lang.Exception as JException
from cbb.frame.base import baseUtil
from cbb.frame.cli import cliUtil
from cbb.frame.context import contextUtil
from cbb.frame.cli.execute_on_all_controllers import (
    ExecuteOnAllControllers,
    ExeOnAllCtrlContext,
    ResultType,
    FuncResult
)


def sph7_nginx_service_check(context, finished=False):
    if "nginx_service" not in context:
        context["nginx_service"] = NginxCheckAndRepair(context)
    logger = context.get("logger")
    ssh = context.get("ssh")
    nginx_service = context.get("nginx_service")
    if not nginx_service.is_need_check():
        # 是补丁完成的场景，需要重新建立SSH连接
        if finished:
            reconnect_ssh(ssh, logger)
        # 非SPH7，不再执行
        return True, ""

    if not nginx_service.is_patch_install_finish():
        # 补丁安装未完成，不再执行
        return True, ""

    if nginx_service.has_run:
        # 已经执行过修复，不再执行
        return True, ""

    # 补丁安装完成后检查nginx服务状态，有问题则启动修复
    nginx_service.has_run = True
    reconnect_ssh(ssh, logger)
    retry_times = 3
    while retry_times > 0:
        retry_times -= 1
        try:
            nginx_service.failed_ctrls = []
            flag, err_msg = check_and_repair_nginx_on_all_controllers(context)
            if flag:
                return True, ""
        except (Exception, JException) as e:
            logger.error("nginx service check error: e={},trace={}"
                         .format(e, traceback.format_exc()))
            reconnect_ssh(ssh, logger)
    return False, nginx_service.get_err_msg()


def reconnect_ssh(ssh, logger):
    try:
        logger.info("reconnect_ssh...")
        ssh.reConnect()
    except JException:
        logger.error("build ssh connection error")


def check_and_repair_nginx_on_all_controllers(context):
    """
    所有控制器的nginx服务的检查与修复
    :return:
    """
    logger = context.get("logger")
    nginx_service = context.get("nginx_service")
    nginx_service.init_manage_ip_and_node_relation()
    engine_ctrl_dict = nginx_service.get_engine_ctrl_dict()
    node_id_and_ip_dict = nginx_service.get_node_id_and_ip_dict()
    exe_context = ExeOnAllCtrlContext(context)
    exe_context.set_target_ctrl_dict(engine_ctrl_dict)
    context["nginx_service"] = nginx_service
    context["ctrl_id2ip_dict"] = node_id_and_ip_dict
    result = _execute_on_all_controllers(exe_context)
    logger.info("result.result_type={}".format(result.result_type))
    if result.result_type != ResultType.SUCCESS:
        err_msg = nginx_service.get_err_msg()
        return False, err_msg
    return True, ""


@ExecuteOnAllControllers
def _execute_on_all_controllers(exe_context):
    """
    单个控制器被装饰的函数
    :param exe_context: ExeOnAllCtrlContext
    :return: FuncResult
    """
    nginx_service = exe_context.original_context.get("nginx_service")
    logger = exe_context.original_context.get("logger")
    ctrl_id2ip_dict = exe_context.original_context.get("ctrl_id2ip_dict")
    logger.info("exe_context.cur_ctrl_id {}".format(exe_context.cur_ctrl_id))
    cur_ctrl_ip = ctrl_id2ip_dict.get(exe_context.cur_ctrl_id)
    if not cur_ctrl_ip:
        logger.info("manager ip is not exist, no need check")
        return FuncResult(ResultType.SUCCESS, "", "")
    if not nginx_service.execute(cur_ctrl_ip):
        logger.info("nginx service repair failed")
        return FuncResult(ResultType.FAILED, "", "")
    return FuncResult(ResultType.SUCCESS, "", "")


class NginxCheckAndRepair:
    def __init__(self, context):
        self.context = context
        self.logger = context.get("logger")
        self.cli = context.get("ssh")
        self.lang = contextUtil.getLang(context)
        self.engine_ctrl_dict = {}
        self.ctrl_ip_node_dict = {}
        self.failed_ctrls = []
        self.cur_ctrl_id = ""
        self.has_run = False
        # 是否需要修复,巡检工具不需要做修复动作
        self.need_repair = True
        self.all_cli_ret = []
        self.not_pass_ctrls = []

    def set_need_repair(self, flag):
        """
        是否需要修复的控制开关，巡检不需要修复。
        :param flag:
        :return:
        """
        self.need_repair = flag

    def is_need_check(self):
        pkg_version = self.context.get("pkgVersion")
        dev_version = str(self.context.get('dev').getProductVersion())
        self.logger.info("pkg version {}, dev_version:{}"
                         .format(pkg_version, dev_version))
        if "6.0.0" in dev_version \
                and (pkg_version == "SPH7" or pkg_version == "SPH8"):
            self.logger.info("need check nginx service")
            return True
        self.logger.info("not need check nginx service")
        return False

    def is_patch_install_finish(self):
        retry_times = 3
        while retry_times > 0:
            retry_times -= 1
            try:
                flag, product_version, _ \
                    = cliUtil.getProductVersion(self.cli, self.lang)
                sph_version = product_version.strip().split(".")[-1]
                self.logger.info("query sph version {}".format(sph_version))
                if flag and sph_version == "SPH7":
                    return True
                return False
            except JException:
                try:
                    self.logger.info("reconnect_ssh....")
                    self.cli.reConnect()
                except JException:
                    self.logger.error("build ssh connection error")
                self.logger.error("get product version error.")
        return False

    def get_err_msg(self):
        if self.failed_ctrls:
            return baseUtil.getPyResource(
                self.lang,
                "nginx.service.repair.failed",
                ",".join(self.failed_ctrls)
            )
        return baseUtil.getPyResource(
            self.lang,
            "nginx.service.check.error"
        )

    def execute(self, cur_ctrl_ip):
        """
        在一个控制器上执行nginx服务的检查与修复
        :return: 修复成功：True， 修复失败：False
        """
        try:
            self.logger.info("begin to execute check and repair {}"
                             .format(cur_ctrl_ip))
            # 巡检场景，不需要修复
            if not self.need_repair:
                cur_node_info = self.ctrl_ip_node_dict.get(cur_ctrl_ip)
                self.cur_ctrl_id = cur_node_info.get("ctrl_id")
                if self.check_nginx_for_inspect(cur_ctrl_ip):
                    return True
                return False

            if self.check_nginx(cur_ctrl_ip):
                self.logger.info("nginx service is normal")
                return True

            self.logger.info("nginx service is abnormal, need repair")
            cur_node_info = self.ctrl_ip_node_dict.get(cur_ctrl_ip)
            self.cur_ctrl_id = cur_node_info.get("ctrl_id")
            manage_port_mac = cur_node_info.get("port_mac")
            self.logger.info("the manage port mac {}".format(manage_port_mac))
            if not manage_port_mac:
                self.failed_ctrls.append(self.cur_ctrl_id)
                return False
            if not self.repair_nginx(manage_port_mac):
                return False
            return True
        except (Exception, JException) as e:
            self.logger.error("execute check and repair error.e={}, trace={}"
                              .format(e, traceback.format_exc()))
            self.failed_ctrls.append(self.cur_ctrl_id)
            return False

    def init_manage_ip_and_node_relation(self):
        """
        初始化所有关系数据信息（以管理IP作为Key）
        获取需要开启nginx服务的控制器：该控制器绑定了管理IP
        show port 查询所有管理口， 过滤出配置了IP的port，记录IP与port mac关系
        show upgrade package 查询ip与ctroller id的对应关系
        将controller id 转换为engine + node id的关系
        :return: {engine:[controller ids]} 如：{0:[0,1],2:[2,3]}
        """
        target_ctrl_dict = {}
        flag, _, _, ports = cliUtil.getManagementPortInfo(self.cli, self.lang)
        if flag is not True:
            raise RepairException("query.info.error")
        flag, ctrl_ips = cliUtil.getCtrlIps(self.cli, self.logger, self.lang)
        if flag is not True:
            raise RepairException("query.info.error")
        self.logger.info("##all ctrl ips {}".format(ctrl_ips))
        ctrl_num = self.get_ctrl_num_one_engine()
        for port in ports:
            ipv4 = str(port.get("IPv4 Address"))
            ctrl_id = self.find_ctrl_by_ip(ctrl_ips, ipv4)
            if ipv4 and ctrl_id:
                target_ctrl_dict[ipv4] \
                    = self._get_node_info(ctrl_id, ctrl_num, port.get("MAC"))
                continue

            ipv6 = str(port.get("IPv6 Address"))
            ctrl_id = self.find_ctrl_by_ip(ctrl_ips, ipv6)
            if ipv6 and ctrl_id:
                target_ctrl_dict[ipv6] \
                    = self._get_node_info(ctrl_id, ctrl_num, port.get("MAC"))
        self.logger.info("##ctrl ip and node relation {}"
                         .format(target_ctrl_dict))
        self.ctrl_ip_node_dict = target_ctrl_dict

    def init_manage_ip_and_node_relation_for_inspection(self):
        """
        初始化所有关系数据信息（以管理IP作为Key）
        获取需要开启nginx服务的控制器：该控制器绑定了管理IP
        show upgrade package 查询ip与ctroller id的对应关系
        将controller id 转换为engine + node id的关系
        :return: {engine:[controller ids]} 如：{0:[0,1],2:[2,3]}
        """
        target_ctrl_dict = {}
        flag, cli_ret, err_msg, ctrl_ips = cliUtil.get_ctrl_ip_name_dict(
            self.cli, self.lang
        )
        if flag is not True:
            raise RepairException("query.info.error")
        self.logger.info("##all ctrl ips {}".format(ctrl_ips))
        ctrl_num = self.get_ctrl_num_one_engine()
        for ctrl_ip, ctrl_id in ctrl_ips.iteritems():
            if ctrl_ip and ctrl_id:
                target_ctrl_dict[ctrl_ip] \
                    = self._get_node_info(ctrl_id, ctrl_num, "")
                continue

        self.logger.info("##ctrl ip and node relation {}"
                         .format(target_ctrl_dict))
        self.ctrl_ip_node_dict = target_ctrl_dict

    def _get_node_info(self, ctrl_id, ctrl_num, mac):
        node_info = dict()
        node_info["ctrl_id"] = ctrl_id
        engine_id, node_id \
            = self.get_node_id_by_ctrl_id(ctrl_id, ctrl_num)
        node_info["engine_id"] = engine_id
        node_info["node_id"] = node_id
        node_info["port_mac"] = mac
        return node_info

    def get_engine_ctrl_dict(self):
        """
        记录需要执行检查的引擎和控制器信息（有管理IP的节点）
        :return:
        """
        engine_ctrl_dict = {}
        for node in self.ctrl_ip_node_dict.values():
            engine_id = str(node.get("engine_id"))
            node_id = str(node.get("node_id"))
            if engine_ctrl_dict.get(engine_id) is None:
                engine_ctrl_dict[engine_id] = []
            engine_ctrl_dict[engine_id].append(node_id)
        self.logger.info("##need to check engine and ctrl {}"
                         .format(engine_ctrl_dict))
        return engine_ctrl_dict

    def get_node_id_and_ip_dict(self):
        """
        记录节点的ID与管理IP的关系
        :return:
        """
        ctrl_id_ip_dict = {}
        for ip, node in self.ctrl_ip_node_dict.items():
            ctrl_id_ip_dict[str(node.get("node_id"))] = ip
        self.logger.info("##node id and ip relation {}"
                         .format(ctrl_id_ip_dict))
        return ctrl_id_ip_dict

    @staticmethod
    def find_ctrl_by_ip(ctrl_ips, ip):
        """
        通过管理IP找到对应的控制器ID
        :param ctrl_ips: 所有的管理IP与控制器ID的关系
        :param ip: 管理IP
        :return: 控制器ID
        """
        for ctrl_ip in ctrl_ips:
            if ctrl_ip[1] == ip:
                return ctrl_ip[0]
            if ctrl_ip[2] == ip:
                return ctrl_ip[0]
        return None

    @staticmethod
    def get_node_id_by_ctrl_id(ctrl_id, ctrl_num):
        """
        将控制器ID，转换为节点ID
        :param ctrl_id: 控制器ID
        :param ctrl_num: 单引擎控制器数
        :return: 节点ID
        """
        engine_id = int(ctrl_id[:-1])
        ctrl_key = ctrl_id[-1]
        ctrl_node_dict = {"A": 0, "B": 1, "C": 2, "D": 3}
        return engine_id, engine_id * ctrl_num + ctrl_node_dict.get(ctrl_key)

    def get_ctrl_num_one_engine(self):
        """
        获取单引擎控制器数量
        :return:
        """
        flag, cli_ret, err_msg, node_tuples \
            = cliUtil.getControllerEngineTopography(self.cli, self.lang)
        ctrl_num = len(node_tuples[2].get("0"))
        return ctrl_num

    def check_nginx(self, cur_ctrl_ip):
        """
        判断当前控制器是否存在DM未拉起的问题：
        1、进入本控小系统下执行：netstat -tlnp命令，查看当前控制器IP是否有对外
        提供8088 端口；
        2、如果没有，则循环三次，每次等10s。
        :return:False: 发现问题，返回mac 地址；True:未发现问题，返回空;
                False: 查询失败，返回空。
        """
        self.logger.info("cur_ctrl_ip {}".format(cur_ctrl_ip))
        cmd = "netstat -tlnp"
        count = -1
        try:
            while True:
                count += 1
                if count >= 3:
                    self.logger.error("nginx query three times")
                    return False
                flag, cli_ret, _ = cliUtil.excuteCmdInMinisystemModel(
                    self.cli, cmd, self.lang)
                self.all_cli_ret.append(cli_ret)
                if flag is not True:
                    self.logger.error("query netstat error.{}".format(cmd))
                    continue

                cli_ret_list = cli_ret.encode("utf8").splitlines()
                for line in cli_ret_list:
                    if cur_ctrl_ip + ":8088" in line:
                        self.logger.info("nginx query succ.")
                        return True
                time.sleep(10)
        except Exception as ex:
            self.logger.error(ex)
            return False
        finally:
            cliUtil.enterCliModeFromSomeModel(self.cli, self.lang)

    def check_nginx_for_inspect(self, cur_ctrl_ip):
        """
        判断当前控制器是否存在DM未拉起的问题：
        1、进入本控小系统下执行：netstat -tlnp命令，查看当前控制器IP是否有对外
        提供8088 端口；
        :return:True:未发现问题，返回空;
                False: 查询失败，返回空。
        """
        self.logger.info("cur_ctrl_ip {}".format(cur_ctrl_ip))
        cmd = "netstat -tlnp"

        try:
            flag, cli_ret, _ = cliUtil.excuteCmdInMinisystemModel(
                self.cli, cmd, self.lang)
            self.all_cli_ret.append(
                "\nController {}:\n{}".format(self.cur_ctrl_id, cli_ret)
            )
            if not cli_ret:
                self.failed_ctrls.append(self.cur_ctrl_id)
                return False

            if flag is not True:
                self.failed_ctrls.append(self.cur_ctrl_id)
                self.logger.error("query netstat error.{}".format(cmd))
                return False

            if cur_ctrl_ip + ":8088" in cli_ret and \
                    "127.0.0.1:8088" in cli_ret:
                self.logger.info("nginx query succ.")
                return True

            self.not_pass_ctrls.append(self.cur_ctrl_id)
            return False
        except Exception as ex:
            self.failed_ctrls.append(self.cur_ctrl_id)
            self.logger.error(ex)
            return False
        finally:
            cliUtil.enterCliModeFromSomeModel(self.cli, self.lang)

    def repair_nginx(self, mac):
        repair_service = RepairingBoundIPBug(self.context, mac)
        flag, err_code = repair_service.repairing()
        if flag:
            self.logger.info("##repair nginx success")
            return True
        self.failed_ctrls.append(self.cur_ctrl_id)
        self.logger.info("##repair nginx failed")
        return False


class RepairErrCode(object):
    LINK_UP_FAILED = "1"
    REPAIR_FAILED = "2"
    RESTORE_IP_FAILED = "3"


class RepairException(Exception):
    def __init__(self, err_msg, err_code=RepairErrCode.REPAIR_FAILED):
        self.err_msg = err_msg
        self.err_code = err_code


class RepairingBoundIPBug(object):
    def __init__(self, context, mac):
        self._cli = contextUtil.getCliCommon(context)
        self._logger = contextUtil.getLogger(context)
        self._lan = contextUtil.getLang(context)
        self._mac = mac
        self._alter_dest_ip = None
        self._original_ip = None
        self._Maintenance_Port_ID = None
        self.subnet_mask = None

    def repairing(self):
        """
        修复
        :return: 修复结果(True/False)
        """
        try:
            # 维护口是否为link down
            if not self._is_maintenance_port_link_down():
                return False, RepairErrCode.LINK_UP_FAILED
            # 获取维护口子网掩码
            self._get_maintenance_port_subnet_mask()
            # 修改维护口ip
            self._alter_maintenance_port_ip(self._alter_dest_ip)
            # 等待15秒，触发绑定ip
            time.sleep(15)
            # 还原维护口ip
            self._alter_maintenance_port_ip(self._original_ip)
            return True, None
        except RepairException as RE:
            self._logger.error("repair deal exception:{}".format(RE.err_msg))
            return False, RE.err_code
        except (Exception, JException) as exception:
            self._logger.error("repair exception:{}".format(str(exception)))
            return False, RepairErrCode.REPAIR_FAILED
        finally:
            cliUtil.enterCliModeFromSomeModel(self._cli, self._lan)

    def _is_maintenance_port_link_down(self):
        """
        该维护口是否是link down状态
        :return: True/False
        """
        cli_ret_lines_list = self._get_Maintenance_Port_info()
        for cli_ret_line in cli_ret_lines_list:
            if cli_ret_line.get("MAC") == self._mac:
                running_status = cli_ret_line.get("Running Status")
                self._Maintenance_Port_ID = cli_ret_line.get("ID")
                self._original_ip = cli_ret_line.get("IPv4 Address")
                self._set_alter_dest_ip(cli_ret_lines_list)
                self._logger.info(
                    "The MAC[{}] to Maintenance port, Running Status[{}], "
                    "Port ID[{}].".format(
                        self._mac, running_status, self._Maintenance_Port_ID))
                return running_status == "Link Down"
        raise RepairException("Not found the Maintenance port to the MAC")

    def _get_maintenance_port_subnet_mask(self):
        """
        获取该维护口所在子网掩码
        """
        cmd = "show port ip eth_port_id=" + self._Maintenance_Port_ID
        flag, cli_ret, _ = cliUtil.excuteCmdInCliMode(
            self._cli, cmd, True, self._lan)
        if not flag:
            raise RepairException("get maintenance port exception.")

        cli_ret_lines_list = cliUtil.getVerticalCliRet(cli_ret)
        if len(cli_ret_lines_list) == 0:
            raise RepairException("parse subnet mask exception.")
        for cli_ret_line in cli_ret_lines_list:
            if "Subnet Mask" in cli_ret_line:
                self.subnet_mask = cli_ret_line.get("Subnet Mask").strip()
                return
        raise RepairException("not found the Subnet Mask.")

    def _set_alter_dest_ip(self, maintenance_port_cli_ret_lines_list):
        """
        设置修改的目标ip
        :param maintenance_port_cli_ret_lines_list: 所有维护口信息
        """
        ip_last_bit_list = list()
        all_maintenance_ips = []
        for cli_ret_line in maintenance_port_cli_ret_lines_list:
            ipv4 = cli_ret_line.get("IPv4 Address")
            self._logger.error("ipv4{}".format(ipv4))
            line_ip_last_bit = self._get_ipv4_last_bit(ipv4)
            ip_last_bit_list.append(line_ip_last_bit)
            all_maintenance_ips.append(ipv4)
        self._logger.info("ip last bit list: {}".format(ip_last_bit_list))
        if len(set(all_maintenance_ips)) != len(all_maintenance_ips):
            raise RepairException("maintenance ip repeat, not to repair.")

        # 维护IP最后一段的范围(11, 254)
        dest_ip_last_bit = 11
        for ip in range(11, 254):
            if ip not in ip_last_bit_list:
                dest_ip_last_bit = ip
                break
        last_point_index = self._original_ip.rindex(".")
        self._alter_dest_ip = self._original_ip[:last_point_index + 1] \
            + str(dest_ip_last_bit)
        self._logger.info("alter dest ip: {}".format(self._alter_dest_ip))

    def _get_ipv4_last_bit(self, ip):
        """
        获取ipv4最后一位
        :param ip: ipv4
        :return: 最后一位的值
        """
        ip_bits = ip.split(".")
        if len(ip_bits) != 4:
            raise RepairException("Unnormal ip adress: {}".format(ip))
        return int(ip_bits[-1].strip())

    def _alter_maintenance_port_ip(self, ip):
        """
        修改维护口ip
        :param ip: 目标ipv4
        """
        cmd = "change system maintenance_ip " \
              "eth_port_id={} ip={} mask={}".format(self._Maintenance_Port_ID,
                                                    ip, self.subnet_mask)
        flag, cli_ret, _ = cliUtil.enterDeveloperMode(self._cli, self._lan)
        if not flag:
            raise RepairException("enter developer mode failed.")
        flag, cli_ret, _ = cliUtil.excuteCmdInCliMode(
            self._cli, cmd, True, self._lan)
        if not flag:
            raise RepairException("alter maintenance port ip exception.")

        # 需要确认直接输入y
        cnt = 2
        while "y/n" in cli_ret and cnt > 0:
            cli_ret = self._cli.execCmd("y")
            cnt -= 1

        if not cliUtil.queryResultWithNoRecord(cli_ret):
            if ip == self._original_ip:
                raise RepairException("exc change maintenance port ip failed.",
                                      RepairErrCode.RESTORE_IP_FAILED)
            raise RepairException("excute change maintenance port ip failed.")

        if not self._is_alter_to_dest_ip(ip):
            raise RepairException("change maintenance port ip failed.")

    def _is_alter_to_dest_ip(self, ip):
        """
        修改是否生效
        :param ip:  修改目标ip
        :return: True/False
        """
        cli_ret_lines_list = self._get_Maintenance_Port_info()
        for cli_ret_line in cli_ret_lines_list:
            port_ID = cli_ret_line.get("ID")
            ipv4 = cli_ret_line.get("IPv4 Address")
            if port_ID == self._Maintenance_Port_ID:
                return ipv4 == ip
        self._logger.error("can't find the Maintenance Port.")
        return False

    def _get_Maintenance_Port_info(self):
        """
        获取所有维护口信息
        :return: 维护口信息
        """
        cmd = "show port general physical_type=ETH logic_type=Maintenance_Port"
        flag, cli_ret, err_msg = cliUtil.excuteCmdInCliMode(
            self._cli, cmd, True, self._lan)
        if not flag or cliUtil.queryResultWithNoRecord(cli_ret):
            raise RepairException("show Maintenance Port msg exception.")

        cli_ret_lines_list = cliUtil.getHorizontalCliRet(cli_ret)
        if len(cli_ret_lines_list) == 0:
            raise RepairException("parse Maintenance Port cli ret exception.")
        return cli_ret_lines_list
