# -*- coding: UTF-8 -*-
# Copyright (c) Huawei Technologies Co., Ltd. 2019-2021. All rights reserved.

import time
from cbb.frame.checkitem.base_dsl_check import BaseCheckItem, CheckStatus
from cbb.frame.adapter import replace_adapter
from cbb.frame.dsl.adapter import NEED_NEW_CTRL_CONN
from cbb.frame.cli.cliUtil import enterCliModeFromSomeModel
from cbb.frame.dsl import fault_mode as ft


class PoolBalanceTaskCheck(BaseCheckItem):
    """
    检查硬盘域后台均衡任务BUGLIST
    """
    DISPLAY_TIMES = 120  # 进度显示时间
    CHECK_BALANCE_TASK_TIMES = 60
    INTERVAL_TIME = 1
    RISK_VERSION = [
        "V300R001C00SPC100",
        "V300R001C01SPC100",
        "V300R001C21SPC100",
        "V300R001C30SPC100",
        "V300R002C00SPC100",
        "V300R002C10SPC100",
        "V300R002C20",
        "6.0.0",
        "6.0.1",
        "6.1.0",
    ]

    def __init__(self, tool_context):
        super(PoolBalanceTaskCheck, self).__init__(tool_context)
        self.dev_version = ""
        self.patch_version = ""
        self.context[NEED_NEW_CTRL_CONN] = False

    def is_risk_version(self):
        """
        只有Dorado涉及该问题，在xml里面限制
        :return:
        """
        (
            flag,
            self.dev_version,
            self.patch_version,
            ret
        ) = replace_adapter.get_system_version_with_ret(self.context)
        self.add_origin_info(ret)
        return self.dev_version in PoolBalanceTaskCheck.RISK_VERSION

    def get_domain_ids(self):
        domain_info = \
            self.dsl("exec_cli 'show disk_domain general "
                     "|filterColumn include columnList=ID' "
                     "| horizontal_parser",)
        return [d.get("ID") for d in domain_info]

    def get_ctrl_ids(self):
        ctrl_info = \
            self.dsl("exec_cli 'show controller general"
                     "|filterColumn include columnList=Controller' "
                     "| vertical_parser_list",)
        return [d.get("Controller") for d in ctrl_info]

    def is_hang_task(self, task_info):
        for task in task_info:
            step_name = task.get("Step Name")
            not_back = task.get("Not Back")
            if ("REB_DEL_DISKTASK" in step_name.upper()
                and not_back != "0") \
                    or ("REB_DEL_REBTASK" in step_name.upper()
                        and not_back != "0"):
                return False
        return True

    def check_single_task(self, domain_id, ctrl_id):
        cmd = 'show zone_rebuild info info_type=task_count ' \
              'pool_id={} controller={}'.format(domain_id, ctrl_id)

        count = 0
        while count < PoolBalanceTaskCheck.CHECK_BALANCE_TASK_TIMES:
            task_cmd = "exec_developer '{}' | horizontal_parser".format(cmd)
            # 兼容未跨引擎建硬盘域的控制器
            no_obj_rule = {ft.FindStr("Error:"): []}
            task_info = self.dsl(task_cmd, return_if=no_obj_rule)
            if self.is_hang_task(task_info):
                return True
            count += 1
            time.sleep(PoolBalanceTaskCheck.INTERVAL_TIME)
            continue

        return False

    def check_task(self, ctrl_ids):
        """
                检查所有控制器是否均衡任务挂起
                :return:
        """
        if not self.is_risk_version():
            return CheckStatus.PASS, ""

        domain_ids = self.get_domain_ids()
        self.logger.info("domain_ids:{}".format(domain_ids))
        if not domain_ids:
            return CheckStatus.PASS, ""

        for domain_id in domain_ids:
            for ctrl in ctrl_ids:
                if not self.check_single_task(domain_id, ctrl):
                    key = "Controller:{}, Domain ID:{}".format(ctrl, domain_id)
                    err_msg = \
                        self.get_msg("pool.balance.task.hang", key)
                    return CheckStatus.NOTPASS, err_msg

        return CheckStatus.PASS, ""

    def check_all_ctrl_task(self):
        """
        检查所有控制器是否均衡任务挂起
        :return:
        """
        try:
            ctrl_ids = self.get_ctrl_ids()
            self.logger.info("ctrl_ids:{}".format(ctrl_ids))
            return self.check_task(ctrl_ids)
        except Exception:
            err_msg = self.get_msg("query.result.abnormal")
            return CheckStatus.NOCHECK, err_msg
        finally:
            enterCliModeFromSomeModel(self.context.get("cli"), self.lang)

    def check_single_ctrl(self, ctrl_id):
        """
         检查单个控制器是否均衡任务挂起（适用于FRU）
        """
        try:
            ctrl_ids = [ctrl_id]
            self.logger.info("ctrl_ids:{}".format(ctrl_ids))
            result, err_msg = self.check_task(ctrl_ids)
            sug_msg = ""
            if result not in [CheckStatus.PASS, CheckStatus.NO_SUPPORT]:
                sug_msg = self.get_msg("pool.balance.task.hang.suggestion")
            return result, err_msg, sug_msg
        except Exception:
            err_msg = self.get_msg("query.result.abnormal")
            sug_msg = self.get_msg("pool.balance.task.hang.suggestion")
            return CheckStatus.NOTPASS, err_msg, sug_msg
        finally:
            enterCliModeFromSomeModel(self.context.get("cli"), self.lang)
