# -*- coding: UTF-8 -*-
from cbb.business.operate.fru.common import common
from cbb.frame.checkitem.base_dsl_check import CheckStatus
from cbb.frame.cli import cliUtil
from cbb.frame.context import contextUtil
from cbb.frame.cli import cli_con_mgr
from cbb.frame.base.exception import UnCheckException
from utils import Products

# 不涉及的型号
NOCHECK_MODEL = ("2100 V3", "2800 V3", "5100K V5", "5110 V5",
                 "5110F V5", "5200K V5" "5210 V5", "5210F V5")
# 版本和补丁配套列表
PATCH_VERSION_REQUIREMENT = {
    "V500R007C30SPC100": "V500R007C30SPH125",
    "V300R006C50SPC100": "V300R006C50SPH125",
    "V500R007C61": "V500R007C61SPH016",
    "V300R006C61": "V300R006C61SPH016"
}
# 紧急补丁
EMERGENCY_PATCH = {
    "V500R007C30SPH190",
    "V500R007C30SPH189",
    "V500R007C30SPH188",
    "V500R007C30SPH187",
    "V500R007C30SPH186",
    "V300R006C50SPH185",
    "V500R007C30SPH183",
    "V500R007C30SPH182",
    "V500R007C30SPH181",
    "V500R007C30SPH162",
    "V500R007C30SPH161"
}


def execute(context):
    """
    NFS4.1服务开关检查
    """
    check_nfs_switch_match = CheckNFSSwitchMatch(context)
    check_nfs_switch_match.execute_check()
    result_dict = check_nfs_switch_match.check_result
    return result_dict["flag"], result_dict["errMsg"], \
        result_dict["suggestion"], \
        "\n".join(check_nfs_switch_match.cli_ret_list)


class CheckNFSSwitchMatch:
    def __init__(self, context):
        self.context = context
        self.cli = cli_con_mgr.get_available_conn(context, context.get("cli"))
        self.logger = context.get("logger")
        self.lang = contextUtil.getLang(context)
        self.cli_ret_list = []
        self.check_result = \
            {"flag": CheckStatus.PASS, "errMsg": "", "suggestion": ""}

    def execute_check(self):
        """
        NFS4.1服务开关检查
        """
        try:
            # 检查型号
            if not self.check_product_model():
                self.check_result["flag"] = CheckStatus.NO_SUPPORT
                return
            # 检查设备版本
            if not self.is_risk_version():
                self.check_result["flag"] = CheckStatus.NO_SUPPORT
                return

            # 检查补丁是否符合要求
            if self.check_patch_version_is_required():
                return

            # 查询默认租户NFS4.1开关状态
            if self.check_nfs_service_is_on():
                return

            # 检查控制器上是否存在NFS4.1业务
            if self.has_nfs_business_on_all_ctrl():
                self.check_result["flag"] = CheckStatus.NOTPASS
                self.check_result["errMsg"], self.check_result["suggestion"] = \
                    common.getMsg(self.lang, "NFS.exists.and.disabled")
                return
            return
        except UnCheckException as e:
            self.logger.error(str(e.errorMsg))
            self.check_result["flag"] = CheckStatus.NOCHECK
            self.check_result["errMsg"], self.check_result["suggestion"] = \
                common.getMsg(self.lang, "execute.error")
            return
        except Exception as e:
            self.logger.error(str(e))
            self.check_result["flag"] = CheckStatus.NOCHECK
            self.check_result["errMsg"], self.check_result["suggestion"] = \
                common.getMsg(self.lang, "execute.error")
            return

    def check_product_model(self):
        """
        检查设备型号
        """
        # 查询设备型号
        flag, product_model, err_msg, cli_ret = \
            cliUtil.get_product_model_with_ret(self.cli, self.lang)
        self.cli_ret_list.append(cli_ret)
        if flag is not True:
            raise UnCheckException("failed to query the product_model.")
        self.logger.info("product_model: {}".format(product_model))
        if product_model in NOCHECK_MODEL:
            return False
        # 查询控制器内存
        if product_model == "2200 V3":
            ctrl_result = cliUtil.getControllerCacheCapacity(self.cli,
                                                             self.lang)
            self.cli_ret_list.append(ctrl_result[1])
            if ctrl_result[0] is not True:
                self.logger.info("Fail to get controller information.")
                raise UnCheckException(ctrl_result[2])
            self.logger.info("get ctrl capacity: {}".format(ctrl_result[1]))
            if "8" in ctrl_result[1]:
                return False
        return True

    def is_risk_version(self):
        """
        检查版本是否涉及
        """
        flag, product_version, patch_version, cli_ret = \
            cliUtil.get_system_version_with_ret(self.cli, self.lang)
        self.cli_ret_list.append(cli_ret)
        if flag is not True:
            raise UnCheckException("failed to query the soft_version.")
        self.logger.info("product_version: {}".format(product_version))
        if any([all([
            Products.compareVersion(product_version, "V500R007C20") >= 0,
            Products.compareVersion(product_version, "V500R007C61") <= 0]),
            all([Products.compareVersion(product_version, "V300R006C30") >= 0,
                 Products.compareVersion(product_version, "V300R006C61") <= 0])]
               ):
            return True
        return False

    def check_patch_version_is_required(self):
        """
        检查补丁是否符合需求
        """
        # 获取设备版本和补丁版本
        flag, product_version, patch_version = \
            cliUtil.getSystemVersion(self.cli, self.lang)
        self.logger.info("the_product_version: {}".format(product_version))
        self.logger.info("patch_version: {}".format(patch_version))

        if not flag:
            self.logger.info("failed to query the soft_version.")
            raise UnCheckException("failed to query the soft_version.")

        # 判断是否安装了需求补丁
        if self.is_target_patch_version(product_version, patch_version):
            return True

        # 补丁场景，判断目标补丁是否满足要求
        target_patch_version = self.context.get("pkgVersion")
        self.logger.info(
            "target_patch_version: {}".format(target_patch_version))
        if self.context.get("tool_type", "") == "patch" and \
                self.is_target_patch_version(product_version,
                                             target_patch_version):
            return True
        return False

    @staticmethod
    def is_target_patch_version(product_version, patch_version):
        """"
        判断补丁是否满足要求
        """
        if not patch_version or patch_version == "--":
            return False
        for required_version, required_patch in \
                PATCH_VERSION_REQUIREMENT.items():
            if all([required_version in product_version,
                    Products.compareVersion(
                        patch_version, required_patch) >= 0,
                    patch_version not in EMERGENCY_PATCH]):
                return True
        return False

    def check_nfs_service_is_on(self):
        """
        检查默认租户下NFS4.1开关是否开启
        """
        self.logger.info("check_nfs_service_is_on")
        cmd = "show service nfs"
        flag, cli_ret, err_msg = \
            cliUtil.excuteCmdInCliMode(self.cli, cmd, True, self.lang)
        self.cli_ret_list.append(cli_ret)
        if flag == cliUtil.RESULT_NOSUPPORT:
            return True
        if flag is not True:
            raise UnCheckException(err_msg)
        nfs_info_dict_list = cliUtil.getHorizontalCliRet(cli_ret)
        for nfs_info_dict in nfs_info_dict_list:
            if nfs_info_dict.get("Support V41 Enabled", "").lower() == "on":
                return True
        return False

    def has_nfs_business_on_all_ctrl(self):
        """
        检查控制器上是否存在NFS4.1业务
        """
        # 获取控制器ID
        flag, ctrl_id_list, err_msg, cli_ret = \
            cliUtil.get_controller_id_list_with_ret(self.cli, self.lang)
        self.cli_ret_list.append(cli_ret)
        if flag is not True:
            raise UnCheckException("failed to query the controller info.")
        # 检查NFS业务
        for ctrl_id in ctrl_id_list:
            if self.has_nfs_business(ctrl_id):
                return True
        return False

    def has_nfs_business(self, ctrl_id):
        """
        检查单个控制器上是否有NFS4.1业务
        """
        cmd = "show nfs nfsv41_stat controller={}".format(ctrl_id)
        flag, cli_ret, err_msg = cliUtil.excuteCmdInDeveloperMode(
            self.cli, cmd, True, self.lang, preExec=False)
        self.cli_ret_list.append(cli_ret)
        if flag is not True:
            raise UnCheckException(err_msg)
        record_list = cliUtil.get_horizontal_cli_ret_without_dot(cli_ret)
        for record in record_list:
            if record.get("Procedure", "") == "SEQUENCE" and \
                    str(record.get("Counts", "")) != "0":
                return True
        return False
