# -*- coding: UTF-8 -*-
# Copyright (c) Huawei Technologies Co., Ltd. 2012-. All rights reserved.
from cbb.frame.cli import cliUtil
from cbb.frame.context import contextUtil
from cbb.frame.base import baseUtil
from cbb.frame.base.config import NO_FIX_PATCH


def execute(context, hot_patch_params, cli=None):
    """检查当前设备是否满足补丁要求

    :param context: 上下文
    :param hot_patch_params: 版本与要求补丁，例如
    [{
        "product_version": "V500R007C10",  # 软件版本
        "required_patch": "V500R007C10SPH013",  # 要求的通用补丁版本
        "product_series": "ConvergedStorage",  # 设备型号系列 ConvergedStorage/Dorado
        "product_models": [],  # 设备型号列表 为空时所有型号都要满足补丁要求 有值时只有列表中的型号需要满足补丁要求
        "emergency_patch": [“V300R001C21SPH171”, ]  # 紧急补丁白名单，缺省值[]
    }, {
        "product_version": "V300R001C21SPC100",
        "required_patch": "V300R001C21SPH111",
        "product_series": "Dorado",
        "product_models": [],
        "emergency_patch": [ ], # 紧急补丁白名单
        "black_emergency_patch": [“V300R001C21SPH178”, ] # 紧急补丁黑名单
    }]
    紧急补丁白名单和紧急补丁黑名单不能两个同时存在
    :param cli: cli连接，评估工具的cli不可通过cbb的getcli获取，需要传参
    :return: True(满足补丁要求)/False(不满足补丁要求)
    """
    # 参数检查
    if type(hot_patch_params) != list:
        raise Exception("invalid param: %s" % str(hot_patch_params))
    for hot_patch_param in hot_patch_params:
        if type(hot_patch_param) != dict:
            raise Exception("invalid param: %s" % str(hot_patch_params))
        if not hot_patch_param.get("product_version") \
                or not hot_patch_param.get("required_patch"):
            raise Exception("invalid param: %s" % str(hot_patch_params))

    # 常用变量
    if not cli:
        cli = contextUtil.getCli(context)
    lang = contextUtil.getLang(context)

    # 获取当前设备的型号/软件版本/补丁
    flag, dev_model, err_msg = cliUtil.getProductModel(cli, lang)
    if not flag:
        raise Exception("can not get product model: %s" % err_msg)
    flag, dev_version, dev_patch = cliUtil.getSystemVersion(cli, lang)
    if not flag:
        raise Exception("can not get product version")
    flag, required_patch, err_msg = check_patch_version(
        context, dev_version, dev_patch, dev_model, hot_patch_params)
    return flag, dev_version, dev_patch, dev_model, required_patch


def check_patch_version(context, dev_version, dev_patch, dev_model,
                        hot_patch_params):
    lang = contextUtil.getLang(context)
    logger = contextUtil.getLogger(context)
    logger.info("enter check patch version:dev_version={},dev_patch={},"
                "dev_model={}".format(dev_version, dev_patch, dev_model))
    # 根据入参检查当前设备版本和补丁
    is_dorado = baseUtil.isDoradoDev(dev_model)
    dev_series = "ConvergedStorage"
    if is_dorado:
        dev_series = "Dorado"
    for hot_patch_param in hot_patch_params:
        product_version = hot_patch_param.get("product_version")
        required_patch = hot_patch_param.get("required_patch")
        product_series = hot_patch_param.get("product_series")
        product_models = hot_patch_param.get("product_models")
        emergency_patch = hot_patch_param.get("emergency_patch", [])
        black_emergency_patch = hot_patch_param.get("black_emergency_patch",
                                                    [])
        if product_version == dev_version and product_series == dev_series \
                and (not product_models or dev_model in product_models):
            # 紧急补丁场景
            if is_emergency_patch(dev_patch):
                # 没有紧急补丁黑名单的情况
                if not black_emergency_patch:
                    # 紧急补丁用白名单方式检查
                    if dev_patch in emergency_patch:
                        return True, "", ""
                    return False, "", get_suggestion(hot_patch_param, lang)
                else:
                    # 在紧急补丁黑名单内不通过
                    if dev_patch in black_emergency_patch:
                        logger.info("check patch version error")
                        return False, required_patch, get_suggestion(
                                                        hot_patch_param, lang)
                    return True, "", ""
            # 通用补丁比较版本大小
            if get_patch_ver_digital(dev_patch) \
                    >= get_patch_ver_digital(required_patch):
                return True, "", ""
            return False, required_patch, get_suggestion(hot_patch_param, lang)

    # 没有匹配上说明当前型号和版本没有补丁要求
    return True, "", ""


def check_patch_version_with_no_suggestion(
        dev_version, dev_patch, dev_model, hot_patch_params):
    """检查版本信息-不带修复建议

    :param dev_version: 设备软件版本
    :param dev_patch: 设备补丁版本
    :param dev_model: 设备产品型号
    :param hot_patch_params: 要求补丁
    :return:
    """
    # 根据入参检查当前设备版本和补丁
    is_dorado = baseUtil.isDoradoDev(dev_model)
    dev_series = "ConvergedStorage"
    if is_dorado:
        dev_series = "Dorado"
    for hot_patch_param in hot_patch_params:
        product_version = hot_patch_param.get("product_version")
        required_patch = hot_patch_param.get("required_patch")
        product_series = hot_patch_param.get("product_series")
        product_models = hot_patch_param.get("product_models")
        emergency_patch = hot_patch_param.get("emergency_patch", [])
        black_emergency_patch = hot_patch_param.get("black_emergency_patch",
                                                    [])
        if product_version == dev_version and product_series == dev_series \
                and (not product_models or dev_model in product_models):
            # 紧急补丁场景
            if is_emergency_patch(dev_patch):
                # 没有黑名单补丁的情况
                if not black_emergency_patch:
                    # 紧急补丁用白名单方式检查
                    if dev_patch in emergency_patch:
                        return True, ""
                    return False, ""
                else:
                    # 在黑名单紧急补丁内不通过
                    if dev_patch in black_emergency_patch:
                        return False, required_patch
                    return True, ""

            # 通用补丁比较版本大小
            if get_patch_ver_digital(dev_patch) \
                    >= get_patch_ver_digital(required_patch):
                return True, ""
            return False, required_patch

    # 没有匹配上说明当前型号和版本没有补丁要求
    return True, ""


def get_suggestion(params, lang):
    # 无解决补丁时，不提示安装xx。
    if params.get("required_patch") == NO_FIX_PATCH:
        return baseUtil.getPyResource(lang, "no.fix.patch.src.version")

    err_key = params.get("err_key", "need.install.general.patch")
    err_msg = baseUtil.getPyResource(lang, err_key).format(params)
    if params.get("patch_url_zh") and params.get("patch_url_en"):
        down_path = baseUtil.getPyResource(
            lang, "patch.download.url").format(params)
        err_msg = err_msg + down_path
    return err_msg


def is_emergency_patch(patch_ver):
    """ 判断是否是紧急补丁，补丁号后两位大于等于70

    :param patch_ver:
    :return:
    """
    digital_ver = get_patch_ver_digital(patch_ver)
    version = str(digital_ver).rjust(3, "0")
    return version[-2:] >= "70"


def get_patch_ver_digital(patch_ver):
    """获取补丁版本号数字，由于V6现在补丁号为SPH5,SPH6这种格式，所以转为int再比较大小

    :param patch_ver:
    :return:
    """
    patch_str_list = patch_ver.split("SPH")
    if len(patch_str_list) >= 2 and patch_str_list[1]:
        return int(patch_str_list[1])
    return 0
