# coding:utf-8
# Copyright (c) Huawei Technologies Co., Ltd. 2019-2019. All rights reserved.
"""
融合存储高端整柜，扩容函数集合
"""
from cbb.business.operate.fru.common import FuncFactory
from cbb.business.operate.expansion import common
from cbb.business.operate.expansion.enclosure.v5HighEnd.daeLocAlgorithm \
    import get_available_location
from cbb.common.conf.v5HighEnd.fullBayConf import MAX_DISK_ENCS_PER_ENG
from cbb.common.conf.v5HighEnd.fullBayConf import ENC_TYPE
from cbb.common.conf.v5HighEnd.fullBayConf import ENCLOSURES_ON_H3_H10_NUM
from cbb.frame.base import baseUtil
from cbb.frame.context import contextUtil
from cbb.frame.rest import restData
from cbb.frame.rest import restUtil


def get_available_enc_num(context, cabinet_name, cur_enc_height):
    """获取可以选择的硬盘框数量

    :param context: 上下文
    :param cabinet_name: 柜名
    :param cur_enc_height: 当前配置的框高
    :return:
    """
    available_start_locs = get_available_start_locs(context, cabinet_name,
                                                    cur_enc_height)
    return len(available_start_locs)


def get_new_enclosures(context, cabinet_name, enclosure_num, height):
    """根据配置的硬盘框类型、数量，生成对应的信息列表

    :param context: 上下文
    :param cabinet_name: 柜名
    :param enclosure_num: 选择的硬盘框数量
    :param height: 硬盘框高度
    :return:
    """
    # 可扩新框的起始位置
    available_start_locs = get_available_start_locs(context, cabinet_name,
                                                    height)
    new_start_locs = available_start_locs[:enclosure_num]
    # 返回绘图所需的所选框数据
    return get_new_encs_list(new_start_locs, height, cabinet_name)


def get_available_start_locs(context, cabinet_name, height):
    """获取可以扩容新框的起始位置列表

    :param context: 上下文
    :param cabinet_name: 柜名
    :param height: 框高
    :return:
    """
    logger = common.getLogger(context.get("logger"), __file__)

    product_model = contextUtil.getItem(context, "productModel")
    dae_enc_type = contextUtil.getItem(context, "daeEnclosureType")
    is_f_series_pdt = baseUtil.is_f_series_product(product_model)
    available_slots_num = contextUtil.getItem(context, "available_slots_num")
    ori_enc_2u_list = contextUtil.getItem(context, "origin_enc_2u_list")
    ori_enc_4u_list = contextUtil.getItem(context, "origin_enc_4u_list")
    new_2u_list = contextUtil.getItem(context, "newEnc2UList", [])
    support_sas_nvme_mix = contextUtil.getItem(context,
                                               "supportSASandNVMeMix", False)

    available_start_locs = \
        get_available_location(cabinet_name, height, dae_enc_type,
                               is_f_series_pdt,
                               available_slots_num=available_slots_num,
                               ori_enc_2u_list=ori_enc_2u_list,
                               ori_enc_4u_list=ori_enc_4u_list,
                               new_2u_list=new_2u_list,
                               support_sas_nvme_mix=support_sas_nvme_mix)
    logger.logInfo("available %sU enclosure start locations on bay(%s): %s" %
                   (str(height), cabinet_name, str(available_start_locs)))
    return available_start_locs


def get_new_encs_list(new_start_locs, height, cabinet_name):
    """生成所选框的信息元组组成的列表

    :param new_start_locs: 新扩框的起始位置列表
    :param height: 框高
    :param cabinet_name: 柜名
    :return:
    """
    new_enc_list = []
    for start_loc in new_start_locs:
        new_enc_list.append(("%s.%sU" % (cabinet_name, str(start_loc)),
                             ENC_TYPE.get(str(height)),
                             start_loc,
                             cabinet_name))
    return new_enc_list


def init_available_slots_num(context):
    """初始化H3/H10槽位可用于后端接口卡的槽位数

    :param context:
    :return:
    """
    logger = common.getLogger(context.get("logger"), __file__)
    # 控制框个数
    ctrl_enc_num = contextUtil.getItem(context, "ctrlEncNum")

    # 多引擎场景H3/H10用于scaleout接口卡,可用为0
    if ctrl_enc_num > 1:
        available_slots_num = 0
    else:
        # 单引擎场景
        # H3/H10可以用于后端接口卡的槽位数量，最大2
        available_slots_num = 2

        # 已插入的后端接口卡ID列表
        back_intf_id_list = FuncFactory.getIntfIdListByLogicType(
            context, restData.Enum.PortLogicTypeEnum.EXP)

        # 所有接口卡信息列表
        all_intf_list = FuncFactory.getFruListInfo(
            context, restData.Enum.ObjEnum.INTF_MODULE)
        for intf_record in all_intf_list:
            intf_id = restUtil.Tlv2Rest.getRecordValue(
                intf_record, restData.PublicAttributes.ID)
            intf_loc = restUtil.Tlv2Rest.getRecordValue(
                intf_record, restData.PublicAttributes.LOCATION)
            # 如果H3/H10任一槽位已插入接口卡，但非后端接口卡，则可用数减1
            if (("H3" in intf_loc or "H10" in intf_loc)
                    and intf_id not in back_intf_id_list):
                available_slots_num -= 1

    logger.logInfo("H3/H10 available slots number for backend interface "
                   "is :%s" % str(available_slots_num))
    contextUtil.setItem(context, "available_slots_num",
                        available_slots_num)
    return


def check_max_config(context):
    """检查当前系统的硬盘框数量是否已达最大规格

    :param context:
    :return:
    """
    logger = common.getLogger(context.get("logger"), __file__)

    # 控制框个数
    ctrl_enc_num = contextUtil.getItem(context, "ctrlEncNum")
    # 后端组网类型：SAS, SMART
    dae_type = contextUtil.getItem(context, "daeEnclosureType")
    # H3/H10槽位可用于后端接口卡的数量
    available_slots_num = contextUtil.getItem(context,
                                              "available_slots_num", 0)
    # 最大级联深度
    sas_depth = contextUtil.getItem(context, "deepth")
    smart_depth = contextUtil.getItem(context, "smart_depth")
    depth = str(smart_depth) if dae_type == "SMART" else str(sas_depth)

    # 每个引擎最大支持的2U框数量
    max_disk_encs_per_eng = \
        MAX_DISK_ENCS_PER_ENG.get(dae_type, {}).get(depth, 0)

    # 系统最大支持的2U框规格
    max_disk_2u_encs_in_sys = max_disk_encs_per_eng * ctrl_enc_num
    logger.logInfo("the max 2U disk enclosures number: %s" %
                   (str(max_disk_2u_encs_in_sys)))

    # 单引擎IP后端组网，如果H3/H10槽位可以用于后端接口卡，需要增加相应的规格
    if ctrl_enc_num == 1 and dae_type == "SMART":
        max_disk_2u_encs_in_sys += \
            ENCLOSURES_ON_H3_H10_NUM.get(depth) * available_slots_num

    origin_enc_2u_list = contextUtil.getItem(context, "origin_enc_2u_list", [])
    origin_enc_4u_list = contextUtil.getItem(context, "origin_enc_4u_list", [])
    # 剩余可以扩容的硬盘框数量（2U）
    max_sup_new_disk_enc_num = \
        max_disk_2u_encs_in_sys - len(origin_enc_2u_list) - \
        len(origin_enc_4u_list) * 2
    contextUtil.setItem(context, "max_sup_new_disk_enc_num",
                        max_sup_new_disk_enc_num)
    return max_sup_new_disk_enc_num > 0


def clear_context_item(context):
    """清除上下文中保存的配置，防止未完全退出扩容工具而再次扩容时数据污染

    :param context:
    :return:
    """
    contextUtil.removeItem(context, "newConfigClustType")
    contextUtil.removeItem(context, "newConfigCtrlNum")
    contextUtil.removeItem(context, "encGraphInfoList")
    contextUtil.removeItem(context, "newEnc2UList")
    contextUtil.removeItem(context, "newEnc4UList")


def clear_selected_config(context):
    """当从连接线缆步骤返回上一步重新配置时，重置上下文中保存的配置

    :param context:
    :return:
    """
    # 重置已选柜
    contextUtil.setItem(context, "selectedCabinets", [])
    contextUtil.setItem(context, "selectedConfig", {})
    # 重置可选柜
    available_cabinets_group = \
        contextUtil.getItem(context, "availableCabinetsGroup", [])
    contextUtil.setItem(context, "availableCabinets", available_cabinets_group)
    # 重置已选框
    contextUtil.setItem(context, "newEnc2UList", [])
    contextUtil.setItem(context, "newEnc4UList", [])
    # 重置新增硬盘柜绘图
    contextUtil.setItem(context, "newBayList", [])
    # 重置新增硬盘框绘图
    contextUtil.setItem(context, "encGraphInfoList", [])
