# -*- coding: UTF-8 -*-
import copy

from cbb.business.operate.expansion import common
from cbb.business.operate.expansion.cabinetFactory import CABINET
from cbb.business.operate.expansion.enclosureConfig import AREA_ENCLOSURE_CONF, BAY_AREA_PARTIONS_CONF
from cbb.frame.cli import cliUtil
from cbb.frame.context import contextUtil
from cbb.frame.tlv import tlvData
from cbb.frame.tlv import tlvUtil

# 一个系统柜支持最大的硬盘柜数量(包含系统柜)-级联深度：柜数量
MAX_DISK_BAYS_PER_BAY = {'1': 2, '2': 3, '4': 6}

##直流电源场景接入了不允许接入的SAS口，二级：6、7号环路，四级：3号环路
ERR_SAS_PORTS_FOR_DC = {'2': ['R5.P6', 'L5.P6', 'R5.P7', 'L5.P7'],
                        '4': ['R5.P3', 'L5.P3']}

SERIES_18000F = ["18800F V3", "18500F V3", "18800F V5", "18500F V5"]


def is18000FDevice(context):
    '''
    @summary: 当前集群是否是18000F V3和V5的设备
    '''
    devObj = contextUtil.getDevObj(context)
    productModel = devObj.get("type")
    if productModel.upper() in SERIES_18000F:
        return True
    return False


def isNewExpansion(context):
    '''
    @summary: 是否是新增直流或二级、一级级联扩容场景
    @return: True-是新扩容场景，False-是原有场景
    '''
    deepth = contextUtil.getItem(context, "deepth")
    powerType = contextUtil.getItem(context, "powerType")
    productModel = contextUtil.getItem(context, "productModel")

    if not common.is18000V3(productModel):
        return False
    ##1、未查询到级联深度 2、四级级联深度+交流电源是18000 V3固有场景，其余为新增场景
    if deepth == "":
        return False
    if deepth == "4" and powerType == tlvData.POWER_TYPE_E["AC"]:
        return False
    return True


def getEnlosureConf(context, height):
    '''
    @summary: 获取 AREA_级联类型_电源类型_框类型配置字典
    '''
    deepth = contextUtil.getItem(context, "deepth")
    powerType = contextUtil.getItem(context, "powerType")
    ##关键字组成：级联深度+电源类型+高度
    keyStr = str(deepth) + getPowerTypeStr(powerType) + str(height) + "U"
    return AREA_ENCLOSURE_CONF[keyStr]


def getBayPartionsConf(context):
    '''
    @summary: 获取 柜中包含的域配置：BAY_AREA_级联类型_MAP字典
    '''
    deepth = contextUtil.getItem(context, "deepth")
    powerType = contextUtil.getItem(context, "powerType")

    ##关键字组成：电源类型+级联深度
    keyStr = getPowerTypeStr(powerType) + str(deepth)
    return BAY_AREA_PARTIONS_CONF[keyStr]


def getPowerTypeStr(powerType):
    '''
    @summary: 将电源类型枚举值转换为字符串
    '''
    if tlvData.POWER_TYPE_E['DC'] == powerType:
        return "DC"
    if tlvData.POWER_TYPE_E['AC'] == powerType:
        return "AC"
    return ""


def isDcErrNet(context):
    '''
    @summary: 判断直流电源场景是否连接了不允许接入的SAS口，二级级联：6、7环路，四级级联：3号环路。
    @return: True:检查通过，False-检查不通过
    '''
    cli = contextUtil.getCli(context)
    lang = contextUtil.getLang(context)
    logger = common.getLogger(context.get("logger"), __file__)

    deepth = contextUtil.getItem(context, "deepth")
    powerType = contextUtil.getItem(context, "powerType")
    errPorts = []

    if powerType == tlvData.POWER_TYPE_E['AC']:
        return (True, errPorts)

    confErrPorts = ERR_SAS_PORTS_FOR_DC.get(deepth, [])
    linkupSasPorts = cliUtil.getLinkupSasPorts(cli, lang)
    logger.logInfo("linkupSasPorts:%s" % str(linkupSasPorts))
    for linkupSasPort in linkupSasPorts:
        if 'CTE' not in linkupSasPort:
            continue
        location = linkupSasPort[5:]
        if location.upper() in confErrPorts:
            errPorts.append(linkupSasPort)

    if len(errPorts) != 0:
        return (False, errPorts)
    return (True, errPorts)


def getNewBayNames(context):
    '''
    @summary: 获取待扩容硬盘柜的name，DKBx_x
    '''

    (smbs, diskBays) = common.getSMBsAndDiskBays(context)
    oldCabinetNames = contextUtil.getItem(context, "oldCabinetNames")
    deepth = contextUtil.getItem(context, "deepth")

    newDiskBays = []
    for smb in smbs:
        # 硬盘柜的序号是从1开始编号
        engId = smb[-1]  # 系统柜编号的最后一位为当前系统的引擎号
        for index in range(1, MAX_DISK_BAYS_PER_BAY[deepth]):
            bayId = "DKB" + engId + '_' + str(index)
            if bayId not in oldCabinetNames:
                newDiskBays.append(bayId)

    return newDiskBays


def isMaxBaysConfig(context):
    '''
    @summary: 检查硬盘柜是否满配
    '''
    deepth = contextUtil.getItem(context, "deepth")
    (smbs, diskBays) = common.getSMBsAndDiskBays(context)
    totalDiskBays = len(smbs) * (MAX_DISK_BAYS_PER_BAY[deepth] - 1)
    if len(diskBays) >= totalDiskBays:
        return True

    return False


def getTotalHeight(context):
    '''
    @summary: 获取所有机柜存放硬盘框的总高度
    '''
    # 一个机柜可用的空间，电源类型：{柜类型：高度（单位U）}
    TOTAL_HEIGHT_PER_CANINET = {"AC1": {'SMB': 32, 'DKB': 8},
                                "DC2": {'SMB': 24, 'DKB': 32},
                                "AC2": {'SMB': 32, 'DKB': 32},
                                "DC4": {'SMB': 24, 'DKB': 32},
                                "AC4": {'SMB': 32, 'DKB': 32},
                                }

    deepth = contextUtil.getItem(context, "deepth")
    powerType = contextUtil.getItem(context, "powerType")
    keyStr = getPowerTypeStr(powerType) + str(deepth)

    (smbs, diskBays) = common.getSMBsAndDiskBays(context)
    smbsHeight = len(smbs) * TOTAL_HEIGHT_PER_CANINET.get(keyStr)['SMB']
    diskBaysHeight = len(diskBays) * TOTAL_HEIGHT_PER_CANINET.get(keyStr)['DKB']
    totalHeight = smbsHeight + diskBaysHeight
    return totalHeight


def checkMaxEnclosureConfig(context, tlv):
    '''
    @summary: 检查硬盘框是否满配，判断机柜空位是否放满
    '''
    totalHeight = getTotalHeight(context)

    countHeight = 0
    enclosureRecords = tlvUtil.getEnclosureRecords(tlv)
    for record in enclosureRecords:
        logicType = tlvUtil.getRecordValue(record, tlvData.ENCLOSURE['logicType'])
        # 非硬盘框跳过
        if logicType != tlvData.ENCLOSURE_TYPE_E["EXP"]:
            continue
        height = tlvUtil.getRecordValue(record, tlvData.ENCLOSURE['height'])
        countHeight += height

    if countHeight >= totalHeight:
        return True

    return False


def checkEnclosureModel(cabinetInfoDict):
    '''
    @summary: 判断机柜内同一分区的硬盘型号是否相同
    '''
    differentCabinetInfo = []
    isSame = True
    for cabinetName in cabinetInfoDict.keys():
        cabinet = cabinetInfoDict.get(cabinetName, None)
        if cabinet == None:
            continue

        partitions = cabinet.get_newpartitions()
        if not common.isSameEnclosureModel(partitions):
            isSame = False
            differentCabinetInfo.append(cabinetName)
            continue

    return (isSame, differentCabinetInfo)


def getPartitionEnclosure(enclosureDict, enclosureName):
    ''' 
    @summary: 根据框名称获取所在分区（环路），硬盘框信息
    '''
    for (partitionNum, enclosureList) in enclosureDict.items():
        for enclosureInfo in enclosureList:
            enclosureXName = enclosureName[:3] + 'X' + enclosureName[4:]
            if enclosureInfo[0] in enclosureXName:
                return (partitionNum, enclosureInfo)
    return (None, None)


def getCabinetInfoDict(context, tlv):
    '''
    @summary: 获取已有硬盘框，并映射为字典
            字典格式：
            {
                "柜号0"：柜对象,
                "柜号1"：柜对象,
                ...
            }
    '''
    logger = common.getLogger(context.get("logger"), __file__)
    cabinetInfoDict = {}
    cabinetDict = contextUtil.getItem(context, "cabinetDict")
    logger.logInfo("cabinetDict: %s" % str(cabinetDict))
    ##不符合标准组网配置的硬盘框列表（例如二级级联深度，级联2个4U框）
    errEnclosureList = []

    encRecords = tlvUtil.getEnclosureRecords(tlv)
    for record in encRecords:
        logicType = tlvUtil.getRecordValue(record, tlvData.ENCLOSURE['logicType'])
        if logicType != tlvData.ENCLOSURE_TYPE_E["EXP"]:
            continue

        parentId = tlvUtil.getRecordValue(record, tlvData.PUB_ATTR["parentID"])
        cabinetName = cabinetDict.get(parentId, "")
        name = tlvUtil.getRecordValue(record, tlvData.PUB_ATTR["name"])
        height = tlvUtil.getRecordValue(record, tlvData.ENCLOSURE["height"])

        ##获取分区号(环路号)
        enclosureDict = getEnlosureConf(context, height)
        (partitionNum, enclosureInfo) = getPartitionEnclosure(enclosureDict, name)
        logger.logInfo("enclosure name: %s, partition number:%s" % (name, partitionNum))
        if partitionNum == None or enclosureInfo == None:
            errEnclosureList.append(name)
            continue

        cabinet = None
        if cabinetName not in cabinetInfoDict.keys():
            cabinetId = common.getCabinetId(cabinetName)
            cabinetType = common.getCabinetType(cabinetName)
            enginId = common.getEnginId(cabinetName)
            cabinet = CABINET(cabinetId, cabinetName, cabinetType, enginId)
            logger.logInfo("getCabinetInfoDict not exist: %s" % cabinetName)
        else:
            cabinet = cabinetInfoDict.get(cabinetName)
            logger.logInfo("getCabinetInfoDict exist: %s" % cabinetName)
        logger.logInfo("%s cabinet id %d" % (cabinetName, id(cabinet)))

        logger.logInfo("cabinetInfoDict: %s" % cabinetInfoDict)
        logger.logInfo("cabinetName: %s, enclosure name:%s" % (cabinet.name, name))
        cabinet.add_enclosure_to_newpartition(enclosureInfo, partitionNum)
        cabinetInfoDict[cabinetName] = cabinet
    errEnclosureList = sorted(errEnclosureList)
    contextUtil.setItem(context, "errEnclosureList", errEnclosureList)
    return cabinetInfoDict


def initClsInfoForCabinets(context, tlv):
    '''
    @summary: 初始化扩容硬盘柜数据（主要为与扩硬盘框存在的差异项）
    '''
    logger = common.getLogger(context.get("logger"), __file__)

    # 扩容类型
    expandType = "expandDiskBayAll"
    logger.logInfo("expandType:%s" % expandType)
    contextUtil.setItem(context, "expandType", expandType)

    availableCabinets = getNewBayNames(context)
    # 具备扩容条件,且未进行扩容的所有柜
    contextUtil.setItem(context, "availableCabinets", availableCabinets)

    # 获取所有柜（已有柜和可扩展柜）及对应的硬盘框框号和类型
    oldCabinetInfoDict = getCabinetInfoDict(context, tlv)
    logger.logInfo("oldCabinetInfoDict:%s" % str(oldCabinetInfoDict))
    cabinetInfoDict = common.getNewCabinetInfoDict(context, oldCabinetInfoDict)
    logger.logInfo("oldCabinetInfoDict:%s" % str(cabinetInfoDict))
    contextUtil.setItem(context, "oldCabinetInfoDict", cabinetInfoDict)

    # 初始化为已有柜信息
    contextUtil.setItem(context, "newCabinetInfoDict", copy.deepcopy(cabinetInfoDict))

    # 具备扩容条件,且为已选择了扩容配置的所有柜
    contextUtil.setItem(context, "selectedCabinets", [])

    # 具备扩容条件的所有柜
    contextUtil.setItem(context, "availableCabinetsGroup", availableCabinets)
    return


def initClsInfoForEnclosures(context, tlv):
    '''
    @summary: 初始化扩容硬盘框数据（主要为与扩硬盘柜存在的差异项）
    '''
    logger = common.getLogger(context.get("logger"), __file__)

    # 已有机柜
    cabinetNames = contextUtil.getItem(context, "oldCabinetNames")

    # 扩容类型
    expandType = "expandDiskEnclosureAll"
    logger.logInfo("expandType:%s" % expandType)
    contextUtil.setItem(context, "expandType", expandType)

    # 获取已有柜及对应的硬盘框框号和类型
    oldCabinetInfoDict = getCabinetInfoDict(context, tlv)
    logger.logInfo("oldCabinetInfoDict:%s" % str(oldCabinetInfoDict))
    contextUtil.setItem(context, "oldCabinetInfoDict", oldCabinetInfoDict)

    # 初始化为已有柜信息
    contextUtil.setItem(context, "newCabinetInfoDict", copy.deepcopy(oldCabinetInfoDict))

    # 具备扩容条件,且未进行扩容的所有柜
    contextUtil.setItem(context, "availableCabinets", cabinetNames)

    # 具备扩容条件,且为已选择了扩容配置的所有柜
    contextUtil.setItem(context, "selectedCabinets", [])

    # 具备扩容条件的所有柜
    contextUtil.setItem(context, "availableCabinetsGroup", cabinetNames)
    return


def convertXEnclosure2Detail(enclosure, cabinetName):
    '''
    @summary: 将配置中带X的未知引擎号，换成具体的引擎号数字。
    '''
    engineId = common.getEnginId(cabinetName)
    encloName = enclosure[0]
    detailEncloName = encloName[:3] + str(engineId) + encloName[4:]

    height = str(enclosure[1])
    encType = common.ENC_TYPE[height]

    return (detailEncloName, encType, enclosure[2], cabinetName)


def convertDetail2XEnclosure(enclosure):
    '''
    @summary: 将列表中具体的框号、柜号，替换成配置中带X的未知框号、引擎号
    '''
    detailEncloName = enclosure[0]
    detailCabinetName = enclosure[3]
    encloName = detailEncloName[:3] + 'X' + detailEncloName[4:]
    cabinetName = detailCabinetName[:3] + 'X' + detailCabinetName[4:]
    return (encloName, enclosure[1], enclosure[2], cabinetName)


def filterPartitions(context, cabinet, height):
    '''
    @summary: 过滤柜中可放置硬盘框的分区,非空可放置硬盘框区+空闲区（按照位置从下往上放置）
    '''
    logger = context.get("logger")
    emptyPartitionDict = {}
    noEmptyPartitionDict = {}
    cabinetName = cabinet.get_name()

    # 获取当前柜的所有环路号
    bayPartitonsDict = getBayPartionsConf(context)
    cabinetXName = cabinetName[:3] + 'X' + cabinetName[4:]
    allPartitionNums = bayPartitonsDict[cabinetXName]

    enclosureDict = getEnlosureConf(context, height)
    for partionNum in allPartitionNums:
        ##step-1 获取环路partionNum中高端未height的满配框列表，和当前环路已经存放了的框列表
        allEnclos = enclosureDict[partionNum]
        currentEnclos = cabinet.get_one_newpartiton(partionNum)
        logger.info("partition's (%s) current enclosures: %s" % (str(partionNum), str(currentEnclos)))
        ##step-2 currentEnclos为空或者是满配框的子集，说明类型相同，都可以继续新增。
        if not set(currentEnclos).issubset(set(allEnclos)):
            continue
        ##step-3 获取差集，即是当前环路中还能放置的height高度框
        avilableEnclos = list(set(allEnclos).difference(set(currentEnclos)))
        avilableEnclos = sorted(avilableEnclos, key=lambda e: e[2])
        logger.info("partition's (%s) avaliable enclosures: %s" % (str(partionNum), str(avilableEnclos)))
        ##step-4 当前环路是否已经放置了框，区分非空域，空域
        if len(currentEnclos) != 0:
            noEmptyPartitionDict.setdefault(partionNum, avilableEnclos)
        else:
            emptyPartitionDict.setdefault(partionNum, avilableEnclos)

    logger.info("cabinet's (%s) no empty partitions dict: %s" % (str(cabinetName), str(noEmptyPartitionDict)))
    logger.info("cabinet's (%s) empty partitions dict: %s" % (str(cabinetName), str(emptyPartitionDict)))
    return (noEmptyPartitionDict, emptyPartitionDict)


def getListFrom2DictValues(noEmptyPartitionDict, emptyPartitionDict):
    '''
    @summary: 将2非空和空域可存放框的字典，转换成值的list合集，按照非空（位置排序）+空（位置排序）
    '''
    noEmptyPartitionList = []
    emptyPartitionList = []

    for partionList in noEmptyPartitionDict.values():
        noEmptyPartitionList.extend(partionList)
    noEmptyPartitionList = sorted(noEmptyPartitionList, key=lambda e: e[2])

    for partionList in emptyPartitionDict.values():
        emptyPartitionList.extend(partionList)
    emptyPartitionList = sorted(emptyPartitionList, key=lambda e: e[2])

    avilablePartitions = noEmptyPartitionList + emptyPartitionList
    return avilablePartitions


def getNewEnclosures(context, cabinet, requiredNum, height):
    '''
    @summary: 获取当前柜可扩的硬盘框列表（名称，类型，所属柜）
    '''
    # 需求框的数量为0时，直接返回满足条件
    encList = []
    if requiredNum == 0:
        return encList

    logger = context.get("logger")
    cabinetName = cabinet.get_name()
    logger.info("cabinet's (%s) partitions:%s" % (cabinetName, str(cabinet.get_newpartitions())))
    count = 0  # 计数新增框个数

    (noEmptyPartitionDict, emptyPartitionDict) = filterPartitions(context, cabinet, height)
    avilablePartitions = getListFrom2DictValues(noEmptyPartitionDict, emptyPartitionDict)
    logger.info("cabinet's (%s) avaliable enclosures: %s" % (str(cabinetName), str(avilablePartitions)))
    for enclosure in avilablePartitions:
        encList.append(convertXEnclosure2Detail(enclosure, cabinetName))
        count += 1
        if count >= requiredNum:
            break

    return encList


def addPartitions2TmpCabinet(context, newCabinet, partitionDict, selectedEncNum):
    '''
    @summary: 生成临时柜存放硬盘框分配信息，不保存在内存。
    '''
    count = 0  # 计数新增框个数
    if selectedEncNum == 0:
        return (count, newCabinet)

    for (partionNum, avilableEnclos) in partitionDict.items():
        for enclosureInfo in avilableEnclos:
            newCabinet.add_enclosure_to_newpartition(enclosureInfo, partionNum)
            count += 1
            if count >= selectedEncNum:
                return (count, newCabinet)
    return (count, newCabinet)


def getUpperLimitOfNewEncs(context, cabinet, height, selectedHeight, selectedEncNum=0):
    '''
    @summary: 在非空闲可放置区/空闲区获取可扩的硬盘框数量
    @param cabinet: 柜对象
    @param height: 硬盘框高度（判断2U还是4U）
    @param selectedEncNum: 已选择框的数量（如：获取4U框上限数量时，需要知道已选定的2U框的数量）
    @param selectedHeight: 已选择框框类型,只支持2U框和4U框
    '''
    logger = context.get("logger")
    cabinetName = cabinet.get_name()
    logger.info(
        "[getUpperLimitOfNewEncs]cabinet's (%s) partitions:%s" % (cabinetName, str(cabinet.get_newpartitions())))
    newCabinet = copy.deepcopy(cabinet)

    ##如果是F系列高端，只允许2U框接入，4U可选上限为0
    if str(height) == '4' and is18000FDevice(context):
        return 0

    # 为已经选择的2U框虚拟分配硬盘框到newCabinet中
    if selectedEncNum != 0:
        (noEmptyPartitionDict, emptyPartitionDict) = filterPartitions(context, newCabinet, selectedHeight)
        (count, newCabinet) = addPartitions2TmpCabinet(context, newCabinet, noEmptyPartitionDict, selectedEncNum)
        (count, newCabinet) = addPartitions2TmpCabinet(context, newCabinet, emptyPartitionDict,
                                                       (selectedEncNum - count))
        logger.info("[getUpperLimitOfNewEncs]new cabinet's (%s) partitions:%s" % (
        cabinetName, str(newCabinet.get_newpartitions())))

    # 为现在选择的框类型获取所有可分配硬盘框数
    (noEmptyPartitionDict, emptyPartitionDict) = filterPartitions(context, newCabinet, height)
    avilablePartitions = getListFrom2DictValues(noEmptyPartitionDict, emptyPartitionDict)
    logger.info(
        "[getUpperLimitOfNewEncs]new cabinet's (%s) avialble partitions:%s" % (cabinetName, str(avilablePartitions)))

    return len(avilablePartitions)


def getNewCabinetInfoFor2UEnc(context, enc2UList, cabinetName):
    '''
    @summary: 选定2U硬盘框后生成新的柜信息，用于生成新增4U框的信息
    @param enc2UList:新增2U硬盘框的信息，格式为:[(框名称，类型，起始位置，所属柜)] 
    '''

    oldCabinetInfoDict = contextUtil.getItem(context, "oldCabinetInfoDict")
    copyCabinetInfoDict = copy.deepcopy(oldCabinetInfoDict.copy())
    newCabinetInfoDict = contextUtil.getItem(context, "newCabinetInfoDict")
    cabinet = copyCabinetInfoDict.get(cabinetName)
    # 未选定2U框则返回原有柜信息
    if len(enc2UList) == 0:
        newCabinetInfoDict[cabinetName] = cabinet
        return newCabinetInfoDict

    enclosureDict = getEnlosureConf(context, 2)  # 为2U框分配数据

    for encInfo in enc2UList:
        encName = encInfo[0]
        ##获取分区号(环路号)
        (partitionNum, enclosureInfo) = getPartitionEnclosure(enclosureDict, encName)
        cabinet.add_enclosure_to_newpartition(enclosureInfo, partitionNum)

    newCabinetInfoDict[cabinetName] = cabinet
    return newCabinetInfoDict
