# -*- coding: UTF-8 -*-

from cbb.business.operate.fru.common import config
from cbb.frame.base import baseUtil
from cbb.frame.context import contextUtil
from cbb.frame.rest import restData
from cbb.common.conf import productConfig
from cbb.frame.rest.restUtil import CommonRest
from cbb.business.operate.fru.common import BaseFactory
from cbb.business.operate.fru.common import LoopCheckFactory
from cbb.business.operate.fru.common import common
from cbb.business.operate.fru.common.common import ExpBordObjForLowAndMidEnd

SAS_ALARMS_ORI = {
    # 单链路：0x000F00CE002A
    "64438009898": {
        "errKey": "sas.cable.board.error",
        "errParamIndex": [1, 2],
        "type": "EXPBOARD",
        "errMsgArgs": ["location"],
    },
    # SAS线缆连接错误告警：0x000F00D00016
    "64438140950": {
        "errKey": "sas.cable.port.error",
        "errParamIndex": [1, 3, 4],
        "type": "SAS_PORT",
        "errMsgArgs": ["location"],
    },
    # 级联端口线缆连接错误：0xF0006000F
    "64424902671": {
        "errKey": "sas.cable.port.error",
        "errParamIndex": [1, 3, 4],
        "type": "SAS_PORT",
        "errMsgArgs": ["location"],
    },
    # SAS线缆连接错误告警：0xF00060010
    "64424902672": {
        "errKey": "sas.cable.port.error",
        "errParamIndex": [1, 4, 5],
        "type": "SAS_PORT",
        "errMsgArgs": ["location"],
    },
    # SAS线缆连接错误告警：0xF00D10046
    "64438206534": {
        "errKey": "sas.cable.port.error",
        "errParamIndex": [1, 3, 4],
        "type": "SAS_PORT",
        "errMsgArgs": ["location"],
    },
    # SAS端口连接不正确：0xF00060011
    "64424902673": {
        "errKey": "sas.cable.pair.port.error",
        "errParamIndex": [(17, 21, 22), (25, 29, 30)],
        "type": "SAS_PORT",
        "errMsgArgs": ["location", "suggestPeerLocation"],
    },
    # 线缆规格与端口不兼容：F00060016
    "64424902678": {
        "errKey": "sas.cable.pair.port.incompatible",
        "errParamIndex": [2, 4, 5],
        "type": "PCIE_PORT",
        "errMsgArgs": ["location"],
    },
    # 新硬件新增告警
    # 端口之间的线缆未连接：0XF00060023
    "64424902691": {
        "errKey": "sas.cable.port.error",
        "errParamIndex": [(9,), (12,)],
        "type": "SAS_PORT",
        "portTypeIndex": [7, 10],
        "errMsgArgs": ["location", "suggestPeerLocation"],
    },
    # 端口上的线缆未连接：0xF00060022
    "64424902690": {
        "errKey": "sas.cable.port.error",
        "errParamIndex": [4],
        "type": "SAS_PORT",
        "portTypeIndex": [2],
        "errMsgArgs": ["location"],
    },
    # 端口之间的线缆未按标准组网连接：0xF00060021
    "64424902689": {
        "errKey": "sas.cable.port.error",
        "errParamIndex": [(4,), (8,)],
        "type": "SAS_PORT",
        "portTypeIndex": [2, 6],
        "errMsgArgs": ["location", "suggestPeerLocation"],
    },
    # 端口间的线缆连接错误：0xF00060020
    "64424902688": {
        "errKey": "sas.cable.port.error",
        "errParamIndex": [(4,), (8,)],
        "type": "SAS_PORT",
        "portTypeIndex": [2, 6],
        "errMsgArgs": ["location", "suggestPeerLocation"],
    },
    # 端口上的线缆连接错误：0xF0006001F
    "64424902687": {
        "errKey": "sas.cable.port.error",
        "errParamIndex": [4],
        "type": "SAS_PORT",
        "portTypeIndex": [2],
        "errMsgArgs": ["location"],
    },
}

SAS_ALARMS_FOR_DORADO_NAS = {
    # 单链路：0x000F00CE002A
    "64438009898": {
        "errKey": "sas.cable.board.error",
        "errParamIndex": [1, 2],
        "type": "EXPBOARD",
        "errMsgArgs": ["location"],
    },
    # SAS线缆连接错误告警：0x000F00D00016
    "64438140950": {
        "errKey": "sas.cable.port.error",
        "errParamIndex": [1, 3, 4],
        "type": "SAS_PORT",
        "errMsgArgs": ["location"],
    },
    # 级联端口线缆连接错误：0xF0006000F
    "64424902671": {
        "errKey": "sas.cable.port.error",
        "errParamIndex": [1, 3, 4],
        "type": "SAS_PORT",
        "errMsgArgs": ["location"],
    },
    # SAS线缆连接错误告警：0xF00060010
    "64424902672": {
        "errKey": "sas.cable.port.error",
        "errParamIndex": [1, 4, 5],
        "type": "SAS_PORT",
        "errMsgArgs": ["location"],
    },
    # SAS线缆连接错误告警：0xF00D10046
    "64438206534": {
        "errKey": "sas.cable.port.error",
        "errParamIndex": [1, 3, 4],
        "type": "SAS_PORT",
        "errMsgArgs": ["location"],
    },
    # SAS端口连接不正确：0xF00060011
    "64424902673": {
        "errKey": "sas.cable.pair.port.error",
        "errParamIndex": [(17, 21, 22), (25, 29, 30)],
        "type": "SAS_PORT",
        "errMsgArgs": ["location", "suggestPeerLocation"],
    },
    # 线缆规格与端口不兼容：F00060016
    "64424902678": {
        "errKey": "sas.cable.pair.port.incompatible",
        "errParamIndex": [2, 4, 5],
        "type": "PCIE_PORT",
        "errMsgArgs": ["location"],
    },
    # 新硬件新增告警
    # 端口之间的线缆未连接：0XF0006002A
    "64424902698": {
        "errKey": "sas.cable.port.error",
        "errParamIndex": [(9,), (12,)],
        "type": "SAS_PORT",
        "portTypeIndex": [7, 10],
        "errMsgArgs": ["location", "suggestPeerLocation"],
    },
    # 端口上的线缆未连接：0xF00060022
    "64424902690": {
        "errKey": "sas.cable.port.error",
        "errParamIndex": [4],
        "type": "SAS_PORT",
        "portTypeIndex": [2],
        "errMsgArgs": ["location"],
    },
    # 端口之间的线缆未按标准组网连接：0xF00060029
    "64424902697": {
        "errKey": "sas.cable.port.error",
        "errParamIndex": [(4,), (8,)],
        "type": "SAS_PORT",
        "portTypeIndex": [2, 6],
        "errMsgArgs": ["location", "suggestPeerLocation"],
    },
    # 端口间的线缆连接错误：0xF00060028
    "64424902696": {
        "errKey": "sas.cable.port.error",
        "errParamIndex": [(4,), (8,)],
        "type": "SAS_PORT",
        "portTypeIndex": [2, 6],
        "errMsgArgs": ["location", "suggestPeerLocation"],
    },
    # 端口上的线缆连接错误：0xF0006001F
    "64424902687": {
        "errKey": "sas.cable.port.error",
        "errParamIndex": [4],
        "type": "SAS_PORT",
        "portTypeIndex": [2],
        "errMsgArgs": ["location"],
    },
}

MULTI_PARAM_SAS_ALARM = [
    "64424902673",
    "64424902691",
    "64424902689",
    "64424902688",
    "64424902698",
    "64424902697",
    "64424902696",
]


class SasCableAlarm():
    """根据SAS线缆告警检查连线正确性

    """
    # 备件类型
    EXPBOARD = "EXPBOARD"
    INTF_MODULE = "INTF_MODULE"
    SAS_PORT = "SAS_PORT"
    CONTROLLER = "CONTROLLER"

    # 备件上下文属性
    LOCATION = "location"
    PARENT_ID = "parentID"

    def __init__(self, context, **kwargs):
        self.context = context
        self.logger = baseUtil.getLogger(context.get("logger"), __file__)
        self.lang = contextUtil.getLang(context)
        self.args = kwargs
        self.productModel = contextUtil.getProductModel(context)
        self.isHightEnd = \
            contextUtil.getItem(context, 'is_kunpeng_high', False) \
            or contextUtil.getItem(context, 'isDorado18000', False)
        self.productVersion = contextUtil.getCurVersion(context)
        is_dorado_nas = productConfig.compare_dorado_version(
            self.productVersion, productConfig.DORADO_NAS)
        self.SAS_ALARMS = SAS_ALARMS_FOR_DORADO_NAS if \
            is_dorado_nas[0] and is_dorado_nas[1] else SAS_ALARMS_ORI

    def execute(self):
        try:
            if self.args and not self.args.get("preCheck", False):
                baseUtil.safeSleep(config.Overtime.SAS_CABLE_ALARM_TIMEOUT)
            rest = contextUtil.getRest(self.context)

            # 获取告警
            sasAlarms = self.getValidAlarms(rest)
            if not sasAlarms:
                self.logger.logPass()
                return []

            # 提示错误信息
            # 更换级联板、后端接口卡屏蔽本端告警，值检查对端告警
            errInfos = self.getErrInfo(sasAlarms)
            if self.args and self.args.get("mask", False) and self.args.get("rowKey"):
                if not self.isHightEnd:
                    return self.checkPeerEndSasCableAlarms(errInfos)
                return self.filter_current_component_alarms(errInfos)
            return errInfos
        except Exception as exception:
            self.logger.logException(exception)
            infoDict = {
                "location": "",
                "suggestPeerLocation": "",
                "type": "",
                "errMsgKey": "dev.conn.failure",
                "errMsgArgs": []
            }
            return [infoDict.copy()]

    def getValidAlarms(self, rest):
        """检查告警

        :param rest:
        :return:
        """
        sasAlarms = []
        records = CommonRest.getAlarms(rest)
        if not records:
            return sasAlarms
        for record in records:
            alarmId = str(CommonRest.getRecordValue(record, restData.RestCfg.Alarm.EVENTID))
            if alarmId in self.SAS_ALARMS:
                sasAlarms.append(record)
        self.logger.logInfo("filter sas alarms:{}".format(sasAlarms))
        return sasAlarms

    def getSingleLocation(self, params, errParamIndex):
        if len(params) < errParamIndex[-1]:
            return None

        values = []
        for index in errParamIndex:
            values.append(params[index])

        return ".".join(values)

    def getPairLocation(self, params, errParamIndex):
        if len(params) < errParamIndex[-1][-1]:
            return None

        pair = []
        for indexList in errParamIndex:
            values = []
            for index in indexList:
                values.append(params[index])
            pair.append(".".join(values))

        return tuple(pair)

    def getErrInfo(self, sasAlarms):

        errInfo = []
        for alarm in sasAlarms:
            alarmId = str(CommonRest.getRecordValue(alarm, restData.RestCfg.Alarm.EVENTID))
            paramInfo = CommonRest.getRecordValue(alarm, restData.RestCfg.Alarm.EVENTPARAM)
            params = paramInfo.split(",")
            errElements = self.SAS_ALARMS.get(alarmId, {})
            errParamIndex = errElements.get("errParamIndex", [])
            errType = errElements.get("type")
            portTypeIndex = errElements.get("portTypeIndex", [])

            # 筛选告警端口类型
            # 0:SAS;1:FC;2:ETH;3:PCIe;4:RDMA
            for index in portTypeIndex:
                if len(params) - 1 >= index:
                    portType = params[index]
                    if str(portType) not in ["0", "2", "4"]:
                        continue

            suggestPeerLocation = ""
            if alarmId not in MULTI_PARAM_SAS_ALARM:
                location = self.getSingleLocation(params, errParamIndex)
            else:
                location, suggestPeerLocation = self.getPairLocation(params, errParamIndex)

            if not location:
                continue

            infoDict = {
                "alarmId": alarmId,
                "location": location,
                "suggestPeerLocation": suggestPeerLocation,
                "type": errType,
                "errMsgKey": errElements.get("errKey"),
                "errMsgArgs": errElements.get("errMsgArgs")
            }
            errInfo.append(infoDict.copy())
        return errInfo

    def checkExpAlarm(self, sasAlarmInfo, expLocations):
        '''
        检查有无级联板的告警
        :param sasAlarmInfo: SAS线缆告警信息
        :param expLocations: 需要检查的级联板
        :return:
        '''
        if not sasAlarmInfo or not expLocations:
            return True

        for exp in expLocations:
            for alarm in sasAlarmInfo:
                loc = alarm.get("location")
                sugLoc = alarm.get("suggestPeerLocation")
                if exp in loc or exp in sugLoc:
                    return False

        return True

    def checkPeerEndSasCableAlarms(self, errInfos):
        """屏蔽待更换本端告警，只检查冗余对端告警

        @param errInfos(list): 解析后的告警信息
        @return:
        """
        rowKey = self.args.get("rowKey")
        fruType = self.args.get("fruType")
        selRow = BaseFactory.json.toDict(self.context.get(rowKey))

        BaseFactory.log.info(self.context, "fruType is : %s" % fruType)
        switch = {
            self.EXPBOARD: self._getExpBoardCtrlId,
            self.SAS_PORT: self._getSasPortCtrlId,
            self.INTF_MODULE: self._getInterfCtrlId,
            self.CONTROLLER: self._get_controller_ctrl_id
        }

        if not switch.get(fruType, None) or not selRow:
            return errInfos

        ctrId, slot = switch.get(fruType)(selRow)
        BaseFactory.log.info(self.context, "ctrl id is : %s, slot is : %s" % (ctrId, slot))

        maskErrs = list(
            filter(lambda err: self._filterPeerEndSasCableAlarms(ctrId, slot,
                                                                 [err.get("location"), err.get("suggestPeerLocation")]),
                   errInfos))
        return maskErrs

    def _get_controller_ctrl_id(self, sel_row):
        location = sel_row.get(self.LOCATION)
        ctrl_id = sel_row.get("id")
        slot = location
        return ctrl_id, slot

    def _getExpBoardCtrlId(self, selRow):
        location = selRow.get(self.LOCATION)
        ctrlId = ExpBordObjForLowAndMidEnd(self.context, location).getCtrlId()
        slot = location
        return ctrlId, slot

    def _getSasPortCtrlId(self, selRow):
        location = selRow.get(self.LOCATION, "")
        if "DAE" in location:
            ctrId, slot = self._getExpBoardCtrlId(selRow)
        elif "CTE" in location:
            ctrId = self._getBackPortCtrlId(location)
            slot = location
        else:
            ctrId, slot = "", ""
        return ctrId, slot

    def _getInterfCtrlId(self, selRow):
        pId = selRow.get(self.PARENT_ID, "")
        location = selRow.get(self.LOCATION, "")
        if self.isCtrlIdValid(pId):
            return pId, location
        return "", location

    def _getBackPortCtrlId(self, location):
        if "CTE" in location:
            ctrl_id = "{}{}".format(location[3], location[5])
        else:
            ctrl_id = ""
        return ctrl_id

    def _filterPeerEndSasCableAlarms(self, ctrlId, slot, locs):
        """筛选出冗余对端告警

        @param ctrlId(str): 待更换端控制器ID
        @param locs(list): 告警中的位置信息
        @return: True - 需要筛选出来的对端告警 False-不需要筛选的对端告警
        """

        for loc in locs:

            if not loc:
                continue
            BaseFactory.log.info(self.context, "ctrlId: {}, alarm loc: {}".format(ctrlId, loc))
            # 获取告警所在控制器ID
            alarmCtrlId, alarmSlot = self._getSasPortCtrlId({self.LOCATION: loc})

            # 获取冗余控制器ID
            redauntCtrlId = LoopCheckFactory.Loop.getRedundantCntrId(self.context, ctrlId)
            # 没有冗余控制器场景
            if not redauntCtrlId:
                return False

            BaseFactory.log.info(self.context, "alarm ctrl ID is : %s" % alarmCtrlId)
            BaseFactory.log.info(self.context, "redaunt ctrl ID is : %s" % redauntCtrlId)
            if alarmCtrlId == redauntCtrlId:
                return True
        return False

    def isCtrlIdValid(self, ctrId):
        if ctrId.isdigit() or (len(ctrId) != 2):
            return False
        return True

    def filter_current_component_alarms(self, alarm_info_list):
        row_key = self.args.get("rowKey")
        fru_type = self.args.get("fruType")
        sel_row = BaseFactory.json.toDict(self.context.get(row_key))

        BaseFactory.log.info(self.context, "fruType is : {}".format(fru_type))
        switch = {
            self.EXPBOARD: self._getExpBoardCtrlId,
            self.SAS_PORT: self._getSasPortCtrlId,
            self.INTF_MODULE: self._getInterfCtrlId,
            self.CONTROLLER: self._get_controller_ctrl_id
        }

        if not switch.get(fru_type, None) or not sel_row:
            return alarm_info_list

        ctr_id, slot = switch.get(fru_type)(sel_row)
        BaseFactory.log.info(self.context, "high end alarms ctrl id is : {}, slot is : {}".format(ctr_id, slot))
        return [
            alarm
            for alarm in alarm_info_list
            if slot not in alarm.get("location", "")
            and slot not in alarm.get("suggestPeerLocation", "")
        ]
