# -*- coding: UTF-8 -*-
import re

from cbb.business.operate.fru.common import common
from cbb.business.operate.fru.common.checkItem import compute_storage_common
from cbb.frame.context import contextUtil
from cbb.frame.base import baseUtil

HMM_ID_REG = re.compile("\d+")

NEED_CHECK_HEALTH_URI = [
    "/redfish/v1/Systems",
    "/redfish/v1/Managers",
    "/redfish/v1/Chassis",
]

HMM_ID_CTRL_ID_MAPPING = {
    "1": "0A",
    "2": "0B"
}


class CheckBmcHealthStatus:
    """
    BMC状态检查
    """

    def __init__(self, context):
        self.context = context
        self.lang = contextUtil.getLang(context)
        self.bmc_hmm_query_uri = "/redfish/v1/systems"
        self.logger = baseUtil.getLogger(context.get("logger"), __file__)

    @staticmethod
    def get_hmm_member(members):
        for member in members:
            if "HMM" in member.get("@odata.id", ""):
                return member.get("@odata.id", "")
        return ""

    def execute(self, pre_check=True):
        """
        管理资源健康状态：redfish/v1/managers/HMMx
        系统资源健康状态：redfish/v1/systems/HMMx
        机箱健康状态：redfish/v1/chassis/HMMx
        :return:
        """
        redfish_conn, hmm_id = self.get_bmc_redfish_conn(pre_check)
        self.logger.logInfo(redfish_conn)
        return self.check_bmc_status(redfish_conn)

    def get_bmc_redfish_conn(self, pre_check):
        """
        获取BMC redfish 连接和HHM id
        :return: redfish 连接，HHM id
        """
        selected_info = contextUtil.getSelectedItem(self.context, "input_selectfru_controllerSelectedRow")
        ctrl_id = selected_info.get("id")
        self.logger.logInfo("ctrl_id:{}".format(ctrl_id))
        # 获取所有BMC板子信息，并判断是否是对端的。
        boards = compute_storage_common.get_all_bmc_manage_boards(self.context)
        board_ips = compute_storage_common.get_fcv_ctrl_enc_manage_board_ips(self.context)
        manage_board_info_list = [board_info for board_info in boards if board_info.ip in board_ips]
        peer_redfish_conn = None
        peer_hmm_id = None
        local_redfish_conn = None
        local_hmm_id = None
        for board in manage_board_info_list:
            redfish_conn = compute_storage_common.get_redfish_connection_by_bmc_manager_board(self.context, board)
            record = compute_storage_common.get_info_by_redfish_rest_conn(redfish_conn, self.bmc_hmm_query_uri)
            hmm_uri = self.get_hmm_member(record.get("Members", [{}]))
            hmm_id = hmm_uri.split("/")[-1]
            node_id = HMM_ID_REG.findall(hmm_id)
            if not node_id:
                continue
            current_ctrl_id = HMM_ID_CTRL_ID_MAPPING.get(node_id[0], "")
            # 如果0A或0B不在更换的控制器ID中，则是对端的BMC板子，则继续检查状态
            if current_ctrl_id in ctrl_id:
                # 下电时需要使用当前XPU对应的BMC接口下电
                self.context["XPU_BMC_MANAGE_BOARD"] = board
                local_redfish_conn, local_hmm_id = redfish_conn, hmm_id
            else:
                peer_redfish_conn, peer_hmm_id = redfish_conn, hmm_id
        # 更换前检查返回对端redfish 连接和HHM id， 更换后检查返回本端redfish 连接和HHM id
        if pre_check:
            return peer_redfish_conn, peer_hmm_id
        return local_redfish_conn, local_hmm_id

    def check_bmc_status(self, redfish_conn):
        result_dict = {"flag": True, "errMsg": "", "suggestion": ""}
        for uri in NEED_CHECK_HEALTH_URI:
            if not self.get_component_status(redfish_conn, uri) == "OK":
                result_dict["errMsg"], result_dict["suggestion"] = common.getMsg(
                    self.lang, "cmp.storage.bmc.status.exception")
                contextUtil.handleFailure(self.context, result_dict)
                return False
        contextUtil.handleSuccess(self.context)
        return True

    def get_component_status(self, redfish_conn, uri):
        """
        获取组件健康状态
        :param redfish_conn: redfish 连接
        :param uri: 组件uri
        :return: 健康状态
        """
        record = compute_storage_common.get_info_by_redfish_rest_conn(redfish_conn, uri)
        hmm_uri = self.get_hmm_member(record.get("Members", [{}]))
        self.logger.logInfo("rui: {}, res:{}".format(uri, record))
        record = compute_storage_common.get_info_by_redfish_rest_conn(redfish_conn, hmm_uri)
        self.logger.logInfo("hmm_uri: {}, res:{}".format(hmm_uri, record))
        return record.get("Status", {}).get("Health", "")
