﻿# -*- coding: UTF-8 -*-
import json
import time

from cbb.business.operate.fru.common import common
from cbb.frame.base import baseUtil
from cbb.frame.context import contextUtil
from cbb.frame.rest import restData
from cbb.frame.rest import restUtil

'''
健康状态检查
检查已更换硬盘的健康状态是否正常。
'''
NOT_MATCH_DISK_TYPE_ALARM = ["64425165079"]  # 保险箱盘类型不匹配告警0xF000A0117
NOT_EXIST_DISK_ALRAM = "1077936769"
DISK_TYPE_MATCH_ALARM_WAIT_TIME_SECS = 90
DISK_TYPE_MATCH_ALARM_OCCUR_TIME_SECS = 30
CHECK_DISK_TIMEOUT_SECS = 100
PROGRESS_INTERVAL_SECS = 1
CHECK_INTERVAL_SECS = 3


def execute(context, select_id):
    """
    :param context: 上下文
    :param select_id: 保持数据的ID
    :return:
    """

    lang = contextUtil.getLang(context)
    logger = baseUtil.getLogger(contextUtil.getLogger(context), __file__)
    disks = json.loads(context.get(select_id))
    total_time = CHECK_DISK_TIMEOUT_SECS + DISK_TYPE_MATCH_ALARM_WAIT_TIME_SECS
    common.threadUpProcess(context, total_time, PROGRESS_INTERVAL_SECS)
    try:
        flag, err_msg, sug = check_alarm(context)
        if not flag:
            contextUtil.setResult(context, False, err_msg, sug)
            return

        sug_list = []
        normal_disks = set()
        begin_time = time.time()
        while True:
            use_time = time.time() - begin_time
            if use_time > CHECK_DISK_TIMEOUT_SECS:
                err_msg, _ = common.getMsg(lang, "disk.health.status.abnormal")
                suggestion = "<br>".join(sug_list)
                contextUtil.setResult(context, False, err_msg, suggestion)
                break

            result = True
            sug_list = []
            rest = contextUtil.getRest(context)
            for disk in disks:
                disk_id = disk.get("id")
                if disk_id in normal_disks:
                    continue

                flag, sug = check_one_disk(rest, lang, logger, disk)
                result = result and flag
                if not flag:
                    sug_list.append(sug)
                    continue
                normal_disks.add(disk_id)

            if result:
                return contextUtil.setResult(context, True, "", "")

            baseUtil.safeSleep(CHECK_INTERVAL_SECS)
        return
    finally:
        common.finishProcess(context)


def check_one_disk(rest, lang, logger, disk):
    disk_id = disk.get("id")
    disk_loc = disk.get("location")
    try:
        rec = restUtil.CommonRest.getDiskById(rest, disk_id)
        hs = restUtil.CommonRest.getRecordValue(rec,
                                                restData.RestCfg.Disk.
                                                HEALTH_STATUS)
        if hs != restData.RestCfg.HealthStatusEnum.NORMAL:
            err_key = "one.disk.health.status.abnormal"
            _, sug = common.getMsg(lang, err_key, suggestionArgs=disk_loc)
            return False, sug

        return True, ""
    except Exception as ex:
        logger.error("get disk info exception:%s" % str(ex))
        if ex.args:
            err_code = ex.args[0]
            if NOT_EXIST_DISK_ALRAM in err_code:
                _, sug = common.getMsg(lang, "pre.copy.no.disk",
                                       suggestionArgs=disk_loc)
        _, sug = common.getMsg(lang, "get.disk.info.error",
                               suggestionArgs=disk_loc)
        return False, sug


def check_alarm(context):
    lang = contextUtil.getLang(context)
    sys_time = contextUtil.getItem(context, "START_SYS_TIME")
    start_time = time.time()
    rest = contextUtil.getRest(context)
    sys_rec = restUtil.CommonRest.getSysUTCTimeRecord(rest)
    cur_systime = restUtil.Tlv2Rest.getRecordValue(sys_rec,
                                                   restData.Sys.
                                                   SystemUtcTime.
                                                   CMO_SYS_UTC_TIME)
    while (time.time() - start_time) < DISK_TYPE_MATCH_ALARM_WAIT_TIME_SECS:
        baseUtil.safeSleep(DISK_TYPE_MATCH_ALARM_OCCUR_TIME_SECS)
        rest = contextUtil.getRest(context)
        recs = restUtil.CommonRest.getSysEvent(rest, sys_time, cur_systime,
                                               event_type=restData.RestCfg.
                                               SpecialUri.
                                               QUERY_CURRENT_ALARM_INFO)
        for rec in recs:
            alarm_id = restUtil.CommonRest.getRecordValue(rec,
                                                          restData.RestCfg.
                                                          Alarm.EVENTID)
            if alarm_id in NOT_MATCH_DISK_TYPE_ALARM:
                err_msg, sug = common.getMsg(lang, "disk.type.not.match")
                return False, err_msg, sug

    return True, "", ""
