# -*- coding: UTF-8 -*-

#  Copyright (c) Huawei Technologies Co., Ltd. 2019-2021. All rights reserved.

import json
import time

from java.lang import Exception as JException

from cbb.frame.base import jsonUtil
from cbb.frame.base.funcUtils import wrap_all_exception_logged


class RestConnection(object):
    AUTHENTICATE_URI = r"https://%s:%s/deviceManager/rest/%s/sessions"
    REST_BASE_URI = r"https://%s:%s/deviceManager/rest/%s/"
    RE_CONNECTION_TIMES = 3
    PROXY_OF_SVP_CONNECTION_TIMES = 2
    RE_CONNECTION_INTERVAL = 5
    REST_PORT_DEFAULT = 8088
    REST_SCOPE_DEFAULT = "0"
    REST_CAN_NOT_CONNECT_CODE = "1073949185"  # 与设备通信异常，请检查网络连接或设备状态是否正常
    REST_INVALID_CODE = ["-401", "1077949069"]
    _instance = {}
    _rest = None
    REST_CMD_TIMEOUT = 5*60


    def __new__(cls, *args, **kwargs):
        sn = args[2]
        if sn not in cls._instance:
            # 根据设备单例
            cls._instance[sn] = super(RestConnection, cls).__new__(cls, *args,
                                                                   **kwargs)
        return cls._instance.get(sn)

    def __init__(self, connectorFactory, ip, devSN, restPort=REST_PORT_DEFAULT, timeOut=None):
        self.connectorFactory = connectorFactory
        self.ip = ip
        self.restPort = restPort
        self.devSN = devSN
        self.baseUri = self.REST_BASE_URI % (self.getUrlIp(ip), str(restPort), str(devSN))
        self.loginUri = self.AUTHENTICATE_URI % (self.getUrlIp(ip), str(restPort), str(devSN))
        self._rest = None

        # 初始连接时设置超时时间，针对初次建立连接、重连及已连接后的所有请求都有效。
        self.responseTimeout = timeOut

    def create(self, user, pawd, scope=REST_SCOPE_DEFAULT, logger=None):
        """
        @summary: 创建rest连接
        @param user: 用户名
        @param pawd: 密码
        @param scope: 用户类型
        @param logger:
        @return:
            True:建立成功
            False:建立失败   
        """
        successFlag = True
        exception = ""
        reConnectionTimes = 0
        while self._rest is None and reConnectionTimes < \
                RestConnection.RE_CONNECTION_TIMES + RestConnection.PROXY_OF_SVP_CONNECTION_TIMES:
            try:
                successFlag = False
                restConnector = self.connectorFactory.createRestConnector(self.loginUri, user, pawd, scope)
                restConnection = restConnector.getConnection()
                if self.responseTimeout is not None:
                    restConnection.setTimeout(self.responseTimeout)
                if reConnectionTimes % 2 == 0:
                    content = restConnection.login().getContent()
                else:
                    login_result = restConnection.loginSvpProxy()
                    if login_result:
                        content = login_result.getContent()
                    else:
                        reConnectionTimes += 1
                        continue
                isJson, resp = self.checkJsonFormat(content)
                responseInfo = resp if isJson else {}
                data = responseInfo.get("data", {})
                iBaseToken = data.get("iBaseToken", None)
                if iBaseToken is not None:
                    successFlag = True
                    self._rest = restConnection
                    break
                try:
                    restConnection.closeSession()
                except Exception as ex:
                    if logger:
                        logger.error(ex)
                reConnectionTimes += 1
                # 不能使用baseUtil中的sleep方法，会出现循环引用问题
                safeSleep(RestConnection.RE_CONNECTION_INTERVAL)
            except Exception as e:
                exception = e
                reConnectionTimes += 1
                safeSleep(RestConnection.RE_CONNECTION_INTERVAL)

        del pawd
        if not successFlag:
            raise Exception(self.REST_CAN_NOT_CONNECT_CODE, unicode(exception))
        return successFlag

    @wrap_all_exception_logged(logger=None)
    def close(self):
        if self._rest is not None:
            try:
                self._rest.closeSession()
            finally:
                self._rest = None

    def getRestConnection(self, user, pawd, scope):
        try:
            if self._rest is None or not self.checkConnetionNormal():
                self.close()
                self.create(user, pawd, scope)
        except Exception as e:
            raise Exception(self.REST_CAN_NOT_CONNECT_CODE, unicode(e))

        return self._rest

    def getBaseUri(self):
        return self.baseUri

    def getRest(self):
        return self._rest

    def checkConnetionNormal(self):
        uri = self.baseUri + "system/"
        paramsStr = ""
        try:
            records = jsonUtil.jsonStr2Dict(
                self._rest.execGet(uri, paramsStr).getContent())
            error = records["error"]
            error_code = str(error.get("code", "")).strip()
            if error_code in self.REST_INVALID_CODE:
                return False
            return True
        except:
            return False

    def getUrlIp(self, ip):
        if ":" in str(ip):
            return "[%s]" % str(ip)
        return str(ip)

    def checkJsonFormat(self, data):
        resp = {}
        try:
            resp = json.loads(data)
        except ValueError:
            return False, resp
        return True, resp

    @wrap_all_exception_logged(logger=None)
    def setReadDataTimeout(self, timeout):
        '''
        @summary: 设置读取数据的超时时间，REST连接已建立情况下修改超时时间
        （仅本次连接生效，连接断开后无效）
        :param timeout:
        '''
        if self._rest is not None:
            self._rest.getSession().setReadDataTimeout(timeout)

    def setReadDataDefaultTimeout(self):
        '''
        @summary: 设置读取数据的超时时间，默认为5分钟
        '''
        self.setReadDataTimeout(self.REST_CMD_TIMEOUT)


def self_sleep(sec):
    """
    睡眠
    :param sec: 睡眠时间（秒）
    :return:
        True: 执行成功
        False: 执行失败
    """
    try:
        time.sleep(sec)
    except (Exception, JException):
        return False
    return True


def safeSleep(seconds):
    """
    安全睡眠时间
    :param seconds:  seconds为睡眠时间，单位：秒；数据类型：整数或小数
    :return:
        True: 执行成功
        False: 执行失败
    """
    # noinspection PyBroadException
    try:
        if type(seconds) not in [int, long, float] or seconds <= 0:
            return False

        startTime = time.clock()
        while True:
            if (time.clock() - startTime) >= seconds:
                return True

            # 睡眠一下，避免长时间占用cpu，该时间设置过长会影响睡眠时间精度
            # noinspection PyBroadException
            self_sleep(0.1)
    except (Exception, JException):
        return False
