# -*- coding: UTF-8 -*-
import re
import traceback

from com.huawei.ism.exception import \
    IsmException  # IGNORE:F0401 #PyLint中引用会报错，但实际可引用，故屏蔽
from com.huawei.ism.tlv import \
    TLVUtils  # IGNORE:F0401 #PyLint中引用会报错，但实际可引用，故屏蔽
from com.huawei.ism.tlv.bean import \
    Param  # IGNORE:F0401 #PyLint中引用会报错，但实际可引用，故屏蔽
from com.huawei.ism.tlv.docoder import \
    ParamType  # IGNORE:F0401 #PyLint中引用会报错，但实际可引用，故屏蔽
from com.huawei.ism.tlv.lang import \
    UnsignedInt32  # IGNORE:F0401 #PyLint中引用会报错，但实际可引用，故屏蔽
from com.huawei.ism.tool.framework.platform.exception import ToolException
from java.lang import Exception as JException
from java.lang import String
from java.util import ArrayList as JavaList

from cbb.frame.cli import cliUtil
from cbb.frame.context import contextUtil
from cbb.frame.tlv.baseFactory import log as Log
from cbb.frame.tlv.baseFactory import persist as Persist
from cbb.frame.tlv.baseFactory import const as Const
from cbb.frame.tlv.tlvEnumFactory import TlvEnum

TLV_CAN_NOT_EXECUTE = "1077936892"  # 命令执行失败


class Tlv():
    """Tlv执行类：已剥离与设备相关的枚举信息

    """
    ________TEST_ZONE________ = None

    ________FUNC_ZONE________ = None

    @staticmethod
    def getBatch(context, cmd_type, param_list, logCmdResult=True):
        """批量查询

        :param context: 上下文对象
        :param cmd_type: TLV命令
        :param param_list: 参数列表
        :param logCmdResult: 是否打印相应结果
        :return:
        """
        # 创建TLV连接
        tlv_conn = Tlv.Conn.getConn(context)
        if None is tlv_conn:
            return None

        # 执行TLV命令
        try:
            # 1 打印命令类型和查询的对象类型，方便问题定位
            # 构造查询对象的描述
            Log.info(context, "TLV_CMD_INFO(getBatch): cmdType="
                     + str(cmd_type) + ", paramList=" + str(param_list))

            # 2 执行命令获取执行结果
            recs = tlv_conn.getBatch(cmd_type, param_list,
                                     Tlv.Const.TIMEOUT_DEFAULT)
            if logCmdResult:
                Tlv.Record.logRecList(context, recs,
                                      "TLV_CMD_RESULT getBatch as follows:")
            else:
                Log.info(context, "TLV_CMD_RESULT getBatch is ignored.")
            # 如果recs的最后一行为结束标识行，则清除它
            if None is not recs and 0 < recs.size():
                tail_rec = recs.get(recs.size() - 1)
                if tail_rec.getParams().size() == 0:
                    raise Exception.new_tool_exception(
                        u"未处理的异常：TLV命令执行结果为空。",
                        Const.HELP_DESC_ZH,
                        "Unprocessed exception: the executed "
                        + "result of TLV command is empty.",
                        Const.HELP_DESC_EN,
                        Tlv.Const.ERRID_EMPTY_RESULT)
                head_index0 = tail_rec.getParams().get(0).getIndex()
                head_index1 = ""
                Log.info(context, "tailRec.getParams=%s" %
                         tail_rec.getParams())
                if tail_rec.getParams().size() > 1:
                    head_index1 = tail_rec.getParams().get(1).getIndex()
                if Tlv.Const.RECORD_ENDFLAG in [head_index0, head_index1]:
                    Log.info(context, "remove the last useless record: "
                             + str(tail_rec))
                    recs.remove(recs.size() - 1)

            # 3 返回结果
            if None is recs:  # 为None的话，只可能为程序异常所致，直接抛出异常。
                raise Exception.new_tool_exception(
                    u"TLV命令执行结果为None。",
                    Const.HELP_DESC_ZH,
                    "The executed result of TLV command is None.",
                    Const.HELP_DESC_EN,
                    Tlv.Const.ERRID_NONE_RESULT)
            return recs

        except IsmException as ex:
            Log.error(context,
                      "An ISM exception occured when exec Tlv.execmd.")
            # 抛出异常
            err_code = ex.getErrorId()
            err_msg = ex.getErrorMessage()
            err_msg = err_msg.rstrip(u"。").rstrip(".")
            Log.error(context, "exception info: " + str(ex))
            raise Exception.new_tool_exception(
                u"TLV命令执行失败。原因为" + err_msg + u"。",
                u"请检查网络连接或设备状态是否正常。",
                "Failed to execute the TLV command. Cause is " + err_msg + ".",
                "Retry when the network and device are normal.", err_code)

        except ToolException as et:
            # 记录错误并直接抛出异常
            Log.error(context,
                      "An ToolException [%s] occured when exec Tlv.execmd:"
                      % et)
            raise Exception.get_tool_exception_by_err_id(et)

        except Exception as e:
            Log.error(context,
                      "An tool exception occured when exec Tlv.execmd:" + str(
                          e))
            # 抛出异常
            if len(e.args) >= len(Exception.args_id) and \
                    (e.args[Exception.args_id["ERRID"]] in
                     [Tlv.Const.ERRID_EMPTY_RESULT,
                      Tlv.Const.ERRID_NONE_RESULT]):
                raise e
            else:
                raise Exception.new_tool_exception(
                    u"TLV命令执行失败。",
                    Const.HELP_DESC_ZH,
                    "Failed to execute the TLV command.",
                    Const.HELP_DESC_EN)

    @staticmethod
    def execmd(context, isBatch, cmdType, paramList,
               logCmdResult=True, isExecCmdToSvp=False):
        """执行TLV命令

        :param context: 上下文对象
        :param isBatch: 返回结果是否为多条记录
        :param cmdType: 具体的TLV命令
        :param paramList: 参数列表
        :param logCmdResult: 是否打印响应
        :param isExecCmdToSvp: 是否是在SVP上执行
        :return:
        """
        # 创建TLV连接
        if isExecCmdToSvp:
            tlvConn = Tlv.Conn.getTlvConnectionSVP(context)
        else:
            tlvConn = Tlv.Conn.getConn(context)

        if tlvConn is None:
            return None

        # 执行TLV命令
        try:
            # 1 打印命令类型和查询的对象类型，方便问题定位
            # 记录命令信息
            Log.info(context, "TLV_CMD_INFO: cmdType="
                     + str(cmdType) + ", paramList=" + str(paramList))

            # 按isBatch调用不同的TLV接口
            if isBatch:
                recs = tlvConn.getBatch(cmdType, paramList,
                                        Tlv.Const.TIMEOUT_DEFAULT)
                if logCmdResult:
                    Tlv.Record.logRecList(
                        context,
                        recs,
                        "TLV_CMD_RESULT bacthNext as follows:")
                else:
                    Log.info(context, "TLV_CMD_RESULT bacthNext is ignored.")
                # 如果recs的最后一行为结束标识行，则清除它
                if recs is not None and recs.size() > 0:
                    tail_rec = recs.get(recs.size() - 1)
                    if tail_rec.getParams().size() == 0:
                        raise
                    head_index = tail_rec.getParams().get(0).getIndex()
                    head_index1 = ""
                    Log.info(context,
                             "tailRec.getParams=%s" % tail_rec.getParams())
                    if tail_rec.getParams().size() > 1:
                        head_index1 = tail_rec.getParams().get(1).getIndex()
                    if Tlv.Const.RECORD_ENDFLAG in [head_index, head_index1]:
                        Log.info(context,
                                 "remove the last useless record: "
                                 + str(tail_rec))
                        recs.remove(recs.size() - 1)
            else:
                recs = tlvConn.invoke(cmdType,
                                      paramList,
                                      Tlv.Const.TIMEOUT_DEFAULT)
                Log.info(context, "TLV_CMD_RESULT invoke: " + str(recs))

            # 3 返回结果
            if recs is None:  # 为None的话，只可能为程序异常所致，直接抛出异常。
                raise
            return recs

        except IsmException as ex:
            Log.error(context,
                      "An ISM exception occured when exec Tlv.execmd.")
            raise ex
        except (Exception, JException) as e:
            raise IsmException(int(TLV_CAN_NOT_EXECUTE), e)

    class Param():
        """TLV参数构造类

        """
        TYPE_UINT = 0  # 无符号整形
        TYPE_STR = 1  # 字符串
        TYPE_ENUM = 2  # 枚举(Fru的type字段需用这种类型)
        TYPE_BOOL = 3  # boolean type

        @staticmethod
        def getParam(index, type, value):
            """生成一个TLV参数

            :param index: =参数在命令中的位置
            :param type: TLV参数类型
            :param value: TLV参数值
            :return: TLV参数对象
            """
            # 参数校验
            if index < 0:
                return None
            if type != Tlv.Param.TYPE_UINT \
                    and type != Tlv.Param.TYPE_STR \
                    and type != Tlv.Param.TYPE_ENUM \
                    and type != Tlv.Param.TYPE_BOOL:
                return None
            # 生成参数
            # 生成参数
            param = None
            if type == Tlv.Param.TYPE_UINT:
                param = Param(index, ParamType.UNSIGN_INT, UnsignedInt32(
                    value))  # @UndefinedVariable #UNSIGN_INT存在，故屏蔽PyDev报错
            elif type == Tlv.Param.TYPE_STR:
                param = Param(index, ParamType.STRING, str(
                    value))  # @UndefinedVariable #STRING存在，故屏蔽PyDev报错
            elif type == Tlv.Param.TYPE_ENUM:
                param = Param(index, ParamType.ENUM, UnsignedInt32(
                    value))  # @UndefinedVariable #ENUM存在，故屏蔽PyDev报错
            elif type == Tlv.Param.TYPE_BOOL:
                param = Param(index, ParamType.BOOL, value)
            # 返回参数
            return param

        @staticmethod
        def getParamList(*all_param):
            """生成TLV参数列表

            :param all_param: 所有的TLV参数，中间以逗号隔开
            :return: TLV参数列表
            """
            paramList = []
            for param in all_param:
                paramList.append(param)
            return TLVUtils.paramList(
                paramList)  # @UndefinedVariable #paramList存在，故屏蔽PyDev报错

        # 查询条件类型，供生成查询条件字符串时使用
        QueryType = {"LIKE": "LIKE",  # 相似
                     "EQUAL": "EQUAL",  # 等于
                     "NOT": "NOT",  # 不等于
                     "RANGE": "RANGE"  # 位于左闭右开的区间，传入的参数为列表
                     }
        # 排序条件类型，供生成查询条件字符串时使用
        SortType = {"ASC": "ASC",  # 升序
                    "DESC": "DESC"  # 降序
                    }

        @staticmethod
        def getQueryConditionString(query_dict_list, sort_dict, range_dict):
            """生成批量查询命令中需要的条件字符串

            :param query_dict_list: 查询条件列表
            :param sort_dict: 二元字典，不存在则为None
            :param range_dict: 二元字典，不存在则为None
            :return: queryCondition字段所需的字符串
            """
            ret_condition = {}
            # 构造查询字符串
            if query_dict_list is not None:
                query_dict = {}
                for cond in query_dict_list:
                    query_type = cond["type"]
                    if Tlv.Param.QueryType["LIKE"] == query_type:
                        query_dict[str(cond["index"])] = \
                            "LIKE" + str(cond["value"])
                    elif Tlv.Param.QueryType["EQUAL"] == query_type:
                        query_dict[str(cond["index"])] = str(cond["value"])
                    elif Tlv.Param.QueryType["NOT"] == query_type:
                        query_dict[str(cond["index"])] = \
                            "NOT" + str(cond["value"])
                    elif Tlv.Param.QueryType["RANGE"] == query_type:
                        range = cond["value"]
                        query_dict[str(cond["index"])] = \
                            "[" + str(range[0]) + "," + str(range[1]) + ")"
                    else:
                        return "Invalid query condition: " + str(cond)
                ret_condition["query"] = [query_dict]
            # 构造排序字符串
            if sort_dict is not None:
                str_sort = str(sort_dict["index"]) + str(sort_dict["type"])
                ret_condition["sortby"] = str_sort
            # 构造区间字符串
            if range_dict is not None:
                str_range = str(range_dict["start"]) + "-" \
                            + str(range_dict["end"])
                ret_condition["range"] = str_range
            # 采用Python的str将条件字典转换为字符串，并将其中的单引号替换为双引号
            return str(ret_condition).replace('\'', '\"')

    class Conn():
        """TLV连接管理类

        """
        # 持久化数据保存KEY
        PERSIST_CONN = "TLV_CONNECTION"

        @staticmethod
        def chgContextLoginInfo(context, ip=None):
            """更换工具上下文中的设备登录信息

            :param context: 所有字段均为选填，不更新时在参数中指定为None即可
            :param ip: ip
            :return: 操作是否成功
            """
            # 获取设别信息
            dev = context.get("dev")
            if dev is None:
                return False

            # 将不为None的登录信息更新到dev中
            if ip is not None:
                dev["ip"] = ip

            # 刷新上下文中的设备对象信息
            context["dev"] = dev
            return True

        @staticmethod
        def getConnByIP(context, ip, user, pawd, port):
            """获取指定设备的TLV连接

            :param context: 上下文
            :param ip: ip
            :param user: 用户名
            :param pawd: 密码
            :param port: 端口
            :return:
            """
            # 从上下文中获取指定IP的TLV连接器
            tlv_connector = Persist.get_object(context,
                                               Tlv.Conn.PERSIST_CONN, ip)

            if tlv_connector is not None:
                # 直接从已有TLV连接器中获取连接
                Log.info(context,
                         "tlv connector is already existed. ip=" + ip)
                try:
                    tlv_conn = tlv_connector.getConnection()

                except ToolException, e:
                    # 记录错误并直接抛出异常
                    Log.error(context,
                              "fail to get tlv connection. ip="
                              + ip
                              + ", the dev is abnormal or the network "
                                "is abnormal.")
                    raise Exception.get_tool_exception_by_err_id(e)

            else:
                # 创建新的TLV连接器和连接
                # 获取TLV连接参数
                conn_factory = contextUtil.getConnectorFactoryObj(context)
                ssl = True  # 连接是否使用安全套接字连接

                # 循环连接(框架异常时直接抛出异常)
                counter = 0
                MAX_RETRAY_TIME = 3
                while True:
                    counter += 1

                    try:
                        # 创建TLV连接
                        try:
                            cli = contextUtil.getCli(context)
                            cliUtil.openTlvChannel(cli)
                        except Exception:
                            Log.error(context, "open tlv channel exception")

                        # 中低端场景
                        if "," not in ip:
                            tlv_connector = \
                                conn_factory.createTlvConnector(ip,
                                                                port,
                                                                user,
                                                                pawd,
                                                                ssl)
                        # 高端场景
                        else:
                            temp_list = ip.split(",")
                            inner_ip_list = JavaList()
                            for inner_ip in temp_list:
                                inner_ip_list.add(String(inner_ip))
                            tlv_connector = conn_factory.createTlvConnector(
                                inner_ip_list, port, user, pawd, ssl)
                        del pawd
                        tlv_conn = tlv_connector.getConnection()
                        # 保存TLV连接（后续使用时不再创建）
                        Persist.set_object(context, Tlv.Conn.PERSIST_CONN, ip,
                                           tlv_connector)
                        Log.info(
                            context,
                            "succeed to create and save tlv connector. ip="
                            + ip)

                        # 连接正常，则退出循环连接
                        break

                    except ToolException, e:
                        # 记录错误并直接抛出异常
                        Log.error(
                            context,
                            "Framework Exception: fail to "
                            "create tlv connector. ip="
                            + ip
                            + ", errorid=" + e.getErrorId()
                            + ", errmsg=" + e.getErrMsg()
                            + ", do not retry.")
                        raise Exception.get_tool_exception_by_err_id(e)

                    except Exception:
                        # 记录错误并重试
                        counter = counter + 1
                        Log.error(context, "the " + str(counter)
                                  + "th failed to create tlv connector. ip="
                                  + ip
                                  + ", errinfo=%s" % traceback.format_exc()
                                  + ", try again...")

                    # 若三次重试仍未建立TLV连接，则报错
                    if counter > MAX_RETRAY_TIME:
                        tlv_conn = None
                        Log.error(context, "fail to create tlv connector. ip="
                                  + ip + ", met the max try time 3.")
                        raise Exception.new_tool_exception(
                            u"创建到设备" + ip + u"的TLV连接失败。",
                            Const.HELP_DESC_ZH,
                            "Failed to create the TLV connection to device "
                            + ip + ".",
                            Const.HELP_DESC_EN)
            # 返回连接创建结果
            return tlv_conn

        @staticmethod
        def getTlvConnectionSVP(context):
            """获取与SVP的连接

            :param context:
            :return:
            """
            IP_DEFAULT = "172.17.126.11"
            PORT_DEFAULT = "19002"
            DEVICE_TYPE = "svp"
            SN = contextUtil.getDevSN(context) + "svp"

            # 从上下文中获取指定IP的TLV连接器
            tlv_connector = Persist.get_object(context, Tlv.Conn.PERSIST_CONN,
                                               IP_DEFAULT)
            tlvConn = None

            if tlv_connector is not None:
                # 直接从已有TLV连接器中获取连接
                Log.info(context,
                         "tlv connector is already existed. ip=" + IP_DEFAULT)
                try:
                    tlvConn = tlv_connector.getConnectionSVP()

                except ToolException as e:
                    # 记录错误并直接抛出异常
                    Log.error(
                        context, "fail to get tlv connection. ip="
                                 + IP_DEFAULT
                                 + ", the dev is abnormal or "
                                   "the network is abnormal.")
                    raise Exception.get_tool_exception_by_err_id(e)

            else:
                # 创建新的TLV连接器和连接
                # 获取TLV连接参数
                conn_factory = contextUtil.getConnectorFactoryObj(context)

                # 循环连接(框架异常时直接抛出异常)
                counter = 0
                MAX_RETRAY_TIME = 3
                while True:
                    counter += 1

                    try:
                        # 创建TLV连接
                        tlv_connector = None

                        tlv_connector = conn_factory.createTlvConnectorSVP(
                            IP_DEFAULT, PORT_DEFAULT, "",
                            DEVICE_TYPE, SN)

                        tlvConn = tlv_connector.getConnectionSVP()
                        # 保存TLV连接（后续使用时不再创建）
                        Persist.set_object(context,
                                           Tlv.Conn.PERSIST_CONN,
                                           IP_DEFAULT,
                                           tlv_connector)
                        Log.info(
                            context,
                            "succeed to create and save tlv connector. ip="
                            + IP_DEFAULT)

                        # 连接正常，则退出循环连接
                        break

                    except ToolException as e:
                        # 记录错误并直接抛出异常
                        Log.error(
                            context,
                            "Framework Exception: fail to "
                            "create tlv connector. ip="
                            + IP_DEFAULT
                            + ", errorid="
                            + e.getErrorId()
                            + ", errmsg="
                            + e.getErrMsg()
                            + ", do not retry.")
                        raise Exception.get_tool_exception_by_err_id(e)

                    except Exception as e:
                        # 记录错误并重试
                        counter = counter + 1
                        Log.error(context, "the "
                                  + str(counter)
                                  + "th failed to create tlv connector. ip="
                                  + IP_DEFAULT
                                  + ", errinfo="
                                  + str(e)
                                  + ", try again...")

                    # 若三次重试仍未建立TLV连接，则报错
                    if counter > MAX_RETRAY_TIME:
                        tlvConn = None
                        Log.error(context,
                                  "fail to create tlv connector. ip="
                                  + IP_DEFAULT
                                  + ", met the max try time 3.")
                        raise Exception.new_tool_exception(
                            u"创建到设备"
                            + IP_DEFAULT
                            + u"的TLV连接失败。",
                            Const.HELP_DESC_ZH,
                            "Failed to create the TLV connection to device "
                            + IP_DEFAULT + ".",
                            Const.HELP_DESC_EN)
            # 返回连接创建结果
            return tlvConn

        @staticmethod
        def getConn(context):
            """获取当前设备TLV连接

            :param context:
            :return:
            """
            return contextUtil.getOldTlv(context)

        @staticmethod
        def getConnThroughSpecificIp(context, ip):
            """与制定IP建立TLV连接

            :param context:
            :param ip:
            :return:
            """
            user = contextUtil.getNewDevObj(context).get("user")
            pawd = contextUtil.getNewDevObj(context).get("pawd")
            tlv_port = Tlv.Const.TLVPORT_DEFAULT
            Log.info(context,
                     "ready to get a new tlv connector. login info: ip="
                     + str(ip)
                     + ", user=" + str(user) + ", tlvPort=" + str(tlv_port))

            current_ip = contextUtil.getNewDevObj(context).get("ip")
            is_delete = Persist.del_object(context,
                                           Tlv.Conn.PERSIST_CONN, current_ip)
            Log.info(context, "delete current link ," + str(is_delete))

            if is_delete:
                contextUtil.getNewDevObj(context)["ip"] = ip
                conn = Tlv.Conn.getConnByIP(context, ip, user, pawd, tlv_port)
                del pawd
            return conn

    class Record():
        """TLV常量类

        """

        @staticmethod
        def getItemCount(record, item_id):
            """获取Record中的元素个数

            :param record: TLV执行后返回的记录对象
            :param item_id: TLV参数项位置
            :return: Record的元素个数
            """
            return record.getFieldCount()

        @staticmethod
        def isItemExisted(record, item_id):
            """判断Record中指定ID的元素是否存在

            :param record: TLV执行后返回的记录对象
            :param item_id: TLV参数项位置
            :return: 是否存在。True=存在； False=不存在
            """
            # 尝试获取指定ID的元素，若能获取到，则说明存在；不存在时将抛出异常
            try:
                obj = Tlv.Record.getValue(record, item_id)
                if obj is not None:
                    is_existed = True
                else:
                    is_existed = False
            except Exception:
                is_existed = False

            return is_existed

        @staticmethod
        def isItemExisted_TEST(context, record, itemId):
            """isItemExisted函数的测试函数

            :param context:
            :param record:
            :param itemId:
            :return:
            """
            # 尝试获取指定ID的元素，若能获取到，则说明存在；不存在时将抛出异常
            isExisted = False
            try:
                obj = Tlv.Record.getValue(record, itemId)
                Log.info(context, "record=" + str(record))
                Log.info(context,
                         "itemId=" + str(itemId) + ", obj=" + str(obj))
                if obj is not None:
                    isExisted = True
                else:
                    isExisted = False
            except Exception:
                isExisted = False

            Log.info(context, "isExisted=" + str(isExisted))
            return isExisted

        @staticmethod
        def getValue(record, itemId):
            """从Record中获取参数

            :param record: TLV执行后返回的记录对象
            :param itemId: TLV参数项位置
            :return: TLV参数项值
            """
            return record.getParamValue(itemId)

        @staticmethod
        def getStrValue(record, item_id):
            """从Record中获取字符串类参数

            :param record: TLV执行后返回的记录对象
            :param item_id: TLV参数项位置
            :return: TLV参数项值
            """
            try:
                return str(record.getParamStrValue(item_id))
            except Exception:
                return str(record.getParamStrValue(item_id).encode("utf-8"))

        @staticmethod
        def getIntValue(record, item_id):
            """从Record中获取Int32型参数

            :param record: TLV执行后返回的记录对象
            :param item_id: TLV参数项位置
            :return: TLV参数项值
            """
            return record.getParamIntValue(item_id).intValue()

        @staticmethod
        def getBoolValue(record, item_id):
            """或者BOOL类型值

            :param record:
            :param item_id:
            :return:
            """
            return record.getParamStrValue(item_id)

        @staticmethod
        def getLongValue(record, itemId):
            """从Record中获取Int64型参数

            :param record: TLV执行后返回的记录对象
            :param itemId: TLV参数项位置
            :return: TLV参数项值
            """
            return record.getParamIntValue(itemId).longValue()

        # 条件类型，供生成TLV命令记录条件用（命名类似于Shell比较运算符）
        ConditionType = {"LIKE": "LIKE",
                         # 相似(采用正则表达式匹配)：支持字符串和数字,
                         # 数字将自动被转换为字符串进行判断
                         "NLIKE": "NLIKE",
                         # 不相似(采用正则表达式匹配)：支持字符串和数字,
                         # 数字将自动被转换为字符串进行判断
                         "EQ": "EQ",  # 等于: EQual：支持字符串和数字
                         "NE": "NE",  # 不等于: Not Equal：支持字符串和数字
                         "GT": "GT",  # 大于：Greater Than：仅支持数字
                         "GE": "GE",  # 大于等于: Greater or Equal：仅支持数字
                         "LT": "LT",  # 小于：Little Than：仅支持数字
                         "LE": "LE",  # 大于：Little or Equal：仅支持数字
                         "EQOR": "EQOR",
                         # 等于[或等于](满足多个EQ条件中任一即可):
                         # EQual OR：支持字符串和数字
                         "NEAND": "NEAND",
                         # 不等于[并且不等于](同时不满足多个NE条件):
                         # Not Equal AND：支持字符串和数字
                         "LIKEOR": "LIKEOR",
                         # 与*相似[或与##相似](满足多个LIKE条件中任一即可):
                         # LIKE OR：支持字符串和数字,数字将自动被转换为字符串
                         }

        @staticmethod
        def getMultiConditionValueList(*all_value):
            """生成多条件值列表，供EQOR、NEAND和LIKEOR使用

            :param all_value: 筛选的所有条件值。如4,1,50 或"x","dfsd","234"
            :return: 条件值构成的字符串列表
            """
            value_list = []
            for value in all_value:
                # 此处将所有条件值均转为字符串，方便后续的比较
                value_list.append(str(value))
            return value_list

        @staticmethod
        def getCondition(arg_index, filter_type, arg_value, paramIdx=-1):
            """生成单个条件字典

            :param arg_index: 记录值项的索引
            :param filter_type: 筛选类型
            :param arg_value: 筛选的条件值
            :param paramIdx: 兼容处理，匹配告警参数索引
            :return: 条件字典
            """
            # 参数校验，不正确时直接抛出异常
            if type(arg_index) != int:
                raise Exception("argIndex is not a integer: "
                                + str(type(arg_index)))
            filter_type = str(filter_type)
            if filter_type not in Tlv.Record.ConditionType.values():
                raise Exception("unsupported record filter type: "
                                + str(filter_type))

            # 返回结果
            return {"index": arg_index,
                    "type": filter_type,
                    "value": arg_value,
                    "paramIdx": paramIdx}

        @staticmethod
        def getConditionList(*all_condition):
            """生成多条件列表

            :param all_condition: 所有的条件，0或多个，由getCondition()获得
            :return: 条件列表
            """
            condition_list = []
            for condition in all_condition:
                condition_list.append(condition)
            return condition_list

        @staticmethod
        def filter(record_list, condition_list):  # context,
            """筛选出满足指定条件的记录

            :param record_list: 待筛选的记录列表
            :param condition_list: 筛选条件列表，不存在则为None
            :return: 筛选结果 （Java ArrayList） or 抛出异常（参数或运行错误时）
            """
            # 入口参数校验
            if None is record_list:
                return JavaList()  # 返回空List
            if None is condition_list:
                return record_list
            # 依次按recordList中的条件对传入的记录集进行筛选
            for cond in condition_list:
                # 初始化新的记录列表用于存放本轮筛选结果
                ret_rec_list = JavaList()
                # 获取条件信息
                index = int(cond["index"])
                type = str(cond["type"])
                value = cond["value"]
                if isinstance(value, list) or isinstance(value, tuple):
                    # 如果匹配值为字典或元组，则将其元素转换为字符串，方便下面匹配
                    tmp_list = []
                    for item in value:
                        tmp_list.append(str(item))
                    value = tmp_list
                # 遍历所有的记录进行筛选
                rec_num = record_list.size()
                for i in range(0, rec_num):
                    # 获取当前记录及相应的条件值（默认为字符串类型）
                    rec = record_list.get(i)
                    # 按不同的条件类型进行筛选
                    if Tlv.Record.ConditionType["LIKE"] == type:
                        rec_value = str(Tlv.Record.getStrValue(rec, index))
                        value = str(value)
                        if None is not re.search(value,
                                                 rec_value):
                            ret_rec_list.add(rec)
                    elif Tlv.Record.ConditionType["NLIKE"] == type:
                        rec_value = str(Tlv.Record.getStrValue(rec, index))
                        value = str(value)
                        if None is re.search(value,
                                             rec_value):
                            ret_rec_list.add(rec)
                    elif Tlv.Record.ConditionType["EQ"] == type:
                        rec_value = Tlv.Record.getValue(rec, index)
                        param_idx = cond.get("paramIdx", -1)
                        # 告警、事件可以指定参数
                        if param_idx != -1:
                            # 告警、事件指定的参数
                            vals = str(rec_value).split(",")
                            if param_idx < len(vals):
                                rec_value = vals[param_idx]

                        if str(rec_value).strip() == str(value).strip():
                            ret_rec_list.add(rec)
                    elif Tlv.Record.ConditionType["NE"] == type:
                        rec_value = Tlv.Record.getValue(rec, index)
                        if str(rec_value) != str(value):
                            ret_rec_list.add(rec)
                    elif Tlv.Record.ConditionType["GT"] == type:
                        # 范围类型为数值（直接按Long取，避免溢出，范围为左闭右开）
                        rec_value = Tlv.Record.getLongValue(rec, index)
                        if rec_value > value:
                            ret_rec_list.add(rec)
                    elif Tlv.Record.ConditionType["GE"] == type:
                        rec_value = Tlv.Record.getLongValue(rec, index)
                        if rec_value >= value:
                            ret_rec_list.add(rec)
                    elif Tlv.Record.ConditionType["LT"] == type:
                        rec_value = Tlv.Record.getLongValue(rec, index)
                        if rec_value < value:
                            ret_rec_list.add(rec)
                    elif Tlv.Record.ConditionType["LE"] == type:
                        rec_value = Tlv.Record.getLongValue(rec, index)
                        if rec_value <= value:
                            ret_rec_list.add(rec)
                    elif Tlv.Record.ConditionType["EQOR"] == type:
                        rec_value = Tlv.Record.getValue(rec, index)
                        if str(rec_value) in value:  # 此时value为字符串列表
                            ret_rec_list.add(rec)
                    elif Tlv.Record.ConditionType["NEAND"] == type:
                        rec_value = Tlv.Record.getValue(rec, index)
                        if str(rec_value) not in value:  # 此时value为字符串列表
                            ret_rec_list.add(rec)
                    elif Tlv.Record.ConditionType["LIKEOR"] == type:
                        rec_value = str(Tlv.Record.getValue(rec, index))
                        for item in value:  # 此时value为字符串列表
                            if re.search(item, rec_value) is not None:
                                ret_rec_list.add(rec)
                                break
                    else:
                        raise Exception(
                            "Invalid query condition type=" + type
                            + " in condition=" + + str(
                                cond))
                # 置待筛选集合为当前筛选结果，进入下轮筛选
                record_list = ret_rec_list
            # 返回筛选结果
            return ret_rec_list

        @staticmethod
        def filter_test(context, recordList, conditionList):  # context,
            # 入口参数校验
            if None is recordList:
                return JavaList()  # 返回空List
            if None is conditionList:
                return recordList
            # 依次按recordList中的条件对传入的记录集进行筛选
            Log.info(context, "conditionList=" + str(conditionList))
            for cond in conditionList:
                # 初始化新的记录列表用于存放本轮筛选结果
                retRecList = JavaList()
                # 获取条件信息
                Log.info(context, "condition=" + str(cond))
                index = int(cond["index"])
                type = str(cond["type"])
                value = cond["value"]
                if isinstance(value, list) or isinstance(value, tuple):
                    # 如果匹配值为字典或元组，则将其元素转换为字符串，方便下面匹配
                    tmpList = []
                    for item in value:
                        tmpList.append(str(item))
                    value = tmpList
                # 遍历所有的记录进行筛选
                recNum = recordList.size()
                for i in range(0, recNum):
                    # 获取当前记录及相应的条件值（默认为字符串类型）
                    rec = recordList.get(i)
                    # 按不同的条件类型进行筛选
                    if Tlv.Record.ConditionType["LIKE"] == type:
                        recValue = str(Tlv.Record.getStrValue(rec, index))
                        value = str(value)
                        if re.search(value, recValue) is not None:
                            retRecList.add(rec)
                    elif Tlv.Record.ConditionType["NLIKE"] == type:
                        recValue = str(Tlv.Record.getStrValue(rec, index))
                        value = str(value)
                        if re.search(value, recValue) is not None:
                            retRecList.add(rec)
                    elif Tlv.Record.ConditionType["EQ"] == type:
                        recValue = Tlv.Record.getValue(rec, index)
                        if str(recValue) == str(value):
                            retRecList.add(rec)
                    elif Tlv.Record.ConditionType["NE"] == type:
                        recValue = Tlv.Record.getValue(rec, index)
                        if str(recValue) != str(value):
                            retRecList.add(rec)
                    elif Tlv.Record.ConditionType["GT"] == type:
                        # 范围类型为数值（直接按Long取，避免溢出，范围为左闭右开）
                        recValue = Tlv.Record.getLongValue(rec, index)
                        if recValue > value:
                            retRecList.add(rec)
                    elif Tlv.Record.ConditionType["GE"] == type:
                        recValue = Tlv.Record.getLongValue(rec, index)
                        if recValue >= value:
                            retRecList.add(rec)
                    elif Tlv.Record.ConditionType["LT"] == type:
                        recValue = Tlv.Record.getLongValue(rec, index)
                        if recValue < value:
                            retRecList.add(rec)
                    elif Tlv.Record.ConditionType["LE"] == type:
                        recValue = Tlv.Record.getLongValue(rec, index)
                        if recValue <= value:
                            retRecList.add(rec)
                    elif Tlv.Record.ConditionType["EQOR"] == type:
                        recValue = Tlv.Record.getValue(rec, index)
                        if str(recValue) in value:  # 此时value为字符串列表
                            retRecList.add(rec)
                    elif Tlv.Record.ConditionType["NEAND"] == type:
                        recValue = Tlv.Record.getValue(rec, index)
                        if str(recValue) not in value:  # 此时value为字符串列表
                            retRecList.add(rec)
                    elif Tlv.Record.ConditionType["LIKEOR"] == type:
                        recValue = str(Tlv.Record.getValue(rec, index))
                        for item in value:  # 此时value为字符串列表
                            if re.search(item, recValue) is not None:
                                retRecList.add(rec)
                                break
                    else:
                        raise Exception("Invalid query condition type="
                                        + type + " in condition="
                                        + str(cond))
                # 置待筛选集合为当前筛选结果，进入下轮筛选
                recordList = retRecList
            # 返回筛选结果
            return retRecList

        @staticmethod
        def topSort(record_list, condition_list):  # context,
            """依次按指定的条件对所有记录进行排序，满足条件的放在前部，不满足条件的放在尾部

            :param record_list: 待排序的记录列表
            :param condition_list: 排序条件列表
            :return: 排序结果 （Java ArrayList）
            """
            # 入口参数校验
            if None is record_list:
                return None
            if None is condition_list:
                return record_list
            # 依次按recordList中的条件对传入的记录集进行排序
            # 初始化的时的记录列表
            ret_rec_list = JavaList()
            for cond in condition_list:
                # 初始化临时记录列表，保存不满足本次排序条件的记录
                tmp_rec_list = JavaList()
                # 获取条件信息
                index = int(cond["index"])
                type = str(cond["type"])
                value = cond["value"]
                if isinstance(value, list) or isinstance(value, tuple):
                    # 如果匹配值为字典或元组，则将其元素转换为字符串，方便下面匹配
                    tmp_list = []
                    for item in value:
                        tmp_list.append(str(item))
                    value = tmp_list
                # 遍历所有的记录进行筛选
                rec_num = record_list.size()
                for i in range(0, rec_num):
                    # 获取当前记录及相应的条件值（默认为字符串类型）
                    rec = record_list.get(i)
                    is_sorted = False
                    # 按不同的条件类型进行筛选
                    if Tlv.Record.ConditionType["LIKE"] == type:
                        rec_value = str(Tlv.Record.getStrValue(rec, index))
                        value = str(value)
                        if re.search(value, rec_value) is not None:
                            is_sorted = True
                    elif Tlv.Record.ConditionType["NLIKE"] == type:
                        rec_value = str(Tlv.Record.getStrValue(rec, index))
                        value = str(value)
                        if None is re.search(value,
                                             rec_value):
                            is_sorted = True
                    elif Tlv.Record.ConditionType["EQ"] == type:
                        rec_value = Tlv.Record.getValue(rec, index)
                        if str(rec_value) == str(value):
                            is_sorted = True
                    elif Tlv.Record.ConditionType["NE"] == type:
                        rec_value = Tlv.Record.getValue(rec, index)
                        if str(rec_value) != str(value):
                            is_sorted = True
                    elif Tlv.Record.ConditionType["GT"] == type:
                        # 范围类型为数值（按Long取，避免溢出，范围为左闭右开）
                        rec_value = Tlv.Record.getLongValue(rec, index)
                        if rec_value > value:
                            is_sorted = True
                    elif Tlv.Record.ConditionType["GE"] == type:
                        rec_value = Tlv.Record.getLongValue(rec, index)
                        if rec_value >= value:
                            is_sorted = True
                    elif Tlv.Record.ConditionType["LT"] == type:
                        rec_value = Tlv.Record.getLongValue(rec, index)
                        if rec_value < value:
                            is_sorted = True
                    elif Tlv.Record.ConditionType["LE"] == type:
                        rec_value = Tlv.Record.getLongValue(rec, index)
                        if rec_value <= value:
                            is_sorted = True
                    elif Tlv.Record.ConditionType["EQOR"] == type:
                        rec_value = Tlv.Record.getValue(rec, index)
                        if str(rec_value) in value:  # 此时value为字符串列表
                            is_sorted = True
                    elif Tlv.Record.ConditionType["NEAND"] == type:
                        rec_value = Tlv.Record.getValue(rec, index)
                        if str(rec_value) not in value:  # 此时value为字符串列表
                            is_sorted = True
                    elif Tlv.Record.ConditionType["LIKEOR"] == type:
                        rec_value = str(Tlv.Record.getValue(rec, index))
                        for item in value:  # 此时value为字符串列表
                            if re.search(item, rec_value) is not None:
                                is_sorted = True
                                break
                    else:
                        raise Exception("Invalid query condition type="
                                        + type + " in condition="
                                        + str(cond))
                    # 处理排序结果
                    if is_sorted:
                        # 当前记录满足排序条件时，则将其存入返回记录集
                        ret_rec_list.add(rec)
                    else:
                        # 若当前记录不满足排序条件，则存入临时记录列表中，留待匹配后续的条件
                        tmp_rec_list.add(rec)
                # 置待筛选集合为当前筛选结果，进入下轮筛选
                if 0 == tmp_rec_list.size():
                    break
                else:
                    record_list = tmp_rec_list
            # 将不满足排序条件的记录放入retRecList尾部
            rec_num = tmp_rec_list.size()
            for i in range(0, rec_num):
                rec = tmp_rec_list.get(i)
                ret_rec_list.add(rec)
            # 返回筛选结果
            return ret_rec_list

        @staticmethod
        def logRecList(context, recordList, strDesc=None):
            # 构造记录字符串
            strRec = ""
            if None is not recordList:
                recNum = recordList.size()
                for index in range(0, recNum):
                    strRec = strRec + str(recordList.get(index)) + "\n"
            else:
                strRec = str(None)
            # 添加描述
            strHead = "\n"
            if type(strDesc) == str:
                strHead = strHead + strDesc + "\n"
            # 打印记录(尽量将所有的函数调用关系均打印出来)
            Log.info(context, strHead
                     + "================logRecList BEGIN==================\n"
                     + strRec
                     + "================logRecList END====================\n"
                     + Log.get_caller_info(10))
            return

    class Const():
        """TLV常量类

        """

        # 默认TLV连接端口号，建立TLV连接时，需使用这个端口
        TLVPORT_DEFAULT = 8080

        # TLV命令执行超时时间(秒)
        TIMEOUT_DEFAULT = 90

        # 2800V3单控下电控制器TLV命令执行超时时间(秒)
        TIMEOUT_FOR_2800V3SINGLE_OFFLINE_CONTROLLER = 3600

        # TLV执行后的结果集的结束标识
        RECORD_ENDFLAG = 4095

        # TLV执行批量查询时的最大条数（自选）
        RECORD_MAXNUM = 100

        # TLV预留异常ID范围：-10000 ~ -11000
        # TLV执行错误ID，执行结果为空（非None，即有一个不存在任何元素的行）。
        ERRID_EMPTY_RESULT = "-10001"
        # TLV执行错误ID，执行结果为None。
        ERRID_NONE_RESULT = "-10002"


class TlvConnection(object):
    RE_CONNECTION_TIMES = 3
    TLV_PORT_DEFAULT = 8080
    TLV_CAN_NOT_CONNECT_CODE = "1073949185"  # 与设备通信异常，请检查网络连接或设备状态是否正常
    _instance = None
    _tlv = None
    _tlvConnector = None

    def __new__(cls, *args, **kwargs):
        if cls._instance is None:
            cls._instance = super(TlvConnection, cls).__new__(cls, *args,
                                                              **kwargs)
        return cls._instance

    def __init__(self, connectorFactory, ip, tlvPort, ssl=True, innerIps=None):
        self.connectorFactory = connectorFactory
        # 高端场景，通过内部IP创建连接
        if innerIps:
            tempList = innerIps.split(",")
            innerIpList = JavaList()
            for innerIp in tempList:
                innerIpList.add(String(innerIp))
            self.ip = innerIpList
        else:
            self.ip = ip
        self.tlvPort = tlvPort
        self.ssl = ssl

    def create(self, user, pawd):
        reConnectionTimes = 0
        while self._tlv is None and \
                reConnectionTimes < TlvConnection.RE_CONNECTION_TIMES:
            tlvConnector = self.connectorFactory.createTlvConnector(
                self.ip, TlvConnection.TLV_PORT_DEFAULT, user, pawd, self.ssl)
            self._tlv = tlvConnector.getConnectionNoException()
            reConnectionTimes += 1
        del pawd
        return self._tlv

    def createForward(self, dev):
        if self._tlvConnector is None:
            self._tlvConnector = \
                self.connectorFactory.createForwardTlvConnector(dev)
        for i in range(TlvConnection.RE_CONNECTION_TIMES):
            self._tlv = self._tlvConnector.getConnForwardNoException()
            if self._tlv:
                break
        return self._tlv

    def close(self):
        if self._tlv is not None:
            try:
                self._tlv.close()
            except Exception:
                return
            finally:
                self._tlv = None
        return

    def check_connect(self):
        try:
            if self._tlv:
                param0 = Tlv.Param.getParam(TlvEnum.pubAttr["type"],
                                            Tlv.Param.TYPE_ENUM,
                                            TlvEnum.Enum.OM_OBJ_E["SYSTEM"])
                param_list = Tlv.Param.getParamList(param0)
                self._tlv.invoke(TlvEnum.cmd["GET"], param_list,
                                 Tlv.Const.TIMEOUT_DEFAULT)
                return True
            return False
        except (JException, Exception):
            return False

    def getTlv(self, user, pawd):
        try:
            if self._tlv is None:
                self.create(user, pawd)
        except Exception:
            raise Exception(self.TLV_CAN_NOT_CONNECT_CODE, "")

        return self._tlv

    def getTlvForward(self, dev):
        try:
            if self._tlv is None:
                self.createForward(dev)
            if not self.check_connect():
                self._tlv.close()
                self._tlv.reConnect()
        except (JException, Exception):
            raise Exception(self.TLV_CAN_NOT_CONNECT_CODE, "")

        return self._tlv


class TlvConnectionSVP(object):
    RE_CONNECTION_TIMES = 3
    IP_DEFAULT = "172.17.126.11"
    PORT_DEFAULT = "19002"
    TLV_CAN_NOT_CONNECT_CODE_SVP = "TLV_CAN_NOT_CONNECT_CODE_SVP"  # 与SVP建立连接异常

    _instance = None
    _tlv = None

    def __new__(cls, *args, **kwargs):
        if cls._instance is None:
            cls._instance = super(TlvConnectionSVP, cls).__new__(cls, *args,
                                                                 **kwargs)
        return cls._instance

    def __init__(self, connectorFactory, deviceType, sn,
                 ip=IP_DEFAULT, tlvPort=PORT_DEFAULT, session=""):
        self.connectorFactory = connectorFactory
        self.deviceType = "svp"
        self.sn = sn + "svp"
        self.ip = ip
        self.tlvPort = tlvPort
        self.session = session

    def create(self):
        try:
            tlvConnector = self.connectorFactory.createTlvConnectorSVP(
                self.ip, self.tlvPort, self.session, self.deviceType, self.sn)
            self._tlv = tlvConnector.getConnectionSVP()
            return self._tlv
        except ToolException:
            raise Exception(self.TLV_CAN_NOT_CONNECT_CODE_SVP, "")
        return None

    def close(self):
        if self._tlv is not None:
            try:
                self._tlv.close()
            except Exception:
                return
            finally:
                self._tlv = None
        return

    def getTlv(self):
        if self._tlv is None:
            self.create()
        return self._tlv
