#!/usr/bin/python
# -*- coding: utf-8 -*-
# 脚本功能：检测集群内管理口是否冗余
import os
import sys
import tempfile
import re
import json
import logging


logging.basicConfig(level=logging.INFO,
                    filename="/OSM/log/cur_debug/messages",
                    format='[%(asctime)s][%(levelname)s][%(message)s][%(filename)s, %(lineno)d]',
                    datefmt='%Y-%m-%d %H:%M:%S')


RETURN_OK = 0
RETURN_ERROR = 1
RETURN_NOT_CHECK = 2
IPV4_LINE = 6
IPV6_LINE = 7
# 直接执行CLI命令的字符串格式：
# {0}表示账号名
# {1}表示需要执行的CLI命令，
# {2}表示命令执行回显重定向文件，用于命令执行后的结果解析
EXEC_CLI_CMD = "/ISM/cli/ismcli -u {0} -c '{1}'  > '{2}'"
USER_EXIST = r'id -u admin > /dev/null 2>&1'
USER_CMD = r"cat /etc/passwd | grep '/ISM/cli/ismcli' | grep ':0:root:'"


def net_check_with_path(path):
    plane_0 = False
    plane_1 = False

    with open(path) as lines:
        lines_array = lines.readlines()

    for line in lines_array:
        if not line.lstrip().startswith('CTE'):
            continue

        record = line.lstrip().split()
        # 第六位和第七位分别代表ipv4和ipv6的地址，如果两个都是'--', 代表端口没有配置IP
        if record[IPV4_LINE] == record[IPV6_LINE]:
            logging.info("The port is not configured with an IP address.")
            continue

        if re.findall(r"CTE\d+\.SMM0\.MGMT\d | CTE\d+\.A\.MGMT", "".join(line), flags=re.S):
            plane_0 = True

        if re.findall(r"CTE\d+\.SMM1\.MGMT\d | CTE\d+\.B\.MGMT", "".join(line), flags=re.S):
            plane_1 = True

        if plane_0 and plane_1:
            return RETURN_OK

    return RETURN_ERROR


def net_check_mgt_is_redundant():
    _fd, tmp_file = tempfile.mkstemp(prefix="DEVM_check_mgt_redundant_", text=True)
    os.close(_fd)

    cli_cmd_str = "show port general logic_type=Management_Port physical_type=ETH running_status=link_up"

    ret = os.system(USER_EXIST)
    if not ret:
        USER_NAME = 'admin'
    else:
        ret_code, output = os.system(USER_CMD)
        if ret_code:
            logging.error('Get user failed, ret({0}), output: {1}'.format(ret_code, output))
            os.remove(tmp_file)
            return RETURN_ERROR
        USER_NAME = output.strip().split(':')[0]

    # 构造cli命令
    cli_cmd_str = EXEC_CLI_CMD.format(USER_NAME, cli_cmd_str, tmp_file)

    ret = os.system(cli_cmd_str)
    if ret:
        logging.warning("NET_CHECK_MGT_IS_REDUNDANT: Cmd(%s) exec failed, ret %d.", cli_cmd_str, ret)
        os.remove(tmp_file)
        return RETURN_ERROR

    ret = net_check_with_path(tmp_file)
    # 删除临时文件
    os.remove(tmp_file)
    return ret


def net_check_version():
    version_file = "/tmp/check_extend_param"
    dst_version = ""
    dst_version_list = ["6.0.0", "6.0.1", "6.1.0", "6.1.2.RC1"]

    with open(version_file, 'r') as f:
        read_str = str(f.read()).replace("'", '"')
        logging.info("NET_CHECK_MGT_IS_REDUNDANT: Read str %s.", read_str)
        extend_param = json.loads(read_str)
        dst_version = str(extend_param.get('targetVersion', ''))
        logging.info("NET_CHECK_MGT_IS_REDUNDANT: Dst version %s.", dst_version)

    # 如果升级到6.1.2之前的版本，不去做检测
    for version in dst_version_list:
        if dst_version.startswith(version):
            return RETURN_NOT_CHECK
    return RETURN_OK


def main():
    try:
        ret = net_check_version()
        if ret == RETURN_NOT_CHECK:
            logging.info("NET_CHECK_MGT_IS_REDUNDANT: Not in target version, do not check.")
            print("True")
            return 0

        ret = net_check_mgt_is_redundant()
        if ret == RETURN_OK:
            logging.info("NET_CHECK_MGT_IS_REDUNDANT: Check is redundant.")
            print("True")
        else:
            logging.info("NET_CHECK_MGT_IS_REDUNDANT: Check is not redundant.")
            print("False")
        return 0

    except Exception as e:
        logging.exception("NET_CHECK_MGT_IS_REDUNDANT: %s", e)
        print("False")
        return 1


if __name__ == '__main__':
    sys.exit(main())
