#!/usr/bin/python
# -*- coding: utf-8 -*-
import logging
import sys
import subprocess
import shlex
import re
import os

# 第一次做的补丁的sgl使用量的告警阈值
FIRST_SPH_SGL_COUNT_THRESHOLD = 200000
# 没有补丁的sgl使用量的告警阈值
NO_SPH_SGL_COUNT_THRESHOLD = 50000
# 不再做检查的版本
NO_NEED_CHECK_VERSION = "6.1.3"
# 错误信息
SGL_ERROR_MSG = "SGL quota of disk domain %s of current software version %s has used %d (more than %d)."

USER_EXIST = r'id -u admin > /dev/null 2>&1'
USER_CMD = r"cat /etc/passwd | grep '/ISM/cli/ismcli' | grep ':0:root:'"
USER_LOGIN_CMD = "/ISM/cli/ismcli -u {0}"

logging.basicConfig(level=logging.INFO,
                    filename="/OSM/log/cur_debug/messages",
                    format='[%(asctime)s][%(levelname)s][%(filename)s, %(lineno)d]: %(message)s',
                    datefmt='%Y-%m-%d %H:%M:%S')


def diagnose(command):
    cmd = 'diagsh --attach=*_12 --cmd="%s"' % command
    process = subprocess.Popen(shlex.split(cmd), stdout=subprocess.PIPE)
    out = process.communicate()
    return out[0].decode("utf-8")


def cli(command):
    sub_cmds = command.split(" | ")
    process = None
    for sub_cmd in sub_cmds:
        if process:
            process = subprocess.Popen(shlex.split(sub_cmd.strip()), stdin=process.stdout, stdout=subprocess.PIPE)
        else:
            process = subprocess.Popen(shlex.split(sub_cmd.strip()), stdout=subprocess.PIPE)
    out = process.communicate()
    return out[0].decode("utf-8")


def get_user_name():
    ret = os.system(USER_EXIST)
    if not ret:
        return 'admin'
    else:
        output = cli(USER_CMD)
        if not output:
            logging.error('Get user failed')
            return ''
        return output.strip().split(':')[0]


def format_patch_version(version):
    find_versions = re.findall(r"SPH(\d+)", version)
    sph_ver = find_versions[0] if find_versions else "0"
    return "SPH{}".format(sph_ver.zfill(5))


def compare_patch(version1, version2):
    v1f = format_patch_version(version1)
    v2f = format_patch_version(version2)
    if v1f == v2f:
        return 0
    return 1 if v1f > v2f else -1


def get_sgl_quota(domain_id):
    cmd = "zone rec showquota " + domain_id
    cmd_result = diagnose(cmd)
    if not cmd_result:
        return 0
    logging.info("%s", cmd_result)
    infos = cmd_result.splitlines(False)
    for info in infos:
        if "Sgl used count:" in info or "Sgl userd count:" in info:
            sgl_used_count = int(info.split(":")[1].strip())
            return sgl_used_count
    return 0


def get_domain_ids():
    user_name = get_user_name()
    if not user_name:
        return []
    cli_login_str = USER_LOGIN_CMD.format(user_name)
    cmd = r"echo -e 'show disk_domain general' | " + cli_login_str + r" | awk 'NR>4' | awk '{print $1}'"
    cli_result = cli(cmd)
    if not cli_result:
        return []
    results = cli_result.splitlines(False)
    domain_ids = [item for item in results if item.isdigit()]
    return domain_ids


def get_version_info():
    user_name = get_user_name()
    if not user_name:
        return {}
    cli_login_str = USER_LOGIN_CMD.format(user_name)
    cmd = r"echo -e 'show system general' | " + cli_login_str + r" | grep -E 'Product Version|Patch Version'"
    cli_result = cli(cmd)
    if not cli_result:
        return {}
    results = cli_result.splitlines(False)
    product_version = ""
    patch_version = ""
    for line in results:
        if "Product Version" in line:
            product_version = line.split(":")[1].strip()
        if "Patch Version" in line and not line.endswith(":"):
            patch_version = line.split(":")[1].strip()
    if product_version.find(".SPH") > -1:
        patch_version = product_version[product_version.find("SPH"):len(product_version)]
        product_version = product_version[0:product_version.find(".SPH")]
    return {"product_version": product_version, "patch_version": patch_version}


def collect_rec_infos(domain_ids):
    info_cmds = ["zone rec showratio -p {}", "zone rec io -dp {}", "zone rec showlist -dp {}"]
    for cmd in info_cmds:
        for domain_id in domain_ids:
            cmd_result = diagnose(cmd.format(domain_id))
            if cmd_result:
                logging.info("%s", cmd_result)


def handle_not_pass(domain_id, sgl_count, threshold, product_version, sph_map):
    logging.error(SGL_ERROR_MSG, domain_id, product_version, sgl_count, threshold)
    if product_version in sph_map:
        print("False")
        print("0x082a0001")
        print("{0};{1};{2};{3};{4}".format(product_version, domain_id, str(sgl_count),
                                           str(threshold), sph_map.get(product_version)))
    else:
        print("False")
        print("0x082a0002")
        print("{0};{1};{2};{3}".format(product_version, domain_id, str(sgl_count), str(threshold)))


def check_rec_sgl_used_count(product_version, patch_version, domain_ids):
    if product_version >= NO_NEED_CHECK_VERSION:
        print("True")
        return

    sph_v1_map = {"6.0.1": "SPH25", "6.1.0": "SPH15", "6.1.2": "SPH6"}
    sph_v2_map = {"6.0.1": "SPH28", "6.1.0": "SPH22", "6.1.2": "SPH16"}

    # 若软件版本大于等于sph_v2_map所列值，则检查结果为通过
    if product_version in sph_v2_map.keys() and compare_patch(patch_version, sph_v2_map.get(product_version)) >= 0:
        print("True")
        return

    for domain_id in domain_ids:
        sgl_count = get_sgl_quota(domain_id)
        if product_version in sph_v1_map.keys() and compare_patch(patch_version, sph_v1_map.get(product_version)) >= 0:
            if sgl_count > FIRST_SPH_SGL_COUNT_THRESHOLD:
                handle_not_pass(domain_id, sgl_count, FIRST_SPH_SGL_COUNT_THRESHOLD, product_version, sph_v2_map)
                return
        elif sgl_count > NO_SPH_SGL_COUNT_THRESHOLD:
            handle_not_pass(domain_id, sgl_count, NO_SPH_SGL_COUNT_THRESHOLD, product_version, sph_v2_map)
            return
    print("True")


def main():
    try:
        domain_ids = get_domain_ids()
        if not domain_ids:
            print("True")
            return 0

        version_info = get_version_info()
        if not version_info:
            print("True")
            return 0
        
        product_version = version_info.get("product_version")
        patch_version = version_info.get("patch_version")

        # 收集一些回收相关数据，用于后续分析相关问题
        collect_rec_infos(domain_ids)

        # 检查sgl的使用量
        check_rec_sgl_used_count(product_version, patch_version, domain_ids)
        return 0
    except Exception as ex:
        logging.exception("check_rec_sgl_used_count: %s", ex)
        print("False")
        return 1

if __name__ == '__main__':
    sys.exit(main())
