#!/usr/bin/python
# -*- coding: utf-8 -*-
# Copyright (c) Huawei Technologies Co., Ltd. 2018-2024. All rights reserved.
import inspect
import json
import sys

from infra.debug.log import swm_logger as log

PYTHON_2 = '2.7' in sys.version


class Node(object):
    NODE_ID_BIT_NUM = 10
    GLOBAL_FLAG_NUM = 30

    def __init__(self, node_id, ip_infos, zone_id=0, is_global=False):
        self.node_id = int(node_id)
        self.zone_id = int(zone_id)
        self.is_global = bool(is_global)
        self.ip_list = [IpInfo(**ip_info) for ip_info in ip_infos]
        self.global_nid = self.get_lsid(self.node_id, self.zone_id) if is_global else 0

    def __str__(self):
        fmt = "[zone_id={zone_id},node_id={node_id},ip_list={ip_list},is_global={is_global},global_nid={global_nid}]"
        return fmt.format(zone_id=self.zone_id, node_id=self.node_id, ip_list=self.ip_list,
                          is_global=self.is_global, global_nid=self.global_nid)

    def __repr__(self):
        return self.__str__()

    @property
    def addr_list(self):
        addr_list = []
        for ip_info in self.ip_list:
            addr_list.extend(ip_info.ips)
        return addr_list

    @classmethod
    def get_lsid(cls, node_id, zone_id):
        host_id = node_id + (zone_id << cls.NODE_ID_BIT_NUM)
        return host_id + (1 << cls.GLOBAL_FLAG_NUM)

    @classmethod
    def convert_keys(cls, parameters):
        key_map = {
            'nodeId': 'node_id',
            'snodeId': 'node_id',
            'zoneId': 'zone_id',
            'ipCfg': 'ip_infos',
        }
        defined_params = cls.get_func_signature(Node.__init__)
        return {key_map.get(k, k): v for k, v in parameters.items() if key_map.get(k, k) in defined_params}

    @classmethod
    def get_func_signature(cls, func):
        if PYTHON_2:
            return list(inspect.getargspec(func).args)
        return list(inspect.signature(func).parameters)

    @classmethod
    def build(cls, init_kwargs):
        """
        构造
        {'nodeId':0, 'zoneId':0, 'ipCfg':[{'usetype': '0', 'ips':['x.x.x.x']}], 'is_global': False}
        或
        {'snodeId':0, 'zoneId':0, 'ipCfg':[{'usetype': '0', 'ips':['x.x.x.x']}], 'is_global': False}
        的格式调用此函数，可正确生成对应的Node对象
        :param init_kwargs:
        :return:
        """
        new_init_kwargs = cls.convert_keys(init_kwargs)
        return cls(**new_init_kwargs)

    def get_host_id(self):
        return self.global_nid if self.is_global else self.node_id


class IpInfo(object):
    def __init__(self, ips, usetype=0):
        self.ips = ips
        # 0 心跳ip，100扩控ip
        self.usetype = usetype

    def __str__(self):
        return ", ".join(self.ips)

    def __repr__(self):
        return self.__str__()


class IpInfoManager(object):
    LOG_TAG = 'IpInfoManager'

    def __init__(self, ip_filepath='/OSM/conf/innerIpInformation.conf', is_global=False):
        self.ip_filepath = ip_filepath
        self.is_global = is_global
        self.ip_infos = self.loads_ipinfo(self.ip_filepath)
        self.local_node = None
        self.all_nodes = None
        self.parse()

    @classmethod
    def loads_ipinfo(cls, filepath):
        try:
            with open(filepath, 'r') as fp:
                ip_infos = json.load(fp)
            return ip_infos.get('networkConfigs', {})
        except Exception:
            log.exception('parse ip config exception!')
        return {}

    @classmethod
    def get_local_node_info(cls, ip_infos, is_global=False):
        local_node = ip_infos.get('localConfig', {})
        zone_cfg = ip_infos.get('zoneCfg', {})
        zone_local_node = zone_cfg.get('localNode', {})
        local_node.update(zone_local_node)
        ips = local_node.pop('ips', [])
        local_node.update({'ipCfg': [{'usetype': '0', 'ips': ips}], 'is_global': is_global})
        return Node.build(local_node)

    @classmethod
    def get_all_node_info(cls, ip_infos, is_global=False):
        zone_cfg = ip_infos.get('zoneCfg', {})
        zone_ips = zone_cfg.get('zoneIps', [])
        if not zone_ips:
            ctl_node = ip_infos.get('nodeCfg', [])
            data_node = ip_infos.get('snodeCfg', [])
            all_node = ctl_node + data_node
            zone_ips = [{'zoneId': 0, 'nodeCfg': all_node}]
        nodes = []
        for zone_ip in zone_ips:
            nodes.extend(cls.get_all_zone_node_info(zone_ip, is_global))
        return nodes

    @classmethod
    def get_all_zone_node_info(cls, zone_ip_info, is_global=False):
        zone_id = int(zone_ip_info.get('zoneId', "0"))
        nodes = cls.get_zone_node_info(zone_ip_info, zone_id, is_global)
        return nodes

    @classmethod
    def get_zone_node_info(cls, ip_infos, zone_id=0, is_global=False):
        ctl_node = ip_infos.get('nodeCfg', [])
        data_node = ip_infos.get('snodeCfg', [])
        all_node = ctl_node + data_node
        for info in all_node:
            info.update({'zoneId': zone_id, 'is_global': is_global})
        nodes = [Node.build(node_info) for node_info in all_node]
        return nodes

    def parse(self):
        try:
            self.local_node = self.get_local_node_info(self.ip_infos, self.is_global)
            self.all_nodes = self.get_all_node_info(self.ip_infos, self.is_global)
            log.info('%s: parsed local node=%s, all nodes=%s', self.LOG_TAG, self.local_node, self.all_nodes)
        except Exception:
            log.exception('parse host info exception!')
