#!/usr/bin/python
# -*- coding: utf-8 -*-
# Copyright (c) Huawei Technologies Co., Ltd. 2018-2022. All rights reserved.
import hashlib
import os
import subprocess
import sys
import time
import yaml

from config import env
from infra.debug.log import swm_logger as log
from infra.util import shell

PYTHON_2 = '2.7' in sys.version
CUR_MANIFEST_PATH = '/startup_disk/image/pkg_cur/manifest.yml'
DATA_NODE_FLAG = "/startup_disk/conf/conf_local/data_node"


class ProductName(object):
    Dorado = 'Dorado'
    DbStore = 'dbstore'
    Converged = 'Converged'
    BackupStorage = 'BackupStorage'
    Micro = 'Micro'
    Micro_Unified = 'Micro_Unified'
    Lite = 'Lite'
    Lite_Converged = 'Lite_Converged'
    ASeries = "ASeries"


def decode_str(value):
    if not PYTHON_2 and isinstance(value, bytes):
        return value.decode()
    return value


def call_system_cmd(cmd, retry_times=2, timeout=10):
    ret_code = 1
    output = ""
    for retry_time in range(retry_times):
        process = subprocess.Popen(cmd, stdin=subprocess.PIPE, stdout=subprocess.PIPE, stderr=subprocess.STDOUT,
                                   close_fds=True)
        timeout_para = {} if PYTHON_2 else {"timeout": timeout}
        try:
            output, stderr = map(decode_str, process.communicate(**timeout_para))
        except Exception:
            if retry_time != retry_times - 1:
                log.warning("Exception occurs when call shell cmd %s. sleep 1s and try again.", cmd)
                time.sleep(1)
            else:
                log.error("Exec cmd(%s) except, retCode(%d), result(%s).", cmd, ret_code, output)
            continue
        ret_code = process.poll()
        if ret_code:
            log.warning("Exec cmd(%s) result abnormal, ret(%d), result(%s).", cmd, ret_code, stderr)
        else:
            log.info("Exec cmd(%s) successfully.", cmd)
        return ret_code, output
    return ret_code, output


def call_system_cmd_list(cmd_list, timeout=10):
    ret_code = 1
    output = ""
    process = None
    for cmd in cmd_list:
        if process:
            process = subprocess.Popen(cmd, stdin=process.stdout, stdout=subprocess.PIPE)
        else:
            process = subprocess.Popen(cmd, stdout=subprocess.PIPE)
    timeout_para = {} if PYTHON_2 else {"timeout": timeout}
    try:
        output, _ = map(decode_str, process.communicate(**timeout_para))
    except Exception:
        log.error("Exec cmd(%s) except, retCode(%d), result(%s).", cmd_list, ret_code, output)
        return ret_code, output
    ret_code = process.poll()
    if ret_code:
        log.warning("Exec cmd(%s) result abnormal, ret(%d), result(%s).", cmd_list, ret_code, output)
    else:
        log.info("Exec cmd(%s) successfully.ret(%s), output(%s).", cmd_list, ret_code, output)
    return ret_code, output


def calc_sum(hash_cls, file_name):
    """
    :param hash_cls:
    :param file_name:
    :return:
    """
    try:
        return update_hash(file_name, hash_cls)
    except Exception:
        log.exception("Fail to open file: %s", file_name)
        return ""


def update_hash(file_name, hash_cls):
    obj = hash_cls()
    max_mem_cache = 10 * 1024 * 1024
    with open(file_name, "rb") as file_handle:
        while True:
            content = file_handle.read(max_mem_cache)
            if not content:
                break
            obj.update(content)
    return obj.hexdigest()


def verify_by_sha256sum(pkg_dir, sha256_check_file):
    """
    :param pkg_dir:
    :param sha256_check_file:
    :return:
    """
    if not os.path.exists(sha256_check_file):
        log.warning("CHECK_UTIL: Sha256sum check file not exist.")
        return True

    try:
        with os.fdopen(os.open(sha256_check_file, os.O_RDONLY, 0o640), 'r') as file_handle:
            check_sums = [x.strip() for x in file_handle.readlines()]
    except Exception as exception:  # system disk IO error
        log.exception("CHECK_UTIL: Load sha256sum check file exception %s.", exception)
        return False

    check_pass = True
    for check in check_sums:
        arr = check.split(' ')
        sha_sum = arr[0]
        file_name = os.path.join(pkg_dir, arr[-1])
        if not os.path.exists(file_name):
            log.error("CHECK_UTIL: Verify pkg failed, cause the file(%s) is not exist.", file_name)
            check_pass = False
            continue
        real_sum = calc_sum(hashlib.sha256, file_name)
        if real_sum != sha_sum:
            log.error("CHECK_UTIL: File(%s) check sum failed(expect: %s, actually: %s).", file_name, sha_sum, real_sum)
            check_pass = False

    log.info("CHECK_UTIL: Verify dir(%s) result: %s.", pkg_dir, check_pass)
    return check_pass


def check_mem_is_enough(max_memory):
    shell.call_system_cmd("echo 3 >/proc/sys/vm/drop_caches")
    # 重试查询，防止出现一次为空无法查询到内存
    for _ in range(3):
        ret, out = shell.call_system_cmd("free -m|grep ^Mem|awk '{print $4}'")
        log.info('CHECK_UTIL: Mem info is %s', out)
        out = out.strip()
        if ret or not out or not out.isdigit():
            time.sleep(1)
            continue
        if int(out) >= max_memory:
            return True
        return False
    return False


def get_pkg_version(manifest_file):
    """

    :param manifest_file:
    :return:
    """
    if not os.path.exists(manifest_file):
        log.warning("Can not get the new version, config %s not "
                    "existing.", manifest_file)
        return "", ""

    try:
        with os.fdopen(os.open(manifest_file, os.O_RDONLY, 0o640), 'r') as fd:
            cfg_yml = yaml.safe_load(fd)
        sys_section = cfg_yml.get("SYS")
        version = sys_section.get("Version", "")
        product = sys_section.get("Product", "")
        return version, product
    except Exception:
        log.exception("VER_MGR: Failed to get version.")
    return "", ""


def get_patch_version(manifest_file):
    """

    :param manifest_file:
    :return:
    """
    if not os.path.exists(manifest_file):
        log.warning("Can not get the new version, config %s not "
                    "existing.", manifest_file)
        return ""

    try:
        with os.fdopen(os.open(manifest_file, os.O_RDONLY, 0o640), 'r') as fd:
            cfg_yml = yaml.safe_load(fd)
        patch_version = str(cfg_yml.get("PatchConf")["patch_version"])
        return patch_version
    except Exception:
        log.exception("VER_MGR: Failed to get version.")
    return ""


def get_product_name():
    return get_product_info("Product")


def get_product_info(name, manifest=CUR_MANIFEST_PATH):
    if not os.path.exists(manifest):
        log.info("Can not get the src product type, %s does not exist.", manifest)
        return ""

    with open(manifest, 'r') as fd:
        cfg_yml = yaml.safe_load(fd)
        try:
            product_info = str(cfg_yml.get("SYS", {}).get(name, ''))
            log.debug('Get system product info %s: %s.', name, product_info)
            return product_info
        except Exception:
            log.exception("Failed to get product info, name (%s).", name)
            return ""


def remote_execute_cmd(host, cmd, timeout=10):
    if env.ARM and not os.path.exists(DATA_NODE_FLAG):
        cmd_list = ['ip', 'vrf', 'exec', 'vrf-inner', 'ssh', '-p', '22', 'ibc_os_hs@{0}'.format(host), cmd]
    else:
        cmd_list = ['ssh', '-p', '22', 'ibc_os_hs@{0}'.format(host), cmd]
    return call_system_cmd(cmd_list, retry_times=1, timeout=timeout)


def console_print(msg):
    print(msg)
