#!/usr/bin/python
# coding=utf-8
# Copyright (c) Huawei Technologies Co., Ltd. 2023-2023. All rights reserved.

import logging
import shlex
import subprocess
import re
import sys
import os

logging.basicConfig(level=logging.INFO,
                    filename="/OSM/log/cur_debug/messages",
                    format='[%(asctime)s][%(levelname)s][%(message)s][%(filename)s, %(lineno)d]',
                    datefmt='%Y-%m-%d %H:%M:%S')
# 默认超时时间90s
CHECK_TIMEOUT = 90


def fc_diagnose_usr(command, timeout=10):
    """
    在app_data下，调用diagnose命令.
    timeout: 执行命令的超时时间，默认10s
    output: 默认需要返回执行结果
    """
    diag_cmd = ['diagsh', '--attach=*_12', "--cmd=%s" % command]
    diag_process = subprocess.Popen(diag_cmd, stdout=subprocess.PIPE, stderr=subprocess.PIPE)

    stdout, stderr = diag_process.communicate(timeout=timeout)
    return stdout.decode("utf-8")


def fc_swm_diagnose(command, timeout=CHECK_TIMEOUT):
    """
    在SWM下调用diagnose命令
    timeout: 执行命令的超时时间，默认为设置的LOG_TIMEOUT
    output: 命令执行超时抛出异常，执行完成返回stderr
    """
    swm_cmd = ['swm_diagnose', '-c', "%s" % command]
    swm_process = subprocess.Popen(swm_cmd, stdout=subprocess.PIPE, stderr=subprocess.PIPE)

    stdout, stderr = swm_process.communicate(timeout=timeout)
    return stdout.decode("utf-8")


def fc_linux(command, timeout=10):
    """
    调用linux命令，传入参数时，保证明文写入，不组装
    timeout: 执行命令的超时时间，默认10s
    output: 默认需要返回执行结果
    """
    cmd_list = command.split(' | ')
    linux_process = None
    for cmd in cmd_list:
        if linux_process:
            linux_process = subprocess.Popen(shlex.split(cmd.strip()),
                                             stdin=linux_process.stdout, stdout=subprocess.PIPE)
        else:
            linux_process = subprocess.Popen(shlex.split(cmd.strip()), stdout=subprocess.PIPE)
    stdout = linux_process.communicate(timeout=timeout)[0]
    return stdout.decode("utf-8")


def fc_get_user_name():
    admin_user = r'id -u admin > /dev/null 2>&1'
    ret = os.system(admin_user)
    if not ret:
        return 'admin'

    other_user = r"cat /etc/passwd | grep '/ISM/cli/ismcli' | grep ':0:root:'"
    user_name = fc_linux(other_user)
    if not user_name:
        logging.warning("Get user failed")
        return ''
    return user_name.strip().split(':')[0]


def fc_cli_cmd(user_name, command, grep_cmd='', timeout=10):
    """
    调用cli命令
    timeout: 执行命令的超时时间，默认10s
    output: 默认需要返回执行结果
    """
    cli_cmd = ['/ISM/cli/ismcli', '-u', "%s" % user_name, '-c', "%s" % command]
    grep_cmd = ['grep', "%s" % grep_cmd]
    cli_process = subprocess.Popen(cli_cmd, stdout=subprocess.PIPE, stderr=subprocess.PIPE)
    grep_process = subprocess.Popen(grep_cmd, stdin=cli_process.stdout,
                                    stdout=subprocess.PIPE, stderr=subprocess.PIPE)
    stdout, stderr = grep_process.communicate(timeout=timeout)
    return stdout.decode("utf-8")


def is_fc_v120_card_find():
    get_card_cmd = 'drvunf showversion'
    card_type = 'ioc_v3r1'

    card_info = fc_diagnose_usr(get_card_cmd)
    if not re.search(card_type, card_info, re.IGNORECASE):
        return False
    return True


def is_version_need_check():
    """
    检查系统版本是否存在风险
    output: dorado和融合系列的6.1.5基线版本，及其补丁6.1.5.SPH1至6.1.5.SPH7版本存在风险；
            数据保护系列的1.2.0基线版本，及其补丁1.2.0.SPH1至1.2.0.SPH7版本存在风险
    若系统版本匹配上存在风险的版本返回True，否则返回False
    """
    # dorado和融合系列版本号
    base_version = '6.1.5'
    patch_version = ['6.1.5.SPH%d' % i for i in range(1, 8)]
    patch_version.append(base_version)

    # 数据保护系列版本号
    protect_base_version = '1.2.0'
    protect_patch_version = ['1.2.0.SPH%d' % i for i in range(1, 8)]
    protect_patch_version.append(protect_base_version)
    
    user_name = fc_get_user_name()
    # 未取得用户名，不执行检查
    if not user_name:
        return False
    get_version_cmd = 'show system general'
    grep_cmd = 'Product Version'

    system_version = fc_cli_cmd(user_name, get_version_cmd, grep_cmd)
    if re.search(base_version, system_version):
        # dorado和融合系列是6.1.5版本，取得版本号后检查是否可能存在风险
        if system_version.split()[3] in patch_version:
            return True
    elif re.search(protect_base_version, system_version):
        # 数据保护系列是1.2.0版本，取得版本号后检查是否可能存在风险
        if system_version.split()[3] in protect_patch_version:
            return True
    return False


def is_upgrade_check_ok():
    card_channel_check = 'fw check_local 18'

    stdout = fc_swm_diagnose(card_channel_check)
    return stdout


def main():
    try:
        logging.info("Start to check fc card apichain.")

        # 只在python3版本执行检查
        if not sys.version_info.major == 3:
            logging.info("No need check.")
            print("True")
            return 0

        # 检查是否存在1822 V120 FC前端接口卡
        if not is_fc_v120_card_find():
            logging.info("No risk of fc cards version for this check.")
            print("True")
            return 0

        # 检查系统版本是否存在风险
        if not is_version_need_check():
            logging.info("No risk of system version for this check.")
            print("True")
            return 0

        if not is_upgrade_check_ok():
            logging.warning("Check fc apichain failed.")
            print("False")
            return 0

        # 检查完成，未抛出异常或超时，返回检查通过，print为True表示通过，False表示不通过
        logging.info("Check fc apichain success.")
        print("True")
        return 0

    except Exception as exp:
        logging.exception("Check fc apichain failed: %s", exp)
        print("False")
        return 0


if __name__ == '__main__':
    sys.exit(main())
