#!/usr/bin/python
# -*- coding: utf-8 -*-
# Copyright (c) Huawei Technologies Co., Ltd. 2023-2023. All rights reserved.

import os
import re
import sys
import logging
from time import sleep

logging.basicConfig(level=logging.INFO,
                    filename="/OSM/log/cur_debug/messages",
                    format='[%(asctime)s][%(levelname)s][%(filename)s, %(lineno)d]: %(message)s',
                    datefmt='%Y-%m-%d %H:%M:%S')

console = logging.StreamHandler()
console.setLevel(logging.CRITICAL)
logging.getLogger('').addHandler(console)


def execute_cmd(cmd):
    cmd_fd = os.popen(cmd)
    data = cmd_fd.read().strip()
    cmd_fd.close()
    return data


def execute_showdisk_cmd(cmd):
    cli_format = 'diagsh --attach=*_12 --cmd="{}"'

    for _ in range(3):
        info = execute_cmd(cli_format.format(cmd))
        if 'Present' in info and 'Power' in info and 'DesPower' in info:
            return True, info
        sleep(2)
    return False, ''


def get_applo_version():
    cmd = 'readlink /startup_disk/image/pkg_cur/apollo'
    return execute_cmd(cmd)


def get_product_version():
    cmd = "cat /startup_disk/image/pkg_cur/manifest.yml | grep \"SpcVersion\" | awk -F ': ' '{print $2}'"
    return execute_cmd(cmd)


def get_patch_version():
    pkg_ver = execute_cmd("readlink /startup_disk/image/pkg_cur | awk -F '/' '{print $1}'")
    file = "/startup_disk/image/{}/hotpatch/patch_cur/patch.yml".format(pkg_ver)
    if os.path.isfile(file):
        cmd = "cat %s | grep \"patch_version\" | awk -F ': ' '{print $2}'" % file
        return execute_cmd(cmd)
    return ''


def is_install_specified_patch(num, patch_version):
    match = re.search(r'SPH(\d+)', patch_version)
    if match:
        version = int(match.group(1))
        if version >= num:
            return True
    return False


def is_direct_disk_power_status_ok():
    cmd = 'sascbb showdisk'
    cmd_ret, info = execute_showdisk_cmd(cmd)

    if cmd_ret is False:
        logging.warning('SAS_DISK_POWER_CHECK: query sas disk info failed, info %s', info)
        return True, ''

    '''
    直连盘满足以下回显结果，则说明当前控制器盘未上电，返回检查不通过
    | SlotId | CardId | PhyId | Present | Power | DesPower | PhyFix | PowerFix | Rate   |
    |   0    |   3    |   0   |   1     |   0   | on       |   0    |    0     | 12G    |
    '''
    slot_array = ''
    ret = True
    try:
        for line in info.splitlines():
            line_split = line.split('|')
            if len(line_split) < 10:
                continue
            if line_split[4].strip() != "1":  # 直连盘不在位，跳过
                continue
            if line_split[6].strip() == "off":  # 直连盘被下电，跳过
                continue
            if line_split[9].strip() == "NoLink":  # 直连盘未协商通过未连接，跳过
                continue
            if line_split[5].strip() != "1":  # 直连盘未写电源标记，返回故障
                slot = line_split[1].strip()
                slot_array = slot_array + '%s ' % slot
                ret = False
                logging.warning('SAS_DISK_POWER_CHECK: sas direct disk power status check failed, slot: %s', slot)
        if ret is True:
            logging.info('SAS_DISK_POWER_CHECK: sas direct disk power status check success')
            return True, ''
        else:
            return False, slot_array[:-1]
    except Exception as exception:
        logging.exception("SAS_DISK_POWER_CHECK: check failed: %s", exception)
        return True, ''


def main():
    patch_map = {'6.1.3': 26, '6.1.5' : 18}
    product_version = get_product_version()
    patch_version = get_patch_version()
    applo_version = get_applo_version()
    logging.info('SAS_DISK_POWER_CHECK: cur version is %s %s', product_version, patch_version)
    logging.info('SAS_DISK_POWER_CHECK: applo version is %s', applo_version)

    try:
        for product, patch in patch_map.items():
            if product in product_version and is_install_specified_patch(patch, patch_version):
                logging.info('SAS_DISK_POWER_CHECK: patch version check success')
                logging.critical('True')
                return 0

        if applo_version != 'apollo-1.1.4.0' and applo_version != 'apollo-1.1.5.0':
            logging.info('SAS_DISK_POWER_CHECK: applo version check ok')
            logging.critical('True')
            return 0

        ret, out = is_direct_disk_power_status_ok()
        if ret:
            logging.critical('True')
        else:
            logging.critical('False')
            logging.critical('10020')
            if patch_version == '':
                logging.critical("{0};{1}".format(product_version, out))
            else:
                logging.critical("{0} {1};{2}".format(product_version, patch_version, out))
        return 0
    except Exception as exception:
        logging.exception("SAS_DISK_POWER_CHECK:check failed: %s", exception)
        logging.critical('True')
        return 0


if __name__ == '__main__':
    sys.exit(main())
