#!/usr/bin/python
# -*- coding: utf-8 -*-
# Copyright (c) Huawei Technologies Co., Ltd. 2018-2022. All rights reserved.
import getopt
import os
import re
import sys
import yaml

from baseline import rpc_opcode
from baseline.return_code import INVALID_NID
from check_util.extend_param_mgr import ExtendParamMgr
from comm_check_func import check_sys_and_hotpatch_version
from config import db as cfg_db
from config import path as cfg_path
from config import env
from config.rpc import RpcRet
from infra.debug.log import swm_logger as log
from infra.rpc.rpc_client import RpcClient
from infra.util import shell
from plat.host.host import IpInfo
from plat.host.host_mgr import HostMgr
from plat.origin_diagnose.diagnose import SwmDiagnose
from service.version.ver_mgr import VerMgr

IP_ENCLOSURE_NID = 1128


class DiskUsageInfo(object):
    """
    Disk usage information class
    """

    def __init__(self, disk_user_id, logic_id, usage, disk_type, *args):
        self.disk_user_id = disk_user_id
        self.logic_id = logic_id
        self.usage = int(usage)
        self.disk_type = disk_type

    def __str__(self):
        print_info = "disk id(%s), logic id(%s), usage(%s), disk_type(%s).". \
            format(self.disk_user_id, self.logic_id, self.usage, self.disk_type)
        return print_info


class CheckResult(object):
    """
    Disk usage check result class
    """

    def __init__(self, max_usage, max_usage_threshold, average_usage,
                 average_usage_threshold, over_threshold_list):
        self.max_usage = max_usage
        self.max_usage_threshold = max_usage_threshold
        self.average_usage = average_usage
        self.average_usage_threshold = average_usage_threshold
        self.over_threshold_list = over_threshold_list


class SASDiskUsageChecker(object):
    """

    """
    _DISK_AVERAGE_USAGE_THRESHOLD = 60
    _DISK_MAX_USAGE_THRESHOLD = 80
    _IGNORE_CHECK_THRESHOLD = 10
    _CHECK_DISK_TYPE = ['SAS', 'SATA', 'NL_SAS', 'SCM']
    _DISK_LIST_FILE_PATH = cfg_path.DISK_LOCATION_TMP_PATH
    _DISK_USAGE_CHECK_DIAGNOSE_CMD = "ld getalldiskuserate"
    _DISK_USAGE_CHECK_CMD = 'diagsh --attach=*_12 --cmd="{0}">{1}'. \
        format(_DISK_USAGE_CHECK_DIAGNOSE_CMD, cfg_path.DISK_LOCATION_USAGE_TMP_PATH)
    _CMD_EXEC_TIMEOUT = 180

    def __init__(self):
        self.all_disk_usage_list = []
        self.need_check_disk_id_list = []
        self.average_usage = 0
        self.max_usage = 0
        self.over_threshold_disk_list = []

    def get_check_result(self):
        """
        Get the max disk usage info, and average disk usage info ,
        and threshold info, and over threshold disk list info.
        :return: CheckResult instance
        """
        self._get_disk_for_check()

        disk_count = len(self.need_check_disk_id_list)
        if disk_count == 0:
            log.info("There have no any disk, no need to check usage.")
            return True, CheckResult(0, self._DISK_MAX_USAGE_THRESHOLD,
                                     0, self._DISK_AVERAGE_USAGE_THRESHOLD, "")

        disk_usage_arr = [int(disk.usage) for disk in self.need_check_disk_id_list]
        self.max_usage = max(disk_usage_arr)
        self.average_usage = sum(disk_usage_arr) // len(self.need_check_disk_id_list)
        self.over_threshold_disk_list = []
        for disk in self.need_check_disk_id_list:
            if disk.usage > self._DISK_AVERAGE_USAGE_THRESHOLD:
                self.over_threshold_disk_list.append(disk.disk_user_id)

        # 只有一个盘有利用率时，只判断盘利用率是否超过最高阈值
        if self.max_usage > self._DISK_MAX_USAGE_THRESHOLD \
                or (len(self.need_check_disk_id_list) > 1 and self.average_usage > self._DISK_AVERAGE_USAGE_THRESHOLD):
            return False, CheckResult(self.max_usage, self._DISK_MAX_USAGE_THRESHOLD,
                                      self.average_usage, self._DISK_AVERAGE_USAGE_THRESHOLD,
                                      self.over_threshold_disk_list)
        return True, CheckResult(0, self._DISK_MAX_USAGE_THRESHOLD,
                                 0, self._DISK_AVERAGE_USAGE_THRESHOLD, "")

    def _get_all_disk_usage_info(self):
        """

        :return:
        """
        if os.path.exists(cfg_path.DISK_LOCATION_USAGE_TMP_PATH):
            os.remove(cfg_path.DISK_LOCATION_USAGE_TMP_PATH)
        ret = shell.call_shell_cmd(self._DISK_USAGE_CHECK_CMD)
        os.chmod(cfg_path.DISK_LOCATION_USAGE_TMP_PATH, 0o640)
        if ret:
            return False

        try:
            with open(cfg_path.DISK_LOCATION_USAGE_TMP_PATH, 'r') as file_to_read:
                for line in file_to_read.readlines():
                    # filter the cmd and title lines
                    if line.find('DiskId') >= 0 or line.find('diagnose') >= 0 \
                            or line.find('diagsh') >= 0:
                        continue

                    line = re.sub(' +', ' ', line)
                    line = line.replace("\t\n", "").replace("\n", "").replace("\t", ",").replace(" ", ",")
                    disk_usage_info = line.split(',')
                    log.info("disk usage info(%s).", disk_usage_info)

                    disk = DiskUsageInfo(*disk_usage_info)
                    # 过滤制定类型的盘，且如果利用率未达到检查阈值，则不忽略
                    if disk.disk_type in self._CHECK_DISK_TYPE \
                            and disk.usage > self._IGNORE_CHECK_THRESHOLD:
                        self.all_disk_usage_list.append(disk)
        except Exception:
            log.exception("Exception")
            return False

        log.info("Get disk(%s) usage infos successfully, total disk(%s).",
                 self._CHECK_DISK_TYPE, len(self.all_disk_usage_list))
        return True

    def _get_disk_for_check(self):
        """

        :return:
        """
        self._get_all_disk_usage_info()
        # DISK_LOCATION_TMP_PATH 配置需要做检查的disk id,可能由SmartKit下发
        if not os.path.exists(cfg_path.DISK_LOCATION_TMP_PATH):
            ExtendParamMgr.get_file_from_other_node(
                cfg_path.DISK_LOCATION_TMP_PATH)
            if not os.path.exists(cfg_path.DISK_LOCATION_TMP_PATH):
                log.info("There have no (%s) file, no need to filter disks.",
                         cfg_path.DISK_LOCATION_TMP_PATH)
                self.need_check_disk_id_list = self.all_disk_usage_list
                return

        disk_id_list = []
        with open(cfg_path.DISK_LOCATION_TMP_PATH) as fd:
            lines = fd.readlines()
        for line in lines:
            disk_id_list.extend(re.split("[;,]", line.replace('\n', '')))
            log.info("Line read from disk info temp file(%s).", disk_id_list)

        self.need_check_disk_id_list = [disk for disk in
                                        self.all_disk_usage_list if
                                        disk.disk_user_id in disk_id_list]
        log.info("The disks need to check are(%s).",
                 [disk.disk_user_id for disk in self.need_check_disk_id_list])
        return


class SSDDiskUsageChecker(SASDiskUsageChecker):
    """
    SSD disk usage check class
    """
    _DISK_AVERAGE_USAGE_THRESHOLD = 60
    _DISK_MAX_USAGE_THRESHOLD = 80
    _IGNORE_CHECK_THRESHOLD = 0
    _CHECK_DISK_TYPE = ['SSD', 'SLC_SSD', 'MLC_SSD', 'NVME_SSD']


class HotUpgChecker(object):
    """

    """
    _GET_ISCSI_HOST_LINK_CMD = "hostlink show iscsi"
    _ROCK_HOST_LINK_ACTICE_VER = "7600501122"
    _GET_ISCSI_ROCE_HOST_LINK_CMD = "hostlink show iscsi,roce"

    @staticmethod
    def _is_path_support_hot_upg(cur_version, dst_version):
        if not os.path.exists(cfg_path.HOT_UPG_PATH_CONFIG_FILE):
            log.info("HOT_UPG_CHECK: Hot upg config file(%s) not existing.", cfg_path.HOT_UPG_PATH_CONFIG_FILE)
            return False
        with open(cfg_path.HOT_UPG_PATH_CONFIG_FILE, 'r') as cfg_fd:
            try:
                cfg_yml = yaml.safe_load(cfg_fd)
                log.info("HOT_UPG_CHECK: Yml content(%s).", cfg_yml)
                paths = cfg_yml.get("FrontHotUpgradePathBlackList", [])
                log.info("HOT_UPG_CHECK: paths(%s).", paths)
                for path in paths:
                    if path.get('src_version', '') == cur_version and \
                            path.get('dst_version', '') == dst_version:
                        log.error("src version(%s), dst version(%s) not support hot upgrade.",
                                  path.get('src_version', ''), path.get('dst_version', ''))
                        return False
            except Exception:
                log.exception("HOT_UPG_CHECK: Failed to read path from yml, exception.")
                return False
        log.info("src version(%s), dst version(%s) support hot upgrade.", cur_version, dst_version)
        return True

    def _have_host_link(self, cur_version):
        """
        Check whether iSCSI host links exist.
        :return:
        """
        if cur_version >= self._ROCK_HOST_LINK_ACTICE_VER:
            cmd = self._GET_ISCSI_ROCE_HOST_LINK_CMD
            expect = 'No iscsi,roce links'
        else:
            cmd = self._GET_ISCSI_HOST_LINK_CMD
            expect = 'No iscsi links'
        log.info("Current version(%s), cmd(%s).", cur_version, cmd)
        ret_val, info = SwmDiagnose().exec_diagnose(cmd)
        log.info("Diagnose cmd exec ret(%s), output info(%s).", ret_val, info)
        if ret_val or not info or \
                str(info[0]).replace('\n', '') == expect:
            log.info("HOT_UPG_CHECK: %s.", expect)
            return False
        return True

    def is_front_support_hot_upg(self):
        """
        check whether the front-end interface card supports hot upgrade
        The conditions for hot upgrade are as follows:
        1) upgrade path supports hot upgrade
        2) no iSCSI host link is available
        3) high device type
        :return:
        """
        board_type = os.getenv('BOARD_TYPE', '')
        high_device = (board_type in [
            'PANGEA_V6R1C00_HI1620_4U4C', 'PANGEA_V6R3C10_HI1620_4U4C', 'PANGEA_V6R3C00_HI1620_4U4C',
            'PANGEA_V6_HI1630_4P_8U2C_FPC', 'PANGEA_V6_HI1630_4P_8U2C'
        ])
        if not high_device:
            log.info("It is not high device type, not support hot upg.")
            return False

        if not os.path.exists(cfg_path.CHECK_EXTEND_PARAM_TMP_FILE):
            ExtendParamMgr.get_file_from_other_node(
                cfg_path.CHECK_EXTEND_PARAM_TMP_FILE)
            if not os.path.exists(cfg_path.CHECK_EXTEND_PARAM_TMP_FILE):
                log.info("No extend param tmp file(%s), "
                         "no need to check hot upg.",
                         cfg_path.CHECK_EXTEND_PARAM_TMP_FILE)
                return False
        res, dst_version = ExtendParamMgr.get_extend_param("targetVersion")
        if not res:
            log.info("No extend param in tmp file(%s), "
                     "no need to check hot upg.",
                     cfg_path.CHECK_EXTEND_PARAM_TMP_FILE)
            return False

        num_ver, cur_version = VerMgr.get_src_version()

        if not dst_version or not cur_version:
            log.error("Dst version(%s), cur version(%s) is not correct.",
                      dst_version, cur_version)
            return False

        # If the upgrade path supports hot upgrade and no
        # iSCSI host link is available,
        # the hot upgrade conditions are met.
        if self._is_path_support_hot_upg(cur_version, dst_version) and \
                not self._have_host_link(num_ver):
            return True
        log.info("Check not pass, may be the path not support "
                 "hot upgrade or have iSCSI initiators.")
        return False


def format_print(usage_print_info, threshold_print_info, check_result):
    usage_print_info = "{0};{1}%;{2}%;{3}%;{4}%".format(usage_print_info, check_result.max_usage,
                                                        check_result.max_usage_threshold,
                                                        check_result.average_usage,
                                                        check_result.average_usage_threshold)
    tmp_str = ','.join(check_result.over_threshold_list)
    threshold_print_info = "{0},{1}".format(threshold_print_info, tmp_str)
    return usage_print_info, threshold_print_info


def check_disk_usage():
    """
    check disk usage
    :return:
    """
    ssd_result, ssd_check_result = SSDDiskUsageChecker().get_check_result()
    sas_result, sas_check_result = SASDiskUsageChecker().get_check_result()
    if os.path.exists(cfg_path.DISK_LOCATION_TMP_PATH):
        os.remove(cfg_path.DISK_LOCATION_TMP_PATH)
    if ssd_result and sas_result:
        print('True')
        print('')
        print('')
        return 0

    usage_print_info = ""
    threshold_print_info = ""
    usage_print_info, threshold_print_info = \
        format_print(usage_print_info, threshold_print_info, sas_check_result)

    usage_print_info, threshold_print_info = \
        format_print(usage_print_info, threshold_print_info, ssd_check_result)

    print('False')
    print('10004')
    if threshold_print_info.lstrip(',').rstrip(',').strip() == '':
        print("{0};{1}".format(usage_print_info.lstrip(';').rstrip(';'), 'NA'))
    else:
        print("{0};{1}".format(usage_print_info.lstrip(';').rstrip(';'),
                               threshold_print_info.lstrip(',').rstrip(',')))
    return 0


def check_front_hot_upg():
    """
    check whether the front-end interface card supports hot upgrade.
    In the fourth line, it must be JSON format and
    the hostCompaEvaluCheck field is mandatory.
    :return:
    """
    hot_upg_checker = HotUpgChecker()
    if hot_upg_checker.is_front_support_hot_upg():
        print('True')
        # 'hostCompaEvaluCheck: True' indicates that the check item is used for host compatibility evaluation.
        print("{'hostCompaEvaluCheck': 'True'}")
    else:
        print('False')
        print("{'hostCompaEvaluCheck': 'True'}")
    return 0


def get_master_id():
    ret, output = SwmDiagnose().exec_diagnose('cls show')
    if ret != 0:
        return INVALID_NID
    re_ret = re.search(r"master_id(\s*):(\s*)(\d+)", output[1])
    if re_ret is not None:
        return int(re_ret.group(3))
    return INVALID_NID


def check_cluster_sn():
    """
    查询集群max_sn是否与主控sn相等
    :param host_list:
    :return:
    """
    resolve_patch = {
        '7600302133': 'SPH10',
    }
    if check_sys_and_hotpatch_version(resolve_patch):
        print("True")
        return 0
    master_id = get_master_id()
    if master_id == INVALID_NID:
        log.error("Failed to get master id")
        print("False")
        return 1
    master_sn = -1
    max_sn = 0
    max_sn_nid = None
    sql = "select max(sn) from %s" % cfg_db.UPGRADE_RECORD_TBL
    log.info("Master id(%s)", master_id)
    for host in HostMgr.get_ctl_host_list():
        host_ips = [ip.addr for ip in host.host_ip if ip.state == IpInfo.STATE_VALID]
        rpc_result, user_result = RpcClient.start_rpc_call(host_ips, rpc_opcode.RPC_OPCODE_EXEC_SQL, sql, timeout=40)
        if rpc_result != RpcRet.OK or not user_result:
            log.warning("Failed to get host(%s) sn, (rpc_result: %s, user_result: %s)", host, rpc_result, user_result)
            print("False")
            return 1
        if user_result[0][0] and int(user_result[0][0]) > max_sn:
            max_sn = int(user_result[0][0])
            max_sn_nid = host.host_id
        if host.host_id == master_id:
            if user_result[0][0]:
                master_sn = int(user_result[0][0])
                log.info("Master sn is %s", master_sn)
            else:
                master_sn = 0
                log.info("Master sn is default 0.")
    if master_sn >= max_sn:
        print("True")
        return 0
    log.error("Cluster max sn belong to nid(%s), nid is(%s).", max_sn_nid, max_sn)
    print("False")
    return 0


def check_enclosure_num():
    """
    查询IP框数量，大于25则报错，判断逻辑在check_xml中
    :param
    :return:
    """
    data_list = HostMgr.get_data_host_list()
    if not data_list:
        log.info("not exist data node.")
        print(0)
        return 0
    print(int(len(data_list) / 2))
    return 0


def check_fw_before_refactor(check_ret_list):
    for ret in check_ret_list[2:-1]:
        # 重构前检查前端卡、后端卡、交换卡、SAS卡、SAS硬盘框、MCU、BBU、管理板
        # 重构前版本中若有不匹配的固件，则incompatible list不为[]
        if ret.find('[]') == -1:
            return False
    return True


def check_fw_ver_compatible_before_refactor():
    if not env.ARM or env.SIMU:
        print('True')
        return 0

    cmd_ret, check_ret_list = SwmDiagnose().exec_diagnose('fw checkall')
    if cmd_ret != 0:
        log.error("Failed to exec fw check.")
        print("False")
        return 1
    if check_fw_before_refactor(check_ret_list):
        print("True")
        return 0
    print("False")
    return 0


def check_base_version_latest(check_ret_list):
    for ret in check_ret_list:
        if ret.find('False') == -1:
            continue
        # bios不匹配跳过
        if ret.find('Bios') != -1:
            continue
        # 控制框不匹配跳过 (控制框:CTE0.A)
        re_ret = re.search(r"CTE[0-9]*\.[ABCD]\s", ret)
        if re_ret is not None:
            continue
        nid = int(ret.strip().split()[0])
        if nid >= IP_ENCLOSURE_NID:
            continue
        return False
    return True


def check_base_is_compatible(check_ret_list):
    for ret in check_ret_list[2:-1]:
        is_compatible = ret.strip().split()[-1]
        if is_compatible == 'False':
            return False
    return True


def check_fw_after_refactor(check_ret_list):
    # 重构后检查前端卡、后端卡、交换卡、SAS卡、SAS硬盘框、MCU、管理板、BIOS、控制框
    if not check_ret_list:
        return True
    # 若不存在is_compatible项，则根据version_latest值判断版本是否兼容（6.1.2RC1版本至6.1.3RC2）
    # 存在is_compatible项，则根据is_compatible判断版本是否兼容，具体看DmiFwVerCheckResult
    if check_ret_list[0].find('is_compatible') == -1:
        return check_base_version_latest(check_ret_list)
    return check_base_is_compatible(check_ret_list)


def check_fw_ver_compatible_after_refactor():
    if not env.ARM or env.SIMU:
        print('True')
        return 0

    cmd_ret, check_ret_list = SwmDiagnose().exec_diagnose('fw check_all')
    if cmd_ret != 0:
        log.error("Failed to exec fw check.")
        print("False")
        return 1

    if check_fw_after_refactor(check_ret_list):
        print("True")
        return 0
    print("False")
    return 0


def show_help():
    usage = (
        "This is used for upgrade pre-check.\n"
        "  -h, --help             show the cmd help info.\n"
        "  -d, --diskusage        check disk usage.\n"
        "  -f, --fronthotupg      check whether the front-end interface card supports hot upgrade.\n"
        "  -n, --check_sn         check whether the master sn is biggest.\n"
        "  -a, --check_enclosure_num           check enclosure num.\n"
        "  -w, --check_fw_ver_compatible_before_refactor   check fw version compatible before refactor.\n"
        "  -W, --check_fw_ver_compatible_after_refactor    check fw version compatible after refactor.\n"
    ).format(os.path.basename(__file__))
    print(usage)


def main(argv=None):
    if argv is None:
        argv = sys.argv
    try:
        opts, args = getopt.getopt(argv[1:], "dhfnawW", ["help", "diskusage", "fronthotupg", "check_sn",
                                                         "check_enclosure_num",
                                                         "check_fw_ver_compatible_before_refactor",
                                                         "check_fw_ver_compatible_after_refactor"])
        for opt, _ in opts:
            if opt in ("-h", "--help"):
                show_help()
                return 0

            if opt in ['-d', '--diskusage']:
                return check_disk_usage()

            if opt in ['-f', '--fronthotupg']:
                return check_front_hot_upg()

            if opt in ['-n', '--check_sn']:
                return check_cluster_sn()

            if opt in ['-a', '--check_enclosure_num']:
                return check_enclosure_num()

            if opt in ['-w', '--check_fw_ver_compatible_before_refactor']:
                return check_fw_ver_compatible_before_refactor()

            if opt in ['-W', '--check_fw_ver_compatible_after_refactor']:
                return check_fw_ver_compatible_after_refactor()

    except Exception as e:
        print("exception")
        print("exception")
        print("exception")
        log.exception(e)
        return 1
    return 0


if __name__ == '__main__':
    sys.exit(main())
