#!/bin/bash
# 内存分段检查
# 脚本检查存在风险输出 True，否则输出 False
# 函数检查存在风险返回 0，否则返回 1

version_615="7600511219"
version_616="7600513292"

function check_apollo_version()
{
    local apollo_version=$(readlink /startup_disk/image/pkg_cur/apollo)
    if [ "${apollo_version}" == "apollo-1.0.0.1" ]; then
        return 0
    fi

    return 1
}

# 检查内存地址存在分段
function check_mem_segmentation()
{
    local seg_idx=($(diagsh --attach="app_data_12" --cmd="dpumm_shm show ramnumainfo" | sed -n '/Shm Ram Numa info/,/NVRAM Numa info/p' | grep "0xffff" | awk '{print $2}'))

    for idx in "${seg_idx[@]}"; do
        if [ "${idx}" == "1" ]; then
            return 0
        fi
    done

    return 1
}

# 检查版本存在问题
function check_version_exit_bug()
{
    local version=$1
    local patch_dir=$2

    # 目标版本是615，补丁版大于等于SPH20不存在问题
    if [ "${version}" == "${version_615}" ]; then
        if [ -f "/startup_disk/image/${version}/hotpatch/${patch_dir}/patch.yml" ]; then
            sphver=$(cat /startup_disk/image/${version}/hotpatch/${patch_dir}/patch.yml | grep "patch_version" | awk -F "SPH" '{print $2}')
            if [ -n "${sphver}" ] && [ "${sphver}" -ge 20 ]; then
                return 1
            fi
        fi
        return 0
    fi

    # 目标版本是616，补丁版大于等于SPH6不存在问题
    if [ "${version}" == "${version_616}" ]; then
        if [ -f "/startup_disk/image/${version}/hotpatch/${patch_dir}/patch.yml" ]; then
            sphver=$(cat /startup_disk/image/${version}/hotpatch/${patch_dir}/patch.yml | grep "patch_version" | awk -F "SPH" '{print $2}')
            if [ -n "${sphver}" ] && [ ${sphver} -ge 6 ]; then
                return 1
            fi
        fi
        return 0
    fi

    return 1
}

# 检查升级目标版本存在问题
function check_target_version_exist_bug()
{
    if [ ! -f "/startup_disk/image/pkg_upd/manifest.yml" ]; then
        return 1
    fi

    local dst_version=$(head /startup_disk/image/pkg_upd/manifest.yml | grep " Version:" | awk '{print $2}')
    check_version_exit_bug "$dst_version" "patch_cur"

    return $?
}

# 检查目标补丁版本存在问题
function check_target_sphver_exist_bug()
{
    if [ ! -f "/OSM/conf/manifest.yml" ]; then
        return 1
    fi

    local cur_version=$(head /OSM/conf/manifest.yml | grep " Version:" | awk '{print $2}')
    check_version_exit_bug "$cur_version" "patch_temp"

    return $?
}

# 检查当前版本存在问题
function check_cur_version_exit_bug()
{
    if [ ! -f "/OSM/conf/manifest.yml" ]; then
        return 1
    fi

    local cur_version=$(head /OSM/conf/manifest.yml | grep " Version:" | awk '{print $2}')
    check_version_exit_bug "$cur_version" "patch_cur"

    return $?
}

# 快速升级评估检查
function pre_upd_evaluate_check()
{
    # 1. 检查apollo版本为1.0.0.1
    check_apollo_version
    if [ $? -ne 0 ]; then
        return 1
    fi

    if [ ! -f "/tmp/check_extend_param" ]; then
        return 1
    fi

    # 2. 检查评估的目标版本存在问题
    # 升级评估：{'targetVersion': '6.1.6', 'hostInfoPkgPath': ''}
    local soft_version=$(cat /tmp/check_extend_param | grep -oP "(?<=targetVersion': ')[^']+(?=')")
    if [ "${soft_version}" == "6.1.5" ] || [ "${soft_version}" == "6.1.6" ]; then
        return 0
    fi

    return 1
}

# 快速升级前检查
function pre_upgrade_check()
{
    # 1. 检查apollo版本为1.0.0.1
    check_apollo_version
    if [ $? -ne 0 ]; then
        return 1
    fi

    # 2. 检查内存分段
    check_mem_segmentation
    if [ $? -ne 0 ]; then
        return 1
    fi

    # 3. 检查升级目标版本存在问题
    check_target_version_exist_bug

    return $?
}

# 热补丁评估检查
function pre_hot_patch_evaluate_check()
{
    # 1. 检查apollo版本为1.0.0.1
    check_apollo_version
    if [ $? -ne 0 ]; then
        return 1
    fi

    # 2. 检查当前软件版本和目标补丁版本存在问题
    if [ ! -f "/OSM/conf/manifest.yml" ]; then
        return 1
    fi

    if [ ! -f "/tmp/check_extend_param" ]; then
        return 1
    fi

    local cur_version=$(head /OSM/conf/manifest.yml | grep " Version:" | awk '{print $2}')
    # 补丁评估：{'targetVersion': '6.1.6.SPH5'}
    local sph_version=$(cat /tmp/check_extend_param | grep -oP "(?<=targetVersion': ')[^']+(?=')" | awk -F "SPH" '{print $2}')
    
    # 软件版本为6.1.5，补丁版本小于SPH20，存在问题
    if [ "${cur_version}" == "${version_615}" ] && [ -n "${sph_version}" ] && [ "${sph_version}" -lt 20 ]; then
        return 0
    fi

    # 软件版本为6.1.6，补丁版本小于SPH6，存在问题
    if [ "${cur_version}" == "${version_616}" ] && [ -n "${sph_version}" ] && [ "${sph_version}" -lt 6 ]; then
        return 0
    fi
    return 1
}

# 打补丁前检查
function pre_hot_patch_check()
{
    # 1. 检查apollo版本为1.0.0.1
    check_apollo_version
    if [ $? -ne 0 ]; then
        return 1
    fi

    # 2. 检查内存分段
    check_mem_segmentation
    if [ $? -ne 0 ]; then
        return 1
    fi

    # 3. 检查目标补丁版本存在问题
    check_target_sphver_exist_bug

    return $?
}

# 日常巡检
function inspection_check
{
    # 1. 检查apollo版本为1.0.0.1
    check_apollo_version
    if [ $? -ne 0 ]; then
        return 1
    fi

    # 2. 检查内存分段
    check_mem_segmentation
    if [ $? -ne 0 ]; then
        return 1
    fi

    # 3. 检查当前版本存在问题
    check_cur_version_exit_bug
    
    return $?
}

g_action=$1
case ${g_action} in
    pre_upd_evaluate)
        pre_upd_evaluate_check
    ;;
    pre_upd)
        pre_upgrade_check
    ;;
    inspection)
        inspection_check
    ;;
    pre_hot_patch_evaluate)
        pre_hot_patch_evaluate_check
    ;;
    pre_hot_patch)
        pre_hot_patch_check
    ;;
    *)
        echo "False"
        exit 0
    ;;
esac

if [ $? -ne 0 ]; then
    echo "False"
else
    echo "True"
fi

exit 0
