# -*- coding: UTF-8 -*-
#  Copyright (c) Huawei Technologies Co., Ltd. 2019-2023. All rights reserved.

import json
import os
import codecs

from psdk.checkitem.common.check_product import check_products
from psdk.platform.base.logger import log


class PyTaskService(object):
    def __init__(self, base_context):
        self.base_context = base_context
        self.work_path = os.path.dirname(__file__)
        self.config_path = os.path.join(self.work_path, "config")
        self.script_path = os.path.join(self.work_path, "scripts")
        self.scene_path = os.path.join(self.work_path, "scene")

    def execute_common_item(self, item_path):
        package_module = self._get_script_package_by_path(item_path)
        module = __import__(package_module, fromlist=["psdk.checkitem.scripts"])
        check_item_obj = getattr(module, "CheckItem")(self.base_context)
        item_config = read_json_file(item_path)
        check_item_obj.set_item_config(item_config)
        status, err_msg = check_item_obj.check()
        result = {"status": status, "detail": err_msg, "originInfo": check_item_obj.get_origin_info()}
        return result

    def _get_script_package_by_path(self, path):
        relative_path = path.replace(self.work_path, "").replace(".json", "").replace(".py", "")
        modules = relative_path.split(os.sep)
        modules = [module for module in modules if module]
        return "psdk.checkitem.scripts." + ".".join(modules[1:])

    def get_check_items(self):
        model = self.base_context.dev_node.model
        version = self.base_context.dev_node.version
        scenario = self.base_context.execute_env.scenario
        tool_name = self.base_context.execute_env.tool_name
        step = self.base_context.execute_env.step
        scene_value = self.base_context.execute_env.scene_value
        log.info("get_check_item:model={},version={},scenario={},tool_name={},step={},scene_value={}"
                 .format(model, version, scenario, tool_name, step, scene_value))
        context_env = ContextEnv(scenario, tool_name, step, model, version, scene_value)
        config_items = CheckItemScan().get_check_items(context_env)
        return {"data": config_items}


class ContextEnv(object):
    """
    :param scene: 一级场景，例如：expansion、fru、inspect、upgrade
    :param tool: 工具, 以expansion为例：exp_ctrl, exp_dd, exp_disk, exp_link
    :param step: 检查场景，要加载哪类检查项，以expansion为例：eval_check, pre_check, post_check
    :param model: 型号
    :param version: 版本
    :param scene_value: 场景参数
    """
    def __init__(self, scene, tool, step, model=None, version=None, scene_value=None):
        self.scene = scene
        self.tool = tool
        self.step = step
        self.model = model
        self.version = version
        self.scene_value = scene_value


class CheckItemScan(object):
    def __init__(self):
        self.work_path = os.path.dirname(__file__)
        self.config_path = os.path.join(self.work_path, "config")
        self.script_path = os.path.join(self.work_path, "scripts")
        self.scene_path = "/".join([self.work_path, "scene"])
        from psdk.platform.adapter import java_common
        self.lang = java_common.get_lang()

    def get_check_items(self, context_env):
        """
        获取指定场景和版本检查项（可提供java侧调用）
        :param context_env: 场景参数
        :return: 过滤后的检查项列表
        """
        check_items = []
        scenario_items = self._get_scenario_items(context_env.scene, context_env.tool, context_env.step)
        # 扩容评估场景，需要加载巡检的检查项
        if context_env.scene == "expansion" and context_env.step == "eval_check":
            scenario_items.extend(self._get_scenario_items("inspect", "inspect_daily", "default"))
        # 健康检查场景，需要单独加载的检查项（取scene_value中的subScene）
        if context_env.scene_value and context_env.scene_value.get('subScene', '') == 'Daily Checks':
            only_health_scene_items = self._get_scenario_items("inspect", "only_health_check_scene", "default")
            self.get_check_items_in_health_scene(scenario_items, only_health_scene_items)
        all_check_items = self._scan_scripts()
        for item in scenario_items:
            item_id = item.get("id")
            item_config = all_check_items.get(item_id)
            if item_config is None:
                continue
            support_products = item_config.get("support_product")
            if not check_products(support_products, context_env.model, context_env.version, context_env.scene,
                                  context_env.tool, context_env.step):
                continue

            resource = item_config["resource_{}".format(self.lang)] or {}
            item.update({
                "itemId": item_config["id"],
                "script": item_config.get("file_path"),
                "name": resource.get("item.name"),
                "method": resource.get("item.method"),
                "criterion": resource.get("item.criterion"),
                "suggestion": resource.get("item.suggestion")
            })

            check_items.append(item)
        return check_items

    def get_check_items_in_health_scene(self, scenario_items, only_health_scene_items):
        for single_item in only_health_scene_items:
            if single_item.get("id") not in [item.get("id") for item in scenario_items]:
                scenario_items.append(single_item)

    def _scan_scripts(self):
        """
        扫描指定路径下的检查项脚本
        :return: 所有检查项配置字典 {id:{support_tool:[]},}
        """
        all_config_items = {}
        for dir_path, _, filenames in os.walk(self.config_path):
            for filename in filenames:
                file_path = os.path.join(dir_path, filename)
                if not file_path.endswith(".json"):
                    continue
                config_dict = read_json_file(file_path)
                config_dict["file_path"] = file_path
                all_config_items[config_dict.get("id", "")] = config_dict
        return all_config_items

    def _get_scenario_items(self, scene, tool, step):
        file_name = "{}_checkitem.json".format(tool)
        config_file = "/".join([self.scene_path, scene, file_name])
        if not os.path.exists(config_file):
            return []
        all_items = read_json_file(config_file)
        filter_items = all_items.get(step, [])
        return filter_items


def read_json_file(file_path):
    config_dict = {}
    try:
        with codecs.open(file_path, "r", encoding="utf-8") as config_f:
            config_dict = json.load(config_f)
    # jython解释器的with在执行__exit__时，有并发问题，这里需要捕获Java异常。
    except:
        # 注意这里打日志不能用error
        log.debug("read config error")
    return config_dict

def get_work_path():
    return  os.path.join(os.path.dirname(__file__))