# -*- coding: UTF-8 -*-
#  Copyright (c) Huawei Technologies Co., Ltd. 2019-2022. All rights reserved.
import re

from psdk.checkitem.common.base_dsl_check import BaseCheckItem

# 系统盘盘符为/dev/sda, 或者/dev/sda + /dev/sdb
SYS_DISK_LIST = ['/dev/sda', '/dev/sdb']
# 系统盘厂商列表及SMART解析项
VENDOR_MAP = {
    "GG51T032S3IN4SS": {"LIFE": "232", "POH": "9"},
    "MC619GXCFCD3S": {"LIFE": "231", "POH": "9"},
    "MC619GXCFCD4S": {"LIFE": "231", "POH": "9"},
    "MC659GXAFCD4T": {"LIFE": "231", "POH": "9"},
    "MC619GXAFCD4T": {"LIFE": "231", "POH": "9"},
    "MC659GXBFCD3T": {"LIFE": "231", "POH": "9"},
    "MC659GXBFCD4T": {"LIFE": "231", "POH": "9"},
    "ME619GXCFCD4T": {"LIFE": "231", "POH": "9"},
    "MD619GXBFCD4S": {"LIFE": "231", "POH": "9"},
    "ME619DXFMDE6TF": {"LIFE": "232", "POH": "9"},
    "MD619HXCHDE3TC": {"LIFE": "232", "POH": "9"}
}

# 寿命标准
WARN_STANDARD = 45000
NO_PASS_STANDARD = 16000
LIFERANGE = (0, 100)
THREHOLD = 4500
LIFEHOLD = 11


class CheckSystemDiskLifeItem(BaseCheckItem):

    def execute(self):
        raise NotImplementedError

    def get_drive_life(self):
        """
        flag: 用于标识巡检项是否通过，0: PASS 1: NO_PASS 2: WARNING
        """
        flag = 0
        """
        不在检查范围则返回巡检通过
        """
        if not self._is_in_check_scope():
            return flag

        """
        获取工具版本信息
        """
        tool_name = ""
        check_tool_version = "exec_mini_system '{} -v' |splitlines |regex '(^Error[\\s\\S]+)'|" \
                             "get_index(0)|get_index(0)".format("disk_repair.sh")
        tool_version = self.dsl(check_tool_version)
        if tool_version:
            check_tool_version_new = "exec_mini_system '{} -v' |splitlines |regex '(^Error[\\s\\S]+)'|" \
                                     "get_index(0)|get_index(0)".format("disktool")
            tool_version_new = self.dsl(check_tool_version_new)
            if not tool_version_new:
                tool_name = "disktool"
        else:
            tool_name = "disk_repair.sh"
        if not tool_name:
            return flag
        dsl_cmd_info = "exec_mini_system '{} -s' |splitlines".format(tool_name)
        disk_info_list = self.dsl(dsl_cmd_info)
        if not disk_info_list:
            return flag
        for sys_letter in SYS_DISK_LIST:
            for disk_info in disk_info_list:
                if sys_letter in disk_info:
                    vendor = self._get_drive_vendor(disk_info_list)
                    flag = self._get_drive_smart(tool_name, sys_letter, vendor, flag)
        return flag

    def _is_in_check_scope(self):
        """
        是否在检查范围
        """
        return True

    def _get_drive_vendor(self, disk_info_list):
        vendor_info = ""
        for vendor_key in VENDOR_MAP:
            for disk_info in disk_info_list:
                if vendor_key in disk_info:
                    vendor_info = vendor_key
                    break
        return vendor_info

    def _get_drive_smart(self, tool_name, sys_letter, vendor, flag):
        poh_info = ""
        life_info = ""
        dsl_cmd_smart = "exec_mini_system '{} -f a {}'|splitlines".format(tool_name, sys_letter)
        dsl_smart_res = self.dsl(dsl_cmd_smart)
        if not vendor:
            return flag
        for smart_info in dsl_smart_res:
            reg1 = r"^\s+{}\s+.*".format(VENDOR_MAP.get(vendor).get("POH"))
            poh_info = re.match(reg1, smart_info)
            if poh_info:
                break
        for smart_info in dsl_smart_res:
            reg2 = r"^{}\s+.*".format(VENDOR_MAP.get(vendor).get("LIFE"))
            life_info = re.match(reg2, smart_info)
            if life_info:
                break
        if poh_info and life_info:
            poh = int(poh_info.group().split()[-1])
            life = int(life_info.group().split()[-1])
            if life != LIFERANGE[1]:
                flag = self._judge_drive_smart(life, poh, flag)
        return flag

    def _judge_drive_smart(self, life, poh, flag):
        if self._judge_drive_smart_exception(life, poh, flag):
            flag = 0
            return flag
        if poh > THREHOLD:
            all_life = int(poh + life * poh / (100 - life))
            if all_life < NO_PASS_STANDARD:
                flag = 1
            elif all_life < WARN_STANDARD:
                flag = 2
        else:
            if 100 - life > LIFEHOLD:
                all_life = int(poh + life * poh / (100 - life - LIFEHOLD))
                if all_life < NO_PASS_STANDARD:
                    flag = 1
                elif all_life < WARN_STANDARD:
                    flag = 2
        return flag

    def _judge_drive_smart_exception(self, life, poh, flag):
        """
        处理例外场景，默认为例外场景不通过，需要继续校验
        """
        return False
