# -*- coding: UTF-8 -*-
#  Copyright (c) Huawei Technologies Co., Ltd. 2019-2024. All rights reserved.

import re
import datetime
from psdk.checkitem.common.base_dsl_check import BaseCheckItem
from psdk.platform.entity.check_status import CheckStatus
from psdk.platform.entity.execute_env import UpgradeExecuteEnv, ExpansionExecuteEnv
from psdk.dsl import fault_mode as ft
from psdk.platform.util.product_util import compare_version

NEW_VERSION = "NEW_VERSION"
OLD_VERSION = "OLD_VERSION"
NOT_SUPPORT = "NOT_SUPPORT"

CERT_RISK = "CERT_RISK"
CERT_NO_RISK = "CERT_NO_RISK"
CERT_NOT_CHECK = "CERT_NOT_CHECK"


class CheckItem(BaseCheckItem):
    model = None
    version = None
    scenario = None

    def execute(self):
        # 获取基本信息
        self.model = self.context.dev_node.model
        self.version = self.context.dev_node.version
        self.scenario = self.context.execute_env.scenario
        risk_cert = set()
        # 获取当前节点存在的证书
        cert_msg = self.dsl("exec_cli 'show certificate general' | horizontal_parser",
                            return_if={ft.CmdNoSupport: "not_support"})
        # 不支持查找证书命令代表没有预置证书
        if cert_msg == "not_support":
            return CheckStatus.PASS, ""
        if not cert_msg:
            return CheckStatus.NOT_CHECK, self.get_msg("query.result.abnormal")
        for cert_detail in cert_msg:
            if cert_detail.get('Type') == 'FTPS Protocol':
                ftp_is_use = self.check_ftp_is_in_use()
                if ftp_is_use == CERT_NOT_CHECK:
                    return CheckStatus.NOT_CHECK, self.get_msg("query.result.abnormal")
                elif ftp_is_use == CERT_RISK:
                    risk_cert.add("FTPS ")
            elif cert_detail.get('Type') == 'HTTPS Protocol':
                http_is_use = self.check_http_is_in_use()
                if http_is_use == CERT_NOT_CHECK:
                    return CheckStatus.NOT_CHECK, self.get_msg("query.result.abnormal")
                elif http_is_use == CERT_RISK:
                    risk_cert.add("HTTPS ")
            elif cert_detail.get('Type') == 'HyperMetro Arbitration':
                hypermetro_is_use = self.check_hyper_metro_cert()
                if hypermetro_is_use == CERT_NOT_CHECK:
                    return CheckStatus.NOT_CHECK, self.get_msg("query.result.abnormal")
                elif hypermetro_is_use == CERT_RISK:
                    risk_cert.add("HyperMetro ")

        if not risk_cert:
            return CheckStatus.PASS, ""
        else:
            return CheckStatus.NOT_PASS, self.get_msg("check.not.pass", "".join(risk_cert))

    # 检查过期时间是否在一天内,包含一天.因为时区调整会导致CLI命令回显产生一天差异.
    def do_check_time_is_in_onedays(self, cert_time, init_cert_time):
        try:
            cert_detailtime_format = datetime.datetime.strptime(cert_time, "%Y-%m-%d")
            certtime_format = datetime.datetime.strptime(init_cert_time, "%Y-%m-%d")
            time_diff = abs(certtime_format - cert_detailtime_format)
            return time_diff.total_seconds() <= 86400
        except ValueError:
            return False

    def get_cert_version(self):
        # 检查是否为DORADO V3
        if re.search("Dorado.*V3", self.model):
            # DORADO V3R2C10-V3R2C20 证书2028年过期
            if (compare_version(self.version, "V300R002C10") >= 0 and
                compare_version(self.version, "V300R002C20") <= 0):
                return NEW_VERSION
            # DORADO V3 其他版本不支持FTP HTTP证书
            else:
                return NOT_SUPPORT
        # 检查是否为Dorado NAS V3R2C10-V3R2C20 证书2028年过期
        elif re.search("Dorado NAS", self.model):
            return NEW_VERSION
        # 检查是否为融合 V3
        elif re.search("V3", self.model):
            # 融合 V3R3C00-V3R6C20 证书2025/2026过期
            if (compare_version(self.version, "V300R003C00") >= 0 and
                compare_version(self.version, "V300R006C20") <= 0):
                return OLD_VERSION
            # 融合 V3R6C30-V3R6C61 证书2028过期
            elif (compare_version(self.version, "V300R006C30") >= 0 and
                  compare_version(self.version, "V300R006C61") <= 0):
                return NEW_VERSION
            # 低于V3R3没有FTP/HTTP
            else:
                return NOT_SUPPORT
        # 检查是否为融合 V5
        elif re.search("V5", self.model):
            # 融合 V5R7C000-V5R7C010 证书2025/2026过期
            if (compare_version(self.version, "V500R007C00") >= 0 and
                compare_version(self.version, "V500R007C10") <= 0):
                return OLD_VERSION
            # 融合V5其他版本 证书2028过期
            else:
                return NEW_VERSION
        # 其他版本不支持FTP/HTTP
        return NOT_SUPPORT

    # 老版本没有显示证书指纹，使用过期时间判断
    def check_ftp_with_old_version(self):
        # 检查证书是否为预置证书(1、没这个命令)
        expire_time = self.dsl(
            "exec_cli 'show certificate general type=ftps_protocol|filterColumn include columnList=Expire\\sTime' | horizontal_parser",
            return_if={ft.CmdNoSupport: "not_support"})
        # 命令不支持则说明没有这个证书 无风险
        if expire_time == "not_support":
            return CERT_NO_RISK
        if not expire_time:
            return CERT_NOT_CHECK
        # 若过期时间为2026-01-18或2028-05-20则为预置证书
        if (self.do_check_time_is_in_onedays(expire_time[0].get('Expire Time'), '2026-01-18') or
                self.do_check_time_is_in_onedays(expire_time[0].get('Expire Time'), '2028-05-20')):
            # 检查ftp服务状态
            ftp_stat = self.dsl(
                "exec_cli 'show service ftp|filterColumn include columnList=Running\\sStatus' | vertical_parser")
            if not ftp_stat:
                return CERT_NOT_CHECK
            # ftp服务在运行 上报风险
            if ftp_stat[0].get('Running Status') == 'Running':
                return CERT_RISK
            else:
                return CERT_NO_RISK
        # 若不为预置证书 上报无风险
        return CERT_NO_RISK

    def check_http_with_old_version(self):
        # 检查证书是否为预置证书(1、没这个命令)
        expire_time = self.dsl(
            "exec_cli 'show certificate general type=https_protocol|filterColumn include columnList=Expire\\sTime' | horizontal_parser",
            return_if={ft.CmdNoSupport: "not_support"})
        # 命令不支持则说明没有这个证书 无风险
        if expire_time == "not_support":
            return CERT_NO_RISK
        if not expire_time:
            return CERT_NOT_CHECK
        # 若过期时间为2025-06-05或2028-05-20则为预置证书
        if (self.do_check_time_is_in_onedays(expire_time[0].get('Expire Time'), '2025-06-05') or
                self.do_check_time_is_in_onedays(expire_time[0].get('Expire Time'), '2028-05-20')):
            # 检查ftp服务状态
            http_stat = self.dsl(
                "exec_cli 'show service http|filterColumn include columnList=Running\\sStatus' | vertical_parser")
            if not http_stat:
                return CERT_NOT_CHECK
            # ftp服务在运行 上报风险
            if http_stat[0].get('Running Status') == 'Running':
                return CERT_RISK
            else:
                return CERT_NO_RISK
        # 若不为预置证书 上报无风险
        return CERT_NO_RISK

    # 新版本证书使用指纹来判断是否为预置证书
    def check_ftp_with_new_version(self):
        # 检查证书是否为预置证书(1、没这个命令)
        ftp_finger_print = self.dsl(
            "exec_cli 'show certificate general type=ftps_protocol detail=yes|filterColumn include columnList=Fingerprint' | vertical_parser",
            return_if={ft.CmdNoSupport: "not_support"})
        # 命令不支持则说明没有这个证书 无风险
        if ftp_finger_print == "not_support":
            return CERT_NO_RISK
        if not ftp_finger_print:
            return CERT_NOT_CHECK
        # 若为预置证书 下述两个指纹分别是2025年过期的和2028年过期的ftp证书指纹
        if ftp_finger_print[0].get('Fingerprint') == '3C:1F:5F:A8:83:21:17:98:AE:FC:35:03:B9:57:DF:AF:41:E8:2A:5F' or \
                ftp_finger_print[0].get('Fingerprint') == '0F:B7:77:4E:CC:C7:CC:94:68:A4:9A:FA:34:4F:69:8A:40:84:86:B5':
            # 检查ftp服务状态
            ftp_stat = self.dsl(
                "exec_cli 'show service ftp|filterColumn include columnList=Running\\sStatus' | vertical_parser")
            if not ftp_stat:
                return CERT_NOT_CHECK
            # ftp服务在运行 上报风险
            if ftp_stat[0].get('Running Status') == 'Running':
                return CERT_RISK
            else:
                return CERT_NO_RISK
        # 若不为预置证书 上报无风险
        return CERT_NO_RISK

    def check_http_with_new_version(self):
        # 检查证书是否为预置证书
        http_finger_print = self.dsl(
            "exec_cli 'show certificate general type=https_protocol detail=yes|filterColumn include columnList=Fingerprint' | vertical_parser",
            return_if={ft.CmdNoSupport: "not_support"})
        # 命令不支持则说明没有这个证书 无风险
        if http_finger_print == "not_support":
            return CERT_NO_RISK
        if not http_finger_print:
            return CERT_NOT_CHECK
        # 若为预置证书 下述两个指纹分别是2025年过期的和2028年过期的http证书指纹
        if http_finger_print[0].get('Fingerprint') == 'AF:3D:71:2B:97:32:FA:7C:4E:CB:18:39:4A:DB:75:82:4F:D8:01:97' or \
                http_finger_print[0].get('Fingerprint') == '16:82:82:D2:57:9E:77:01:62:A2:E2:65:9B:11:9B:B7:F3:8E:00:7A':
            # 检查http服务状态
            http_stat = self.dsl(
                "exec_cli 'show service http|filterColumn include columnList=Running\\sStatus' | vertical_parser")
            if not http_stat:
                return CERT_NOT_CHECK
            # ftp服务在运行 上报风险
            if http_stat[0].get('Running Status') == 'Running':
                return CERT_RISK
            else:
                return CERT_NO_RISK
        # 若不为预置证书 上报无风险
        return CERT_NO_RISK

    def check_ftp_is_in_use(self):
        cert_version = self.get_cert_version()
        if cert_version == OLD_VERSION:
            if self.model == "2800 V5" and self.version == 'V500R007C10':
                return self.check_ftp_with_old_version_notype()
            return self.check_ftp_with_old_version()
        elif cert_version == NEW_VERSION:
            return self.check_ftp_with_new_version()
        elif cert_version == NOT_SUPPORT:
            return CERT_NO_RISK
        return CERT_NO_RISK

    def check_http_is_in_use(self):
        cert_version = self.get_cert_version()
        if cert_version == OLD_VERSION:
            if self.model == "2800 V5" and self.version == 'V500R007C10':
                return self.check_http_with_old_version_notype()
            return self.check_http_with_old_version()
        elif cert_version == NEW_VERSION:
            return self.check_http_with_new_version()
        elif cert_version == NOT_SUPPORT:
            return CERT_NO_RISK
        return CERT_NO_RISK

    def check_hyper_metro_default(self):
        # 检查证书是否为预置证书
        hypermetro_finger_print = self.dsl(
            "exec_cli 'show certificate general type=hypermetro_arbitration |filterColumn include columnList=CA\\sFingerprint' | horizontal_parser",
            return_if={ft.CmdNoSupport: "not_support"})
        # 若CA指纹为2F:73:81:36:78:77:BA:D4:8E:4D:01:36:E6:CA:F4:E4:39:63:04:21 且证书过期时间为2025-06-05 则为预置证书
        #此命令只会在存在双活证书的场景下调用，不支持说明是老版本不支持查询finger,这种情况下继续使用过期时间检查
        if hypermetro_finger_print == "not_support" or hypermetro_finger_print[0].get(
                'CA Fingerprint') == '2F:73:81:36:78:77:BA:D4:8E:4D:01:36:E6:CA:F4:E4:39:63:04:21':
            # 检查证书过期时间
            expire_time = self.dsl(
                "exec_cli 'show certificate general type=hypermetro_arbitration |filterColumn include columnList=Expire\\sTime' | horizontal_parser",
                return_if={ft.CmdNoSupport: "not_support"})
            # 命令不支持则设置未完成检查
            if expire_time == "not_support":
                return CERT_NO_RISK
            if not expire_time:
                return CERT_NOT_CHECK
            # 若过期时间为2025-06-05则为预置证书
            if self.do_check_time_is_in_onedays(expire_time[0].get('Expire Time'), '2025-06-05'):
                return CERT_RISK
            else:
                return CERT_NO_RISK
        # 若CA指纹为60:B2:B7:BE:EF:2F:83:2C:59:6E:FF:75:CD:4C:F8:2B:19:B0:79:04 且证书过期时间为2029-12-05 则为预置证书
        elif hypermetro_finger_print[0].get(
                'CA Fingerprint') == '60:B2:B7:BE:EF:2F:83:2C:59:6E:FF:75:CD:4C:F8:2B:19:B0:79:04':
            # 检查证书过期时间
            expire_time = self.dsl(
                "exec_cli 'show certificate general type=hypermetro_arbitration |filterColumn include columnList=Expire\\sTime' | horizontal_parser",
                return_if={ft.CmdNoSupport: "not_support"})
            # 命令不支持则设置未完成检查
            if expire_time == "not_support":
                return CERT_NO_RISK
            if not expire_time:
                return CERT_NOT_CHECK
            # 若过期时间为2029-12-05则为预置证书
            if self.do_check_time_is_in_onedays(expire_time[0].get('Expire Time'), '2029-12-05'):
                return CERT_RISK
            else:
                return CERT_NO_RISK
        # 若不为预置证书 上报无风险
        return CERT_NO_RISK

    def check_hyper_metro_cert(self):
        hypermetro_default = self.check_hyper_metro_default()
        if hypermetro_default == CERT_RISK:
            quorum_server = self.dsl("exec_cli 'show quorum_server general' | horizontal_parser")
            if not quorum_server:
                return CERT_NO_RISK
            else:
                return CERT_RISK
        return hypermetro_default

    def check_http_with_old_version_notype(self):
        expire_time = 'not_support'
        cert_msg = self.dsl("exec_cli 'show certificate general' | horizontal_parser",
                            return_if={ft.CmdNoSupport: "not_support"})
        for cert_detail in cert_msg:
            if cert_detail.get('Type') == "HTTPS Protocol":
                expire_time = cert_detail.get('Expire Time')
            else:
                continue
        if expire_time == "not_support":
            return CERT_NO_RISK
        if not expire_time:
            return CERT_NOT_CHECK
        # 若过期时间为2025-06-05或2028-05-20则为预置证书
        if (self.do_check_time_is_in_onedays(expire_time, '2025-06-05') or
                self.do_check_time_is_in_onedays(expire_time, '2028-05-20')):
            # 检查ftp服务状态
            http_stat = self.dsl(
                "exec_cli 'show service http|filterColumn include columnList=Running\\sStatus' | vertical_parser")
            if not http_stat:
                return CERT_NOT_CHECK
            # ftp服务在运行 上报风险
            if http_stat[0].get('Running Status') == 'Running':
                return CERT_RISK
            else:
                return CERT_NO_RISK
        # 若不为预置证书 上报无风险
        return CERT_NO_RISK

    def check_ftp_with_old_version_notype(self):
        expire_time = 'not_support'
        cert_msg = self.dsl("exec_cli 'show certificate general' | horizontal_parser",
                            return_if={ft.CmdNoSupport: "not_support"})
        for cert_detail in cert_msg:
            if cert_detail.get('Type') == "FTPS Protocol":
                expire_time = cert_detail.get('Expire Time')
            else:
                continue
        # 命令不支持则说明没有这个证书 无风险
        if expire_time == "not_support":
            return CERT_NO_RISK
        if not expire_time:
            return CERT_NOT_CHECK
        # 若过期时间为2026-01-18或2028-05-20则为预置证书
        if (self.do_check_time_is_in_onedays(expire_time, '2026-01-18') or
                self.do_check_time_is_in_onedays(expire_time, '2028-05-20')):
            # 检查ftp服务状态
            ftp_stat = self.dsl(
                "exec_cli 'show service ftp|filterColumn include columnList=Running\\sStatus' | vertical_parser")
            if not ftp_stat:
                return CERT_NOT_CHECK
            # ftp服务在运行 上报风险
            if ftp_stat[0].get('Running Status') == 'Running':
                return CERT_RISK
            else:
                return CERT_NO_RISK
        # 若不为预置证书 上报无风险
        return CERT_NO_RISK