# -*- coding: UTF-8 -*-
#  Copyright (c) Huawei Technologies Co., Ltd. 2019-2021. All rights reserved.

from psdk.checkitem.common.base_dsl_check import BaseCheckItem
from psdk.platform.entity.check_status import CheckStatus
from psdk.dsl import fault_mode as ft


class CheckItem(BaseCheckItem):
    def execute(self):
        # 步骤2 步骤2 执行命令：show disk general |filterColumn include columnList=ID,Disk\sDomain\sID,Type|filterRow column=Type predict=equal_to value=NearLine\sSAS，记录Type字段值为NearLine SAS的盘所对应的Disk Domain ID的值；
        disk_list = self.dsl(
            "exec_cli 'show disk general |filterColumn include columnList=ID,Disk\sDomain\sID,Type|filterRow column=Type predict=equal_to value=NearLine\sSAS' | horizontal_parser "
        )
        if not disk_list:
            return CheckStatus.PASS, ""
        # 步骤3 执行命令：show storage_pool general |filterColumn include columnList=ID,Disk\sDomain\sID，记录步骤2中Disk Domain ID所对应的存储池ID；
        storage_pool_ret = self.dsl(
            "exec_cli 'show storage_pool general |filterColumn include columnList=ID,Disk\sDomain\sID' | horizontal_parser "
        )
        if not storage_pool_ret:
            return CheckStatus.PASS, ""

        storage_pool_id_list = self.get_pool_list(disk_list, storage_pool_ret)
        if not storage_pool_id_list:
            return CheckStatus.PASS, ""
        # 步骤4
        err_storage_pool_id_list = self.check_storage_pool_name(
            storage_pool_id_list
        )
        # 步骤5
        flag, err_file_system_id_list = self.check_file_system(
            err_storage_pool_id_list
        )
        if flag is True:
            return CheckStatus.PASS, ""
        self.err_msg_list = list()
        if err_file_system_id_list:
            self.err_msg_list.append(
                self.get_msg(
                    "check.not.pass.file.system",
                    ",".join(err_file_system_id_list),
                )
            )
        # 步骤6
        self.check_lun_dedup(err_storage_pool_id_list)
        if self.err_msg_list:
            return CheckStatus.NOT_PASS, "\n".join(self.err_msg_list)
        return CheckStatus.PASS, ""

    def check_lun_dedup(self, err_pool_id_list):
        """
        步骤6 执行命令：show lun dedup_compress|filterColumn include columnList=ID,Pool\sID，记录步骤3中存储池ID所创建的重删压缩LUN。
        :param err_pool_id_list:
        :return:
        """
        err_lun_id_list = list()
        lun_dedup_list = self.dsl(
            "exec_cli 'show lun dedup_compress|filterColumn include columnList=ID,Pool\sID' | horizontal_parser "
        )
        for lun in lun_dedup_list:
            if lun.get("Pool ID") in err_pool_id_list:
                err_lun_id_list.append(lun.get("ID"))
        if err_lun_id_list:
            self.err_msg_list.append(
                self.get_msg("check.not.pass.lun", ",".join(err_lun_id_list))
            )

    def check_file_system(self, err_pool_id_list):
        """
        执行命令：show file_system general|filterColumn include columnList=ID,Pool\sID,Dedup\sEnabled,Compression\sEnabled，记录步骤3中存储池ID所创建的文件系统ID、Dedup Enabled、Compression Enabled字段值；
        :param err_pool_id_list:
        :return:
        """
        err_file_system_id_list = list()

        file_system_ret = self.dsl(
            "exec_cli 'show file_system general|filterColumn include columnList=ID,Pool\sID,Dedup\sEnabled,Compression\sEnabled' | horizontal_parser ",
            return_if={ft.CmdNoSupport: "notSupport"},
        )
        if file_system_ret == "notSupport":
            return True, err_file_system_id_list
        for file_system in file_system_ret:
            dedup_enabled = file_system.get("Dedup Enabled")
            compression_enabled = file_system.get("Compression Enabled")
            pool_id = file_system.get("Pool ID")
            if (
                dedup_enabled == "Yes" or compression_enabled == "Yes"
            ) and pool_id in err_pool_id_list:
                err_file_system_id_list.append(file_system.get("ID"))
        return False, err_file_system_id_list

    def check_storage_pool_name(self, storage_pool_id_list):
        """
        步骤4 执行命令：show storage_pool tier pool_id=Pool\sID，记录存在容量层（Name字段值为Capacity）的存储池ID；
        :param storage_pool_id_list:
        :return:
        """
        err_storage_pool_id_list = list()

        for storage_pool_id in storage_pool_id_list:
            dsl_cmd = "exec_cli 'show storage_pool tier pool_id={}' | vertical_parser ".format(
                storage_pool_id
            )
            storage_pool_ret = self.dsl(dsl_cmd)
            err_storage_pool_id_list.extend(
                [
                    pool.get("Pool ID")
                    for pool in storage_pool_ret
                    if pool.get("Name") == "Capacity"
                ]
            )
        return err_storage_pool_id_list

    @staticmethod
    def get_pool_list(disk_domain_list, storage_pool_ret):
        """
        记录步骤2中Disk Domain ID所对应的存储池ID；
        :param disk_domain_list:  查询的硬盘域
        :param storage_pool_ret:  查询的存储池
        :return: storage_pool_id_list
        """
        disk_domain_id_list = [
            disk_domain.get("Disk Domain ID")
            for disk_domain in disk_domain_list
            if disk_domain.get("Disk Domain ID") != "--"
        ]

        storage_pool_id_list = [
            storage_pool.get("ID")
            for storage_pool in storage_pool_ret
            if storage_pool.get("Disk Domain ID") in disk_domain_id_list
        ]

        return storage_pool_id_list
