# -*- coding: UTF-8 -*-
#  Copyright (c) Huawei Technologies Co., Ltd. 2019-2021. All rights reserved.

from psdk.checkitem.common.base_dsl_check import BaseCheckItem
from psdk.platform.entity.check_status import CheckStatus
from psdk.dsl import fault_mode as ft

# 32位文件系统支持的最大对象号
MAX_32_BIT_ALLOC_ID = 0xFFFFFFFF

# 巡检判断32位开关打开文件系统已分配对象的阈值
MAX_32_BIT_ALLOC_ID_THRESHOLD = (int(MAX_32_BIT_ALLOC_ID * 0.9))

# 文件系统ID转换成对象集ID的初始值
BEGIN_OBJECT_SET_ID = 274877906944


class CheckItem(BaseCheckItem):
    def execute(self):
        # 步骤2 执行命令：show file_system general|filterColumn include columnList=ID,Work\sController,Traverse\sDir\sAdapter，查询所有文件系统ID、工作控制器Work Controller和32位主机适配开关；
        file_system_list = self.dsl(
            "exec_cli 'show file_system general|filterColumn include columnList=ID,Work\sController,Traverse\sDir\sAdapter' | horizontal_parser ",
            return_if={ft.CmdNoSupport: "notSupport", },
        )
        # 2100V3; 2200V3-8G；2800V3等不支持文件系统的设备，直接返回通过
        if file_system_list == "notSupport":
            return CheckStatus.PASS, ""

        # 不存在文件系统，直接返回通过
        if not file_system_list:
            return CheckStatus.PASS, ""

        # 获取所有文件系统32-bit开关状态为Yes的文件系统
        u32_file_system_dict = self.check_file_system(file_system_list)
        if not u32_file_system_dict:
            return CheckStatus.PASS, ""

        # 获取所有32-bit开关打开的文件系统的已分配对象号
        flag, file_system_next_alloc_id = self.check_file_system_alloc_id(u32_file_system_dict)

        # 查询命令不支持，意味着非超级管理员不能进入developer下，提示切换权限用户
        if not flag:
            return CheckStatus.NOT_CHECK, self.get_msg("check.not.finish")

        if not file_system_next_alloc_id:
            return CheckStatus.PASS, ""

        # 查询32-bit开关打开的文件系统，已分配id是否超过阈值
        error_file_system_list = self.check_error_file_system(file_system_next_alloc_id)
        if not error_file_system_list:
            return CheckStatus.PASS, ""

        return CheckStatus.NOT_PASS, self.get_msg("check.not.pass.file.system", ",".join(error_file_system_list))

    @staticmethod
    def check_file_system(file_system_list):
        """
        存在文件系统的“32位主机适配开关”为“Yes”.
        :param file_system_list:待遍历文件系统链表
        :return:32位开关打开的文件系统
        """
        u32_file_system_dict = dict()
        for file_system in file_system_list:
            if file_system.get("Traverse Dir Adapter") == "Yes":
                u32_file_system_dict[file_system.get("ID")] = file_system.get("Work Controller")

        return u32_file_system_dict

    def check_file_system_alloc_id(self, u32_file_system_dict):
        """
        步骤4 查询32主机适配开关为Yes的文件系统，执行命令：show space object_set_info object_set_id=? controller=?|filterColumn include columnList=Next\sAlloc\sID
        :param u32_file_system_dict:待遍历32位开关打开的文件系统
        :return:文件系统已分配对象
        """
        # 不能进入developer的用户，直接返回操作未完成
        dlp_cmd = "exec_cli 'change user_mode current_mode user_mode=developer'"
        developer_info = self.dsl(dlp_cmd, return_if={ft.CmdNoSupport: "notSupport", })
        if developer_info == "notSupport":
            return False, ""

        file_system_next_alloc_id = list()
        for file_system_id, ctrl_id in u32_file_system_dict.items():
            object_set_id = str(int(file_system_id) + BEGIN_OBJECT_SET_ID)
            dsl_cmd = "exec_developer 'show space object_set_info object_set_id={} controller={}|filterColumn include columnList=Next\sAlloc\sID' | vertical_parser ".format(
                object_set_id, ctrl_id
            )
            file_system_next_alloc_info = self.dsl(dsl_cmd, return_if={ft.CmdNoSupport: "notSupport", })
            if file_system_next_alloc_info == "notSupport":
                return False, ""

            if file_system_next_alloc_info:
                file_system_next_alloc_id.append({file_system_id: file_system_next_alloc_info[0].get("Next Alloc ID")})
        return True, file_system_next_alloc_id

    @staticmethod
    def check_error_file_system(file_system_next_alloc_id):
        """
        存在文件系统已分配id > 3865470565(4294967295 * 0.9)，则认为是检查不通过
        :param file_system_next_alloc_id:待遍历文件系统链表
        :return:异常文件系统链表
        """
        error_alloc_id_list = [
            list(file_system.keys())[0]
            for file_system in file_system_next_alloc_id
            if (
                int(list(file_system.values())[0]) > MAX_32_BIT_ALLOC_ID_THRESHOLD
            )
        ]
        return error_alloc_id_list
