# -*- coding: UTF-8 -*-
#  Copyright (c) Huawei Technologies Co., Ltd. 2024-2024. All rights reserved.
# 低端BBU更换后故障不恢复风险检查（巡检 非内置）

import re

from psdk.checkitem.common.base_dsl_check import BaseCheckItem
from psdk.dsl.dsl_common import get_version_info
from psdk.platform.entity.check_status import CheckStatus
from psdk.platform.util.product_util import compare_patch_version


class CheckItem(BaseCheckItem):
    def execute(self):
        self.logger.info("==========[check_bbu_fault_recovery_risk] Enter execute.==========")
        # 低端型号，在json中限制
        # 风险BBU
        risk_bbu = self.get_risk_bbu()

        # 不存在风险BBU
        if not risk_bbu:
            self.logger.info("[check_bbu_fault_recovery_risk] The risky BBU does not exist.")
            return CheckStatus.PASS, ""

        # 当前设备是否安装补丁
        result, suggestion_version = self.is_hotpatch_installed()
        if result:
            # 版本存在对应补丁且已安装，或版本不存在对应补丁，检查通过
            self.logger.info("[check_bbu_fault_recovery_risk] No hot patch exists or the hot patch has been installed.")
            return CheckStatus.PASS, ""
        # 提示用户打补丁
        return CheckStatus.WARNING, self.get_msg("check.not.pass", " ".join(risk_bbu), suggestion_version)

    def get_risk_bbu(self):
        """
        获取风险BBU：固件版本<60.00.13T12，健康状态为Fault,存在模块故障
        """
        risk_bbu = []
        bbu_list = self.dsl("exec_cli 'show bbu general' | horizontal_parser")
        for bbu in bbu_list:
            bbu_id = bbu.get("ID", "")
            # 检查BBU固件版本是否为60.00.13T12之前
            results = self.dsl("exec_cli 'show bbu general bbu_id=%s' | vertical_parser" % bbu_id)
            if not self.is_bbu_version_match(results[0].get("Firmware Version")):
                continue
            # 检查BBU的健康状态是否为Fault
            if not self.is_bbu_health_status_fault(bbu.get("Health Status", "")):
                continue
            # 检查BBU是否存在模块故障
            if not self.is_bbu_model_alarm_exist(bbu_id):
                continue
            risk_bbu.append(bbu_id)
        return risk_bbu

    def is_bbu_version_match(self, fw_ver):
        """
        检查方法: 步骤3 执行命令：show bbu general bbu_id=ID（更换的BBU ID），查询BBU固件版本和健康状态；
        检查标准：
        2 若步骤3中查询BBU固件版本为60.00.13T12之前，则继续检查，否则检查结果为通过；
        """
        # 60.00.13T12
        base_version = (60, 0, 13, 12)
        self.logger.info("[check_bbu_fault_recovery_risk] BBU firmware version is {}.".format(fw_ver))
        if not re.match(r"60\.(\d+)\.(\d+)T(\d+)", fw_ver):
            return False
        for i, v in enumerate(re.findall("\d+", fw_ver)):
            if int(v) == base_version[i]:
                continue
            return int(v) < base_version[i]
        return False

    def is_bbu_health_status_fault(self, health_status):
        """
        检查方法: 步骤3 执行命令：show bbu general bbu_id=ID（更换的BBU ID），查询BBU固件版本和健康状态；
        检查标准：
        3 若步骤3中查询BBU健康状态为Fault，则继续检查，否则检查结果为通过；
        """
        self.logger.info("[check_bbu_fault_recovery_risk] BBU health status is {}.".format(health_status))
        return health_status == "Fault"

    def is_bbu_model_alarm_exist(self, selected_bbu):
        """
        检查方法：步骤4 执行命令：show alarm，查看告警信息。
        检查标准：4 若步骤4中已更换的BBU存在ID为0xF00D2001C的告警，则检查为不通过，否则检查为通过。
        """
        alarm_lst = self.dsl("exec_cli 'show alarm|filterColumn include columnList=Sequence,ID"
                             "|filterRow column=ID predict=match value={}' | horizontal_parser".format("0xF00D2001C"))
        for alarm in alarm_lst:
            alarm_info = self.dsl("exec_cli 'show alarm sequence={}' | vertical_parser".format(alarm.get("Sequence")))
            detail = alarm_info[0].get("Detail")
            bbu_module = re.findall(".*BBU module.*\\((.*)\\)", detail)
            self.logger.info("[check_bbu_fault_recovery_risk] Alarm detail is {}".format(bbu_module))
            if not bbu_module:
                continue
            msg_lst = bbu_module[0].split(',')
            if len(msg_lst) < 2:
                continue
            ctrl = "".join(msg_lst[0].strip().split()[2:])
            bbu = "".join(msg_lst[1].strip().split()[2:])
            # 兼容低端环境，BBU模块为CTE0.A.BBU的格式，已包含控制框ID，无需再拼接
            bbu_id = bbu if bbu.startswith(ctrl + ".") else "{}.{}".format(ctrl, bbu)
            self.logger.info("[check_bbu_fault_recovery_risk] Fault bbu is {}".format(bbu_id))
            if bbu_id == selected_bbu:
                break
        else:
            self.logger.info("[check_bbu_fault_recovery_risk] No alarm is generated on the device.")
            return False
        return True

    def is_hotpatch_installed(self):
        '''
        检查当前设备是否安装指定的补丁版本，若不涉及或已安装，则返回True，若未安装，则返回False。
        '''
        hotpatch_dict = {
            "6.1.2": "SPH60",
            "6.1.6": "SPH10",
            "6.1.5": "SPH36",
            "V500R007C71SPC100": "V500R007C71SPH150",
            "V500R007C60SPC300": "V500R007C60SPH335",
            "V500R007C60SPC100": "V500R007C60SPH115"
        }
        version_info = get_version_info(self.dsl)
        software_version = version_info.get("base_version").get("Current Version")
        hotpatch_version = version_info.get("patch_version").get("Current Version")
        self.logger.info("[check_bbu_fault_recovery_risk] Software Version:{}, Hotpatch Version:{}."
                         .format(software_version, hotpatch_version))

        # 当前版本不涉及BBU补丁
        if software_version not in hotpatch_dict:
            self.logger.info("[check_bbu_fault_recovery_risk] The current version does not have a BBU patch.")
            return True, None
        base_version = hotpatch_dict.get(software_version)
        # 用于错误提示
        suggestion_version = hotpatch_dict.get(software_version) if software_version.startswith("V") else \
            "{}.{}".format(software_version, hotpatch_dict.get(software_version))

        return compare_patch_version(hotpatch_version, base_version) >= 0, suggestion_version
