# -*- coding: UTF-8 -*-
#  Copyright (c) Huawei Technologies Co., Ltd. 2023-2024. All rights reserved.

# BBU寿命到期补丁升级路径检查（升级前工勘 非内置 BUGLIST202307100007）
# 说明：设备升级不在此处实现，由工具在升级策略中配置

from psdk.checkitem.common.base_dsl_check import BaseCheckItem
from psdk.platform.entity.check_status import CheckStatus
from psdk.platform.protocol.cli_common import get_system_version_without_ret
from psdk.platform.util.product_util import compare_patch_version


class CheckItem(BaseCheckItem):
    def execute(self):
        self.logger.info("[check_bbu_life_expire_patch_ver] Enter execute.")
        # 检查当前版本是否为指定版本
        flag = self.is_current_version_matches()
        if not flag:
            self.logger.info("[check_bbu_life_expire_patch_ver] "
                             "The current baseline version does not match.")
            return CheckStatus.PASS, ""
        # 检查设备型号是否为2200 V3、2600 V3，这两种型号的备电为CBU，而非BBU
        if self.is_product_model_matches():
            self.logger.info("[check_bbu_life_expire_patch_ver] "
                             "No BBU is installed on the device.")
            return CheckStatus.PASS, ""
        # 检查升级的目标版本是否为指定版本
        flag, target_version = self.is_target_version_matches()
        if not flag:
            self.logger.info("[check_bbu_life_expire_patch_ver] "
                             "The target version does not match.")
            return CheckStatus.PASS, ""
        # 基线版本对应的热补丁版本
        target_version_sph = "V500R007C30SPH139" if target_version.startswith("V5") else "V300R006C50SPH139"
        # 检查带补丁升级目标补丁是否为SPH139之前版本
        patch_ver = self.context.execute_env.java_env.getUpgPatchPkgVer()
        self.logger.info("[check_bbu_life_expire_patch_ver] patch_ver = %s." % patch_ver)
        if patch_ver is not None:
            self.logger.info("[check_bbu_life_expire_patch_ver] The current scenario is device upgrade.")
            patch_ver = str(patch_ver)  # 接口返回值为None或unicode类型
            if not patch_ver or compare_patch_version(patch_ver, target_version_sph) == -1:
                return CheckStatus.PASS, ""

        return CheckStatus.NOT_PASS, self.get_err_msg("check.not.pass", "item.suggestion",
                                                      err_args=(target_version_sph))

    def is_current_version_matches(self):
        """
        检查方法:执行命令：show upgrade package，查询当前版本；
        检查标准:若当前版本为V300R001C20SPC200、V300R002C10SPC100、V300R002C10SPC200、V300R003C00SPC100、
        V300R003C10SPC100、V300R003C20、V300R003C20SPC200、V300R006C00SPC100，则进行下一步检查，否则退出检查并返回通过；
        """
        matches_version = (
            "V300R001C20SPC200",
            "V300R002C10SPC100",
            "V300R002C10SPC200",
            "V300R003C00SPC100",
            "V300R003C10SPC100",
            "V300R003C20",
            "V300R003C20SPC200",
            "V300R006C00SPC100"
        )
        _, software_version, _ = get_system_version_without_ret(self.dsl("exec_cli 'show upgrade package'"))
        self.logger.info("[check_bbu_life_expire_patch_ver] Software version is %s." % software_version)
        return software_version in matches_version

    def is_product_model_matches(self):
        """
        检查方法:执行命令：show system general，查询设备型号；
        检查标准:若设备型号为2100 V3、2200 V3、2600 V3，则退出检查并返回通过，否则进行下一步检查；
        """
        matches_models = ("2100 V3", "2200 V3", "2600 V3")
        product_model = self.context.dev_node.model
        self.logger.info("[check_bbu_life_expire_patch_ver] product_model is %s." % product_model)
        return product_model in matches_models

    def is_target_version_matches(self):
        """
        检查方法:执行命令：从SmartKit工具获取目标基线版本和补丁版本；
        检查标准:若目标基线版本为V500R007C30SPC100或V300R006C50SPC100，则进行下一步检查，否则退出检查并返回通过；
        """
        matches_version = (
            "V500R007C30SPC100",
            "V300R006C50SPC100"
        )
        target_version = self.get_upgrade_target_version()
        self.logger.info("[check_bbu_life_expire_patch_ver] Target version is %s." % target_version)
        return target_version in matches_version, target_version
