# -*- coding: UTF-8 -*-
#  Copyright (c) Huawei Technologies Co., Ltd. 2024-2024. All rights reserved.
import ast
from psdk.checkitem.common.base_dsl_check import BaseCheckItem
from psdk.platform.entity.check_status import CheckStatus


# 冗余所需BBU数量
BBU_REDUNDANCY_COUNT = 2


class CheckItem(BaseCheckItem):
    # 存储从CLI查询到的BBU信息
    bbu_info_dict = {}

    def execute(self):
        self.logger.info("Checking the redundancy of BBU before replace BBU.")
        try:
            selected_fru = ast.literal_eval(self.context.execute_env.selected_fru)
            device_cur = selected_fru.get("location", "")
            ctrl_cur = selected_fru.get("parentID", "")
        except Exception as e:
            self.logger.error("Get device info from selected_fru failed.")
            return CheckStatus.NOT_PASS, self.get_err_msg("check.not.pass", "item.suggestion.bbuId")

        self.logger.info("The selected device is {}".format(device_cur))
        self.init_bbu_info()
        return self.check_bbu_redundancy(ctrl_cur)

    def init_bbu_info(self):
        self.bbu_info_dict = {}
        bbu_info = self.dsl("exec_rest '/backup_power'")
        for bbu in bbu_info:
            ctrl_id = bbu.get("CONTROLLERID")
            bbu_location = bbu.get("LOCATION")
            bbu_status = bbu.get("HEALTHSTATUS")
            if str(bbu_status) != "1":
                continue
            if ctrl_id not in self.bbu_info_dict.keys():
                self.bbu_info_dict[ctrl_id] = []
            self.bbu_info_dict[ctrl_id].append(bbu_location)
        self.logger.info("The BBU info is {}".format(self.bbu_info_dict))

    def check_bbu_redundancy(self, ctrl_cur):
        # 检查对称控制器的BBU状态
        if "A" in ctrl_cur:
            ctrl_id = ctrl_cur.replace("A", "B")
        else:
            ctrl_id = ctrl_cur.replace("B", "A")
        if ctrl_id not in self.bbu_info_dict.keys() or len(self.bbu_info_dict[ctrl_id]) < BBU_REDUNDANCY_COUNT:
            self.logger.error("BBU is not redundancy, check failed.")
            return CheckStatus.NOT_PASS, self.get_err_msg("check.not.pass", "item.suggestion")
        return CheckStatus.PASS, ""
