# -*- coding: UTF-8 -*-
#  Copyright (c) Huawei Technologies Co., Ltd. 2024-2024. All rights reserved.
import ast
from psdk.checkitem.common.base_dsl_check import BaseCheckItem
from psdk.platform.entity.check_status import CheckStatus


# 更换时需检查BBU冗余的器件列表
FRU_LIST = ["FAN2", "FAN7"]
# 冗余所需BBU数量
BBU_REDUNDANCY_COUNT = 2


class CheckItem(BaseCheckItem):
    # 存储从CLI查询到的BBU信息
    bbu_info_dict = {}

    def execute(self):
        self.logger.info("Checking the redundancy of BBU before replace fan.")
        try:
            self.init_bbu_info()
            selected_fru = ast.literal_eval(self.context.execute_env.selected_fru)
            for fru in selected_fru:
                device_cur = fru.get("location")
                ctrl_cur = fru.get("parentID")
                self.logger.info("The selected device is {}".format(device_cur))
                if not self.is_need_check(device_cur):
                    self.logger.info("Fan {} no need to check.".format(device_cur))
                    continue

                result, msg = self.check_bbu_redundancy(ctrl_cur)
                if result == CheckStatus.NOT_PASS:
                    return result, msg
            return CheckStatus.PASS, ""
        except Exception as e:
            self.logger.error("Get device info from selected_fru failed: {}.".format(str(e)))
            return CheckStatus.NOT_PASS, self.get_err_msg("check.not.pass", "item.suggestion.bbuId")

    @staticmethod
    def is_need_check(device):
        for item in FRU_LIST:
            if item in device:
                return True
        return False

    def init_bbu_info(self):
        self.bbu_info_dict = {}
        bbu_info = self.dsl("exec_rest '/backup_power'")
        for bbu in bbu_info:
            ctrl_id = bbu.get("CONTROLLERID")
            bbu_location = bbu.get("LOCATION")
            bbu_status = bbu.get("HEALTHSTATUS")
            if str(bbu_status) != "1":
                continue
            if ctrl_id not in self.bbu_info_dict.keys():
                self.bbu_info_dict[ctrl_id] = []
            self.bbu_info_dict[ctrl_id].append(bbu_location)
        self.logger.info("The BBU info is {}".format(self.bbu_info_dict))

    def check_bbu_redundancy(self, ctrl_cur):
        # 检查对称控制器的BBU状态
        if "A" in ctrl_cur:
            ctrl_id = ctrl_cur.replace("A", "B")
        else:
            ctrl_id = ctrl_cur.replace("B", "A")
        if ctrl_id not in self.bbu_info_dict.keys() or len(self.bbu_info_dict[ctrl_id]) < BBU_REDUNDANCY_COUNT:
            self.logger.error("BBU is not redundancy, check failed.")
            return CheckStatus.NOT_PASS, self.get_err_msg("check.not.pass", "item.suggestion")
        return CheckStatus.PASS, ""
