# -*- coding: UTF-8 -*-
#  Copyright (c) Huawei Technologies Co., Ltd. 2023-2024. All rights reserved.
import re
from datetime import datetime

from psdk.checkitem.common.base_dsl_check import BaseCheckItem
from psdk.platform.entity.check_status import CheckStatus
from psdk.platform.util.base_util import get_common_msg
from psdk.platform.util.product_util import is_support_cross_engine_ssh

THREE_DAYS = 3


class CheckItem(BaseCheckItem):
    def execute(self):
        # 是否支持跨引擎ssh连接
        if not is_support_cross_engine_ssh(self.context.dev_node.version, self.context.dev_node.model):
            return CheckStatus.PASS, ""

        # 查询系统当前时间
        cur_sys_time = self.dsl("exec_cli 'show system general' | regex 'Time.*(\d{4}-\d{2}-\d{2})' | get_index(0)")
        if not cur_sys_time:
            return CheckStatus.PASS, ""

        sys_poweron_time = self.query_sys_poweron_time()
        # 若系统上电时间与当前系统时间的时间差小于3天时，则巡检结果为通过，否则继续检查
        if not sys_poweron_time:
            return CheckStatus.NOT_PASS, self.get_msg("check.not.pass")
        if abs((datetime.strptime(cur_sys_time, '%Y-%m-%d') - datetime.strptime(sys_poweron_time,
                                                                                '%Y-%m-%d')).days) < THREE_DAYS:
            return CheckStatus.PASS, ""

        # 获取CCDB最新补丁包时间
        if not self.is_supper_admin():
            return CheckStatus.NOT_CHECK, get_common_msg(self.lang, "loginUser.name.level.must.be.super.admin")
        ccdb_backup_info_list = self.dsl("exec_on_all {}", self.query_ccdb_backup_info)
        all_backup_time = []
        for node_id, backup_info in ccdb_backup_info_list.items():
            if backup_info:
                for backup_time in backup_info:
                    time = backup_time[0:10]
                    all_backup_time.append(time)
        # 无备份包情况
        if not all_backup_time:
            return CheckStatus.NOT_PASS, self.get_msg("check.suggestion.norecord")

        for backup_time in all_backup_time:
            if abs((datetime.strptime(cur_sys_time, '%Y-%m-%d') - datetime.strptime(backup_time,
                                                                                     '%Y-%m-%d')).days) < THREE_DAYS:
                return CheckStatus.PASS, ""
        return CheckStatus.NOT_PASS, self.get_msg("check.suggestion.other")

    # 获取CCDB补丁包和补丁包创建时间
    def query_ccdb_backup_info(self):
        result = self.dsl("exec_mini_system 'ccdb.sh -c showbackupdb'", end_str_list=["minisystem>", "minisystem> "])
        ccdb_backup_time_info = re.findall(r'backup-(.*?).tar', str(result))
        return ccdb_backup_time_info

    # 获取节点上电时间
    def query_node_poweron_time(self):
        node_poweron_time = ""
        node_traceflows = self.dsl("exec_diagnose 'sys showtrace' | horizontal_parser")
        for traceflow in node_traceflows:
            if traceflow.get("Setup") == "NODE_POWER_ON":
                return traceflow.get("Date Time").split(" ")[0]
        return node_poweron_time

    # 获取系统上电时间
    def query_sys_poweron_time(self):
        poweron_times = self.dsl("exec_on_all {}", self.query_node_poweron_time)
        poweron_min_times = ""
        for node_id, poweron_time in poweron_times.items():
            if poweron_min_times == "":
                poweron_min_times = poweron_time
                continue
            if (datetime.strptime(poweron_min_times, '%Y-%m-%d') - datetime.strptime(poweron_time,
                                                                                     '%Y-%m-%d')).days > 0:
                poweron_min_times = poweron_time
        return poweron_min_times
