# -*- coding: UTF-8 -*-
#  Copyright (c) Huawei Technologies Co., Ltd. 2019-2021. All rights reserved.

from psdk.checkitem.common.base_dsl_check import BaseCheckItem
from psdk.platform.entity.check_status import CheckStatus
from psdk.dsl.dsl_common import get_version_info
import re

DISK_DOMAIN_VERSION = 36
LOGICPOOL_TOTAL_NUM = 18


class CheckItem(BaseCheckItem):
    def execute(self):
        # 检查方法步骤1：使用dsl执行命令查询产品型号及版本信息
        version_info = get_version_info(self.dsl)
        patch_version = version_info.get("patch_version").get("Current Version")
        version = version_info.get("base_version").get("Current Version")

        if self.check_softverison_patchversion_risk(version, patch_version):
            return CheckStatus.PASS, ""

        # 检查方法步骤2：使用dsl执行命令查询存储池是否存在
        storagepools_id_arr = self.dsl("exec_developer 'show storage_pool general |filterColumn include columnList=ID' | horizontal_parser | get_key('ID')")
        if not storagepools_id_arr:
            return CheckStatus.PASS, ""

        check_result_list = []
        diskdomain_verison_36_result = -1
        for storage_pool_id in storagepools_id_arr:
            storage_pool_infos = self.dsl("exec_developer 'show storage_pool general pool_id={}|filterColumn include columnList=Total\sCapacity,Disk\sDomain\sID' | vertical_parser".format(storage_pool_id[0]))
            storage_total_cap_in_byte = self.trans_subscribe_cap_2_byte([item.get("Total Capacity") for item in storage_pool_infos])
            disk_domain_id = [item.get("Disk Domain ID") for item in storage_pool_infos]
            disk_domain_ver = self.dsl("exec_developer 'show pool_manager disk_pool disk_pool_id={} |filterColumn include columnList=Version' | vertical_parser | get_key('Version')".format(disk_domain_id[0]))

            # 硬盘域版本号小于36，从低版本升级上来
            if int(disk_domain_ver[0]) < DISK_DOMAIN_VERSION:
                check_result_list.append("YES")
                continue

            get_all_lun_info = self.dsl("exec_cli 'show lun general pool_id={}|filterColumn include columnList=ID,Dedup\sEnabled,Compression\sEnabled' | horizontal_parser".format(storage_pool_id[0]))
            if not get_all_lun_info:
                check_result_list.append("YES")
                continue

            # 版本号等于36，并且是V3R2C00版本，才会有升级风险，因此需要判断容量
            if int(disk_domain_ver[0]) == DISK_DOMAIN_VERSION and "V300R002C00" in version:
                if self.check_has_dedup_and_comp_lun(get_all_lun_info) != 1:
                    check_result_list.append("YES")
                    continue

                # 查询所有logicpool的相关容量信息，并进行统计
                total_used_cap = 0
                for index in range(0, LOGICPOOL_TOTAL_NUM):
                    logic_pool_cap_info = self.dsl("exec_developer 'show pool_manager logic_pool_capacity disk_pool_id={} logic_pool_index={}|filterColumn include columnList=Total\sUsed\sCapacity,Compressed\sCapacity,Compress\sOrginal\sCapacity,Host\sWrite\sCapacity' | horizontal_parser".format(disk_domain_id[0], index))
                    total_used_cap += self.get_logicpool_total_used_cap(logic_pool_cap_info, index)

                # 如果logicpool查出来评估的最恶劣情况，会大于存储池总容量的90%，则有风险
                if total_used_cap >= storage_total_cap_in_byte * 0.90:
                    diskdomain_verison_36_result = 1
                check_result_list.append("NO")
            else:
                self.set_check_result_by_type(check_result_list, self.check_has_dedup_and_comp_lun(get_all_lun_info))

        for result in check_result_list:
            if result == "YES":
                continue

            if "V300R002C00" in version:
                if diskdomain_verison_36_result == 1:
                    return CheckStatus.NOT_PASS, self.get_msg("check.not.pass")
                else:
                    return CheckStatus.WARNING, self.get_msg("check.not.pass")
            else:
                return CheckStatus.WARNING, self.get_msg("check.not.pass")

        return CheckStatus.PASS, ""

    @staticmethod
    def check_softverison_patchversion_risk(current_ver, patch_ver):
        """
        检查标准1：系统软件历史版本是V300R002C00SPC100，V300R002C10SPC100 SPH132及之前版本，V300R002C20 SPH9及之前版本，则检查结果为不通过，需要继续检查
        """
        if "V300R002C00" in current_ver:
            return False
        expected_patch_dict = {
            "V300R002C10SPC100": "V300R002C10SPH132",
            "V300R002C20": "V300R002C20SPH009"
        }
        if current_ver not in expected_patch_dict.keys():
            return True
        expected_patch_ver = expected_patch_dict.get(current_ver)
        return patch_ver and patch_ver >= expected_patch_ver

    @staticmethod
    def trans_subscribe_cap_2_byte(subscribe_cap_info):
        if re.search("GB", subscribe_cap_info[0]):
            getvalue = subscribe_cap_info[0].split("GB")[0]
            getvalue = float(getvalue)
            getvalue = getvalue * 1024 * 1024 * 1024
        elif re.search("TB", subscribe_cap_info[0]):
            getvalue = subscribe_cap_info[0].split("TB")[0]
            getvalue = float(getvalue)
            getvalue = getvalue * 1024 * 1024 * 1024 * 1024
        elif re.search("MB", subscribe_cap_info[0]):
            getvalue = subscribe_cap_info[0].split("MB")[0]
            getvalue = float(getvalue)
            getvalue = getvalue * 1024 * 1024
        elif re.search("KB", subscribe_cap_info[0]):
            getvalue = subscribe_cap_info[0].split("KB")[0]
            getvalue = float(getvalue) * 1024
        else:
            getvalue = subscribe_cap_info[0].split("B")[0]
        return getvalue

    @staticmethod
    def check_has_dedup_and_comp_lun(get_all_lun_info):
        dedup_info = [item.get("Dedup Enabled") for item in get_all_lun_info]
        compress_info = [item.get("Compression Enabled") for item in get_all_lun_info]
        index = 0
        find_dedup_comp_lun = -1
        for lunIndex in range(len(get_all_lun_info)):
            if "".join(dedup_info[index]) == "Yes" and "".join(compress_info[index]) == "Yes":
                find_dedup_comp_lun = 1
                break
            index += 1
        return find_dedup_comp_lun

    @staticmethod
    def get_logicpool_total_used_cap(logicpool_cap_info, index):
        logic_pool_total_used_cap = 0
        logicpool_no_dedupcomp_index = [0, 10, 11, 12, 13, 14, 15, 16, 17]
        logicpool_nodedup_comp_index = [2, 4, 6, 8]
        logicpool_dedup_comp_index = [3, 5, 7, 9]

        all_controller_cap_index = -1

        if index in logicpool_no_dedupcomp_index:
            total_used_cap_in_logic_pool = int("".join(
                [item.get("Total Used Capacity") for item in logicpool_cap_info][all_controller_cap_index]))
            logic_pool_total_used_cap = total_used_cap_in_logic_pool
        elif index in logicpool_nodedup_comp_index:
            compressed_cap_in_logic_pool = int("".join(
                [item.get("Compressed Capacity") for item in logicpool_cap_info][all_controller_cap_index]))
            host_write_cap_in_logic_pool = int("".join(
                [item.get("Host Write Capacity") for item in logicpool_cap_info][all_controller_cap_index]))
            logic_pool_total_used_cap = compressed_cap_in_logic_pool + host_write_cap_in_logic_pool * 0.01
        elif index in logicpool_dedup_comp_index:
            compress_ori_cap_in_logic_pool = int("".join(
                [item.get("Compress Orginal Capacity") for item in logicpool_cap_info][all_controller_cap_index]))
            host_write_cap_in_logic_pool = int("".join(
                [item.get("Host Write Capacity") for item in logicpool_cap_info][all_controller_cap_index]))
            logic_pool_total_used_cap = compress_ori_cap_in_logic_pool + host_write_cap_in_logic_pool * 0.01

        return logic_pool_total_used_cap

    @staticmethod
    def set_check_result_by_type(check_result, type):
        if type != 1:
            check_result.append("YES")
        else:
            check_result.append("NO")
