# -*- coding: UTF-8 -*-
#  Copyright (c) Huawei Technologies Co., Ltd. 2019-2021. All rights reserved.

from psdk.checkitem.common.base_dsl_check import BaseCheckItem
from psdk.platform.entity.check_status import CheckStatus
from psdk.platform.protocol.cli_common import get_system_version_without_ret

KB_TO_BYTE = 1024  # 单位KB
MB_TO_BYTE = KB_TO_BYTE * 1024  # 单位MB
GB_TO_BYTE = MB_TO_BYTE * 1024  # 单位MB
TB_TO_BYTE = GB_TO_BYTE * 1024  # 单位MB
PB_TO_BYTE = TB_TO_BYTE * 1024  # 单位MB

NINE_PERCENT = 0.09  # 9%
USER_CAP_THRESHOLD = 80  # 80%
LEAK_CAP_THRESHOLD = 9 * 1024 * 1024 * 1024 * 1024  # 9TB

product_version_map = {
    '6.0.0': '6.0.0',
    '6.0.1': '6.0.1',
    '6.1.0': '6.1.0',
    '6.1.2': '6.1.2',
    '6.1.3': '6.1.3',
    '6.1.5': '6.1.5',
    '1.0.0': '1.0.0',
}


class CheckItem(BaseCheckItem):

    def execute(self):
        # 检查方法步骤0：获取产品版本信息，非指定版本检查直接通过
        flag, software_version, hot_patch_version = \
            get_system_version_without_ret(
                self.dsl("exec_cli 'show upgrade package'"))
        if software_version not in product_version_map:
            return CheckStatus.PASS, ""

        # 检查方法步骤1：使用dsl执行命令查询硬盘域的信息
        disk_domain_s = self.dsl("exec_cli 'show disk_domain general |filterColumn include columnList=ID"
                                 "'| horizontal_parser")
        if not disk_domain_s:
            # 无硬盘域，则检查直接通过
            self.logger.info("no disk_domain. check pass.")
            return CheckStatus.PASS, ""

        # 检查方法步骤2：使用dsl执行命令查询存储池ID和对应的硬盘域ID（存储池和硬盘域一一对应）
        storage_pool_s = self.dsl("exec_cli 'show storage_pool general |filterColumn include columnList=ID,"
                                  "Disk\sDomain\sID' | horizontal_parser")
        if not storage_pool_s:
            # 无存储池，则检查直接通过
            self.logger.info("no storage_pool. check pass.")
            return CheckStatus.PASS, ""
        # 2.1 建立poolID到diskDomainId的映射表
        pool_domain_id_map = {}
        for pool in storage_pool_s:
            # storage_pool_s是字典列表，列表中每一项都是字典
            pool_domain_id_map[pool['ID']] = pool['Disk Domain ID']

        # 2.2 获取storage_pool_id列表
        pool_id_s = [item.get('ID') for item in storage_pool_s]

        # 检查方法步骤3：遍历每个storage_pool，获取对应的logicPool是否重删及容量信息
        for pool_id in pool_id_s:
            logic_pool_s = self.dsl("exec_developer "
                                    "'show pool_manager logic_pool disk_pool_id={}"
                                    "|filterColumn include columnList=Index' "
                                    "| horizontal_parser".format(pool_domain_id_map.get(pool_id)))
            if not logic_pool_s:
                continue

            logic_pool_index_s = [item.get('Index') for item in logic_pool_s]
            # 3.1 提取支持重删的logical pool
            dedup_logic_pool_index_s = self.get_dedup_lp(self, logic_pool_index_s, pool_domain_id_map, pool_id)
            if not dedup_logic_pool_index_s:
                continue

            # 3.2 提取重删的logical pool的容量信息
            dedup_logic_pool_cap_s = self.get_dedup_logical_pool_cap(self, pool_id,
                                                                     dedup_logic_pool_index_s, pool_domain_id_map)

            disk_domain_cap = self.dsl("exec_cli 'show disk_domain general disk_domain_id={}"
                                       "'| vertical_parser".format(pool_domain_id_map.get(pool_id)))
            if not disk_domain_cap:
                continue

            # 3.3 提取disk_domain_cap表中的值
            disk_domain_sum_cap = self.get_dedup_disk_domain_cap(self, disk_domain_cap)

            # 3.4 根据容量总字典判断是否合理，不合理时检查不通过
            if self.judge_task_need_start(dedup_logic_pool_cap_s, disk_domain_sum_cap):
                return CheckStatus.NOT_PASS, "check.no.pass"

        # 遍历所有存储池结束，无不合理的重删容量，则检查通过
        return CheckStatus.PASS, ""

    @staticmethod
    def get_dedup_logical_pool_cap(self, pool_id, dedup_logic_pool_index_s, pool_domain_id_map):
        dedup_logic_pool_cap_s = {'Host Write Capacity': 0, 'Deduplicate Orginal Capacity': 0,
                                  'Deduplicated Capacity': 0}
        for dedup_logic_pool_index in dedup_logic_pool_index_s:
            dedup_logic_pool_cap = self.dsl("exec_developer 'show pool_manager logic_pool_capacity "
                                            "disk_pool_id={} logic_pool_index={} "
                                            "|filterColumn include columnList=Node\sID,Host\sWrite\sCapacity,"
                                            "Deduplicate\sOrginal\sCapacity,Deduplicated\sCapacity"
                                            "'| horizontal_parser".format(pool_domain_id_map.get(pool_id),
                                                                          dedup_logic_pool_index))
            if not dedup_logic_pool_cap:
                continue
            dedup_logic_pool_cap_s['Host Write Capacity'] += int(
                dedup_logic_pool_cap[-1].get('Host Write Capacity'))
            dedup_logic_pool_cap_s['Deduplicate Orginal Capacity'] += int(
                dedup_logic_pool_cap[-1].get('Deduplicate Orginal Capacity'))
            dedup_logic_pool_cap_s['Deduplicated Capacity'] += int(
                dedup_logic_pool_cap[-1].get('Deduplicated Capacity'))
        return dedup_logic_pool_cap_s

    @staticmethod
    def cap_conversion(res_cap):
        if 'PB' in res_cap:
            res_cap = float(res_cap.replace('PB', '')) * PB_TO_BYTE
        elif 'TB' in res_cap:
            res_cap = float(res_cap.replace('TB', '')) * TB_TO_BYTE
        elif 'GB' in res_cap:
            res_cap = float(res_cap.replace('GB', '')) * GB_TO_BYTE
        elif 'MB' in res_cap:
            res_cap = float(res_cap.replace('MB', '')) * MB_TO_BYTE
        elif 'KB' in res_cap:
            res_cap = float(res_cap.replace('KB', '')) * KB_TO_BYTE
        elif 'B' in res_cap:
            res_cap = float(res_cap.replace('B', ''))
        elif '--' == res_cap:
            res_cap = float(0)
        return res_cap

    @staticmethod
    def get_dedup_disk_domain_cap(self, disk_domain_cap):
        disk_domain_sum_cap = {'Total Capacity': 0, 'Free Capacity': 0,
                               'Hot Spare Capacity': 0}
        for disk_pool_cap in disk_domain_cap:
            for key in disk_pool_cap.keys():
                if key == 'Total Capacity':
                    disk_domain_sum_cap['Total Capacity'] = self.cap_conversion(disk_pool_cap[key])
                elif key == 'Free Capacity':
                    disk_domain_sum_cap['Free Capacity'] = self.cap_conversion(disk_pool_cap[key])
                elif key == 'Hot Spare Capacity':
                    disk_domain_sum_cap['Hot Spare Capacity'] = self.cap_conversion(disk_pool_cap[key])
        return disk_domain_sum_cap

    @staticmethod
    def get_dedup_lp(self, logic_pool_index_s, pool_domain_id_map, pool_id):
        dedup_logic_pool_index_s = []
        for logic_pool_index in logic_pool_index_s:
            logic_pool_dedup_cfg = self.dsl(
                "exec_developer "
                "'show pool_manager logic_pool disk_pool_id={} logic_pool_index={}"
                "|filterColumn include columnList=Suppor\sDDP' "
                "| vertical_parser".format(pool_domain_id_map[pool_id], logic_pool_index))
            if logic_pool_dedup_cfg[0]['Suppor DDP'] == '0':
                continue
            else:
                dedup_logic_pool_index_s.append(logic_pool_index)
        return dedup_logic_pool_index_s

    @staticmethod
    def judge_task_need_start(dedup_logic_pool_cap_s, disk_domain_sum_cap):
        # 单位为B
        dedup_in_capacity = dedup_logic_pool_cap_s['Deduplicate Orginal Capacity']
        data_in_capacity = dedup_logic_pool_cap_s['Host Write Capacity']
        data_out_capacity = dedup_logic_pool_cap_s['Deduplicated Capacity']

        domain_used_cap = (disk_domain_sum_cap['Total Capacity'] -
                           disk_domain_sum_cap['Free Capacity'] - disk_domain_sum_cap['Hot Spare Capacity'])
        domain_avail_phy_cap = (disk_domain_sum_cap['Total Capacity'] - disk_domain_sum_cap['Hot Spare Capacity'])
        if (domain_used_cap * 100) >= (domain_avail_phy_cap * USER_CAP_THRESHOLD):
            # * 100是为了换算为百分比。判断已使用容量 > 可使用物理容量的80%
            if dedup_in_capacity > data_in_capacity:  # 重删前大小 > 数据大小
                # 重删前大小 - 数据大小 = 泄漏量
                if (dedup_in_capacity - data_in_capacity) >= LEAK_CAP_THRESHOLD:  # 判断泄漏量 >= 泄漏量阈值，则返回True，需要强起引用计数泄露扫描
                    return True
                elif (dedup_in_capacity - data_in_capacity) > domain_avail_phy_cap * NINE_PERCENT:
                    # 泄漏量 > 可使用物理容量的9%，返回True，需要强起引用计数泄露扫描
                    return True
                elif data_in_capacity == 0:
                    return True
            elif (dedup_in_capacity == 0) and (data_in_capacity == 0):
                if data_out_capacity != 0:
                    # 泄漏量为0，重删后总量大于0
                    return True

        return False
