# -*- coding: UTF-8 -*-
#  Copyright (c) Huawei Technologies Co., Ltd. 2019-2022. All rights reserved.
import logging

from psdk.checkitem.common.base_dsl_check import BaseCheckItem
from psdk.dsl.dsl_common import get_version_info
from psdk.platform.entity.check_status import CheckStatus
from psdk.platform.util.base_util import get_common_msg
from psdk.platform.util.product_util import compare_patch_version
from psdk.dsl import fault_mode as ft

CHECK_BASE_VERSION = "6.1.2"
NOT_CHECK_FLAG_PATCH_VERSION = "SPH10"
STASH_DB_AV_DELAY = 20000000
STASH_DB_HISTORY_AV_DELAY = 1000000
STASH_DB_HISTORY_MAX_DELAY = 48000000
AV_RESULT_INDEX = 0
AVH_RESULT_INDEX = 0
DDP_LICENSE_NAME = ['SmartCompression Software License Premium', 'SmartCompression Software License Enhanced',
                    'SmartCompression Software License Exclusive', 'SmartCompression Software License General',
                    'SmartCompression Software License Advanced', 'SmartCompression Software License Standard']


class CheckItem(BaseCheckItem):

    @staticmethod
    def check_license_name(item):
        for license_name in DDP_LICENSE_NAME:
            if license_name in item.get('Feature Name'):
                return True
        return False

    def execute(self):
        # 步骤1 以admin用户登录设备，并查看产品型号、版本和补丁版本；
        # 若步骤1中产品型号和版本为Dorado V6 6.1.2 SPH10补丁之前的版本(不包括SPH10)则继续检查，其余产品和版本检查通过
        if not self.is_supper_admin():
            return CheckStatus.NOT_CHECK, get_common_msg(self.lang, "loginUser.name.level.must.be.super.admin")
        version_info = get_version_info(self.dsl)
        cur_version = version_info.get("base_version").get("Current Version")
        patch_version = version_info.get("patch_version").get("Current Version")
        if (cur_version != CHECK_BASE_VERSION) or (
                (cur_version == CHECK_BASE_VERSION) and (
                compare_patch_version(patch_version, NOT_CHECK_FLAG_PATCH_VERSION) >= 0)):
            return CheckStatus.PASS, ""
        if not self.has_ddp_license():
            return CheckStatus.PASS, ""
        results = self.dsl("exec_on_all {}", self.check_single_stash_db_delay)
        if 0 in results.values():
            return CheckStatus.NOT_PASS, self.get_err_msg("check.not.pass", "item.suggestion")
        return CheckStatus.PASS, ""

    def has_ddp_license(self):
        show_license_results = self.dsl("exec_cli 'show license' | vertical_parser",
                                        return_if={ft.FindStr("Error:"): "not_check"})
        check_license_result = []
        if (len(show_license_results) == 0) or (show_license_results == 'not_check'):
            return True
        for item in show_license_results:
            if 'Effective Capacity' == item.get('Feature Name') or self.check_license_name(item):
                return True
        return False

    def check_single_stash_db_delay(self):

        # 检查方法步骤4：查看当前写stashdb的平均时延(FTDS点POOL_PDA_WRITE_STASH_DB的平均值);
        av_results = self.dsl("exec_diagnose 'io show -m valid -n STASH_DB' | horizontal_parser",
                              return_if={ft.FindStr("Error:"): "not_check"})
        if av_results == 'not_check':
            return 1
        # 若步骤4中FTDS点POOL_PDA_WRITE_STASH_DB的平均值超过20000000(该点单位为us，检查阈值为20s)则检查不通过，否则继续检查
        if len(av_results) == 0:
            return 1
        if int(av_results[AV_RESULT_INDEX].get("Av")) > STASH_DB_AV_DELAY:
            return 0
        else:
            # 若步骤4中FTDS点POOL_PDA_WRITE_STASH_DB的历史平均值超过1000000(该点单位为us，检查阈值为1s)则检查不通过，否则检查通过
            avh_results = self.dsl("exec_diagnose 'io show -m valid -h -n STASH_DB' | horizontal_parser",
                                   return_if={ft.FindStr("Error:"): "not_check"})
            if avh_results == "not_check":
                return 1
            if len(avh_results) == 0:
                return 1
            if (int(avh_results[AVH_RESULT_INDEX].get("Av")) > STASH_DB_HISTORY_AV_DELAY) or (
                    int(avh_results[AVH_RESULT_INDEX].get("Max")) > STASH_DB_HISTORY_MAX_DELAY):
                return 0
        return 1
