# -*- coding: UTF-8 -*-
#  Copyright (c) Huawei Technologies Co., Ltd. 2019-2022. All rights reserved.

import re
from psdk.checkitem.common.base_dsl_check import BaseCheckItem
from psdk.platform.entity.check_status import CheckStatus
from psdk.dsl.dsl_common import get_version_info, get_engine_height
from psdk.platform.util.product_util import compare_patch_version

# dorado高端设备型号
DORADO_HIGH_MODELS = ["OceanStor Dorado 8000 V6", "OceanStor Dorado 18000 V6", "OceanStor Dorado 6800 V6",
                      "OceanStor Dorado 18500 V6", "OceanStor Dorado 18800 V6", "OceanStor Dorado 18800K V6",
                      "OceanProtect X9000", "OceanProtect X9000K", "OceanStor 6810",
                      "OceanStor 18510", "OceanStor 18500K", "OceanStor 18810"
                      ]
# 框高对应的单框最大控制器数量
ENC_HEIGHT_TO_CTRL_NUM = {2: 2, 3: 2, 4: 4, 6: 4}

# 解决扩控CPU利用率升高问题的补丁
PASS_VERSIONS = {"6.0.1": "SPH33", "6.1.0": "SPH30", "6.1.2": "SPH22", "6.1.3": "SPH10"}

# 内双活的枚举值
HYPER_METRO_INNER_FLAG = 103


class CheckItem(BaseCheckItem):
    def execute(self):
        version_info = get_version_info(self.dsl)
        cur_version = version_info.get("base_version").get("Current Version")
        patch_version = version_info.get("patch_version").get("Current Version")

        # 版本为6.1.2.SPH22及其之后的版本，则检查通过
        sver = cur_version[:len("6.0.1")]
        if sver in PASS_VERSIONS and compare_patch_version(patch_version, PASS_VERSIONS.get(sver)) >= 0:
            return CheckStatus.PASS, ""

        # 版本为6.1.3，则检查是否为多引擎，如果是多引擎，则检查通过
        if (cur_version.startswith("6.1.3")) and self.is_multiple_engines():
            return CheckStatus.PASS, ""

        # 不存在存储池，则检查通过
        storage_pools = self.dsl("exec_cli 'show storage_pool general' | horizontal_parser")
        if not storage_pools:
            return CheckStatus.PASS, ""

        domain_ids = [item.get("Disk Domain ID") for item in storage_pools]

        # 如果有存储池新增引擎，即新增pool，且硬盘域空闲率低于回收阈值，则检查不通过
        extend_pools = self.get_extend_pools(domain_ids)
        if extend_pools and self.is_low_disk_ratio(extend_pools):
            return CheckStatus.NOT_PASS, self.get_msg("check.not.pass", cur_version)

        # 如果是高端2扩4场景，或者内双活场景，且硬盘域空闲率低于回收阈值，则检查不通过
        if self.is_high_expand_or_inner_hypermetro() and self.is_low_disk_ratio(domain_ids):
            return CheckStatus.NOT_PASS, self.get_msg("check.not.pass", cur_version)
        
        return CheckStatus.PASS, ""

    def get_extend_pools(self, domain_ids):
        """
        在原硬盘域上扩展的新引擎信息，将扩盘引擎和当前引擎做差值计算
        """
        expand_infos = self.get_expand_disk_engines()
        if not expand_infos:
            return []

        local_infos = self.get_logic_disk_engines(domain_ids)
        extend_pools = []
        # 判断原存储池上是否增加了新引擎，增加了则添加存储池id
        for domain_id, expand_engines in expand_infos.items():
            local_engines = local_infos.get(domain_id)
            if not local_engines:
                continue
            for expand in expand_engines:
                if expand not in local_engines:
                    extend_pools.append(domain_id)
        return extend_pools

    def get_expand_disk_engines(self):
        """
        获取扩盘的引擎信息
        """
        disk_engines = {}
        exp_disk_list = self.context.execute_env.ori_env.get("exp_disk_list")
        if not exp_disk_list:
            exp_disk_list = self.context.execute_env.ori_env.get("expDiskList")
        if not exp_disk_list:
            return disk_engines
        for exp_disk in exp_disk_list:
            # 新建硬盘域场景，硬盘域id为空，这种场景不检查
            disk_domain = exp_disk.get("diskDomain")
            if not disk_domain or disk_domain == 'none':
                continue
            if disk_domain not in disk_engines:
                disk_engines[disk_domain] = []
            disk_engines[disk_domain].append(exp_disk.get("eng"))
        return disk_engines

    def get_logic_disk_engines(self, domain_ids):
        """
        获取当前指定硬盘域的引擎
        """
        disk_engines = {}
        for domain_id in domain_ids:
            dd_results = self.dsl("exec_cli 'show disk_domain general disk_domain_id={}' "
                                  "| vertical_parser".format(domain_id))
            enclosure = dd_results[0].get("Controller enclosure")
            if enclosure:
                disk_engines[domain_id] = enclosure.split(",")
        return disk_engines

    def is_high_expand_or_inner_hypermetro(self):
        """
        判断是否是高端2扩4或者内双活场景
        """
        if self.context.execute_env.tool_name != "exp_ctrl":
            return False

        model = self.context.dev_node.model
        if model not in DORADO_HIGH_MODELS:
            return False

        # 内双活
        if self.has_inner_hypermetro_license():
            return True

        # 2扩4
        dev_infos = self.dsl("exec_diagnose 'sys showcls' | vertical_parser")
        dev_info = dev_infos[0]
        current_ctrl_num = int(dev_info.get("node cfg"))
        expand_ctrl_num = self.context.execute_env.ori_env.get("expCtrl")
        if int(current_ctrl_num) == 2 and int(expand_ctrl_num) == 4:
            return True
        return False

    def has_inner_hypermetro_license(self):
        """
        检查是否有内双活的许可
        """
        license_lines = self.dsl("exec_cli 'show license' | splitlines")
        for line in license_lines:
            if "Feature ID" in line:
                feature_id = int(line.split(":")[1].strip())
                if feature_id == HYPER_METRO_INNER_FLAG:
                    return True
        return False

    def is_low_disk_ratio(self, domain_ids):
        """
        判断硬盘域的空闲率是否小于垃圾回收阈值，domain_ids是硬盘域id列表
        """
        if not domain_ids:
            return False
        for domain_id in domain_ids:
            # 查指定硬盘域的垃圾量回收信息
            show_lines = self.dsl("exec_developer 'show zone_recycle ratio pool_id={} zone_type=data |"
                                  "filterColumn include columnList=Free\sRatio,Start\sRatio' "
                                  "| horizontal_parser".format(domain_id))
            if not show_lines:
                return False
            ratio_line = show_lines[-1]
            # 当前的空闲率
            cur_ratio = int(ratio_line.get("Free Ratio"))
            # 回收启动阈值
            start_ratio = int(ratio_line.get("Start Ratio"))
            if cur_ratio < start_ratio:
                return True
        return False

    def is_multiple_engines(self):
        """
        检查是否多引擎，包括扩容前和扩容后
        """
        # 当前是否多引擎
        enclosures = self.dsl("exec_cli 'show enclosure' | horizontal_parser")
        cte_ids = [enclosure.get("ID") for enclosure in enclosures if enclosure.get("ID").startswith("CTE")]
        if len(cte_ids) > 1:
            return True

        if self.context.execute_env.tool_name != "exp_ctrl":
            return False

        # 当前是单引擎，检查是否新增引擎
        dev_infos = self.dsl("exec_diagnose 'sys showcls' | vertical_parser")
        dev_info = dev_infos[0]
        node_max = int(dev_info.get("node max"))
        group_max = int(dev_info.get("group max"))
        current_ctrl_num = int(dev_info.get("node cfg"))
        expand_ctrl_num = int(self.context.execute_env.ori_env.get("expCtrl"))
        engine_height = 0
        try:
            engine_height = int(node_max / group_max)
        except ZeroDivisionError as zero_ex:
            self.logger.error("error is {}".format(zero_ex))
        peer_ctrl_num = ENC_HEIGHT_TO_CTRL_NUM.get(engine_height, 4)
        if current_ctrl_num % peer_ctrl_num != 0 and expand_ctrl_num - current_ctrl_num < peer_ctrl_num:
            # 不新增引擎，即单引擎
            return False
        # 新增引擎，后是多引擎
        return True
